/*_############################################################################
  _## 
  _##  threads.cpp  
  _## 
  _##
  _##  AGENT++ API Version 3.5.31
  _##  -----------------------------------------------
  _##  Copyright (C) 2000-2010 Frank Fock, Jochen Katz
  _##  
  _##  LICENSE AGREEMENT
  _##
  _##  WHEREAS,  Frank  Fock  and  Jochen  Katz  are  the  owners of valuable
  _##  intellectual  property rights relating to  the AGENT++ API and wish to
  _##  license AGENT++ subject to the  terms and conditions set forth  below;
  _##  and
  _##
  _##  WHEREAS, you ("Licensee") acknowledge  that Frank Fock and Jochen Katz
  _##  have the right  to grant licenses  to the intellectual property rights
  _##  relating to  AGENT++, and that you desire  to obtain a license  to use
  _##  AGENT++ subject to the terms and conditions set forth below;
  _##
  _##  Frank  Fock    and Jochen   Katz   grants  Licensee  a  non-exclusive,
  _##  non-transferable, royalty-free  license  to use   AGENT++ and  related
  _##  materials without  charge provided the Licensee  adheres to all of the
  _##  terms and conditions of this Agreement.
  _##
  _##  By downloading, using, or  copying  AGENT++  or any  portion  thereof,
  _##  Licensee  agrees to abide  by  the intellectual property  laws and all
  _##  other   applicable laws  of  Germany,  and  to all of   the  terms and
  _##  conditions  of this Agreement, and agrees  to take all necessary steps
  _##  to  ensure that the  terms and  conditions of  this Agreement are  not
  _##  violated  by any person  or entity under the  Licensee's control or in
  _##  the Licensee's service.
  _##
  _##  Licensee shall maintain  the  copyright and trademark  notices  on the
  _##  materials  within or otherwise  related   to AGENT++, and  not  alter,
  _##  erase, deface or overprint any such notice.
  _##
  _##  Except  as specifically   provided in  this  Agreement,   Licensee  is
  _##  expressly   prohibited  from  copying,   merging,  selling,   leasing,
  _##  assigning,  or  transferring  in  any manner,  AGENT++ or  any portion
  _##  thereof.
  _##
  _##  Licensee may copy materials   within or otherwise related   to AGENT++
  _##  that bear the author's copyright only  as required for backup purposes
  _##  or for use solely by the Licensee.
  _##
  _##  Licensee may  not distribute  in any  form  of electronic  or  printed
  _##  communication the  materials  within or  otherwise  related to AGENT++
  _##  that  bear the author's  copyright, including  but  not limited to the
  _##  source   code, documentation,  help  files, examples,  and benchmarks,
  _##  without prior written consent from the authors.  Send any requests for
  _##  limited distribution rights to fock@agentpp.com.
  _##
  _##  Licensee  hereby  grants  a  royalty-free  license  to  any  and   all 
  _##  derivatives  based  upon this software  code base,  that  may  be used
  _##  as a SNMP  agent development  environment or a  SNMP agent development 
  _##  tool.
  _##
  _##  Licensee may  modify  the sources  of AGENT++ for  the Licensee's  own
  _##  purposes.  Thus, Licensee  may  not  distribute  modified  sources  of
  _##  AGENT++ without prior written consent from the authors. 
  _##
  _##  The Licensee may distribute  binaries derived from or contained within
  _##  AGENT++ provided that:
  _##
  _##  1) The Binaries are  not integrated,  bundled,  combined, or otherwise
  _##     associated with a SNMP agent development environment or  SNMP agent
  _##     development tool; and
  _##
  _##  2) The Binaries are not a documented part of any distribution material. 
  _##
  _##
  _##  THIS  SOFTWARE  IS  PROVIDED ``AS  IS''  AND  ANY  EXPRESS OR  IMPLIED
  _##  WARRANTIES, INCLUDING, BUT NOT LIMITED  TO, THE IMPLIED WARRANTIES  OF
  _##  MERCHANTABILITY AND FITNESS FOR  A PARTICULAR PURPOSE  ARE DISCLAIMED.
  _##  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
  _##  INDIRECT,   INCIDENTAL,  SPECIAL, EXEMPLARY,  OR CONSEQUENTIAL DAMAGES
  _##  (INCLUDING,  BUT NOT LIMITED  TO,  PROCUREMENT OF SUBSTITUTE  GOODS OR
  _##  SERVICES; LOSS OF  USE,  DATA, OR PROFITS; OR  BUSINESS  INTERRUPTION)
  _##  HOWEVER CAUSED  AND ON ANY THEORY  OF  LIABILITY, WHETHER IN CONTRACT,
  _##  STRICT LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
  _##  IN  ANY WAY OUT OF  THE USE OF THIS  SOFTWARE,  EVEN IF ADVISED OF THE
  _##  POSSIBILITY OF SUCH DAMAGE. 
  _##
  _##
  _##  Stuttgart, Germany, Thu Sep  2 00:07:56 CEST 2010 
  _##  
  _##########################################################################*/
#include <libagent.h>

#include <agent_pp/threads.h>
#include <agent_pp/mib_entry.h>
#include <agent_pp/mib.h>
#include <snmp_pp/log.h>

#ifdef AGENTPP_NAMESPACE
namespace Agentpp {
#endif

static const char *loggerModuleName = "agent++.threads";

#ifdef _THREADS
Synchronized ThreadManager::global_lock;
#endif

/**
 * Default constructor
 */
ThreadManager::ThreadManager()
{
}

/**
 * Destructor
 */
ThreadManager::~ThreadManager()
{
#if defined(_THREADS) && !defined(NO_FAST_MUTEXES)
    if (trylock())
	unlock();
    else
	unlock();
#endif    
}

/**
 * Start synchronized execution.
 */
void ThreadManager::start_synch()
{
#ifdef _THREADS
	lock();
#endif
}

/**
 * End synchronized execution.
 */
void ThreadManager::end_synch()
{
#ifdef _THREADS
	unlock();
#endif
}

/**
 * Start global synchronized execution.
 */
void ThreadManager::start_global_synch()
{
#ifdef _THREADS
	global_lock.lock();
#endif
}

/**
 * End global synchronized execution.
 */
void ThreadManager::end_global_synch()
{
#ifdef _THREADS
	global_lock.unlock();
#endif
}

SingleThreadObject::SingleThreadObject(): ThreadManager()
{
	start_synch();
}

SingleThreadObject::~SingleThreadObject()
{
	end_synch();
}

#ifdef _THREADS

/*--------------------- class Synchronized -------------------------*/

#define ERR_CHK_WITHOUT_EXCEPTIONS(x) \
	do { \
	    int result = (x); \
	    if (result) { \
		    LOG_BEGIN(loggerModuleName, ERROR_LOG | 0); \
		    LOG("Constructing Synchronized failed at '" #x "' with (result)"); \
		    LOG(result); \
		    LOG_END; \
	    } \
	} while(0)

Synchronized::Synchronized()
{
#ifdef POSIX_THREADS
	pthread_mutexattr_t attr;
	ERR_CHK_WITHOUT_EXCEPTIONS( pthread_mutexattr_init(&attr) );
	ERR_CHK_WITHOUT_EXCEPTIONS( pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE) );

	memset(&serializer, 0, sizeof(serializer));
	ERR_CHK_WITHOUT_EXCEPTIONS( pthread_mutex_init(&serializer, &attr) );
	ERR_CHK_WITHOUT_EXCEPTIONS( pthread_mutexattr_destroy(&attr) );

	ERR_CHK_WITHOUT_EXCEPTIONS( pthread_mutex_init(&monitor, NULL) );

	memset(&cond, 0, sizeof(cond));
	ERR_CHK_WITHOUT_EXCEPTIONS( pthread_cond_init(&cond, 0) );
#elif defined(WIN32)
	// Semaphore initially auto signaled, auto reset mode, unnamed
	semEvent = CreateEvent(0, false, false, 0);
	// Semaphore initially unowned, unnamed
	semMutex = CreateMutex(0, false, 0);
	isLocked = false;
#endif
}


Synchronized::~Synchronized()
{
#ifdef POSIX_THREADS
	int result;

	result = pthread_cond_destroy(&cond);
	if (result) {
		LOG_BEGIN(loggerModuleName, ERROR_LOG | 2);
		LOG("Synchronized cond_destroy failed with (result)(ptr)");
		LOG(result);
		LOG((unsigned long)this);
		LOG_END;
	}
        pthread_mutex_lock(&monitor); // another thread owns the mutex, let's wait ...
        pthread_mutex_unlock(&monitor);
	result = pthread_mutex_destroy(&monitor);
	if (result) {
		LOG_BEGIN(loggerModuleName, ERROR_LOG | 2);
		LOG("Synchronized mutex_destroy for monitor failed with (result)(ptr)");
		LOG(result);
		LOG((unsigned long)this);
		LOG_END;
	}
	result = pthread_mutex_destroy(&serializer);
#ifdef NO_FAST_MUTEXES
        if( result == EBUSY ) {
            // wait for other threads ...
            if( EBUSY == pthread_mutex_trylock(&serializer) )
                pthread_mutex_lock(&serializer); // another thread owns the mutex, let's wait ...
            do {
                pthread_mutex_unlock(&serializer);
                result = pthread_mutex_destroy(&serializer);
            } while( EBUSY == result );
        }
#endif
	if (result) {
		LOG_BEGIN(loggerModuleName, ERROR_LOG | 2);
		LOG("Synchronized mutex_destroy for serializer failed with (result)(ptr)");
		LOG(result);
		LOG((unsigned long)this);
		LOG_END;
	}
#elif defined(WIN32)
	CloseHandle(semEvent);
	CloseHandle(semMutex);
#endif
}


void Synchronized::wait() {
#ifdef POSIX_THREADS
	cond_timed_wait(0);
#elif defined(WIN32)
	wait(INFINITE);
#endif  
}

#ifdef POSIX_THREADS
int Synchronized::cond_timed_wait(const struct timespec *ts) 
{
  int result;
  pthread_mutex_lock(&monitor);
  if (ts) 
	result = pthread_cond_timedwait(&cond, &monitor, ts);
  else 
	result = pthread_cond_wait(&cond, &monitor);
  pthread_mutex_unlock(&monitor);
  return result;
}
#endif

bool Synchronized::wait(unsigned long timeout)
{
	bool timeoutOccurred = false;
#ifdef POSIX_THREADS
	struct timespec ts;
	struct timeval  tv;
	gettimeofday(&tv, 0);
	ts.tv_sec  = tv.tv_sec  + (int)timeout/1000;
	ts.tv_nsec = (tv.tv_usec + (timeout %1000)*1000) * 1000; 

	int err;
	if ((err = cond_timed_wait(&ts)) > 0) {
		switch(err) {
		case ETIMEDOUT:
		  timeoutOccurred = true;
		  break;
		default:
		  LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
		  LOG("Synchronized: wait with timeout returned (error)");
		  LOG(err);
		  LOG_END;
		  break;
		}
	}
#elif defined(WIN32)
	isLocked = false;
	if (!ReleaseMutex(semMutex)) {
		LOG_BEGIN(loggerModuleName, ERROR_LOG | 2);
		LOG("Synchronized: releasing mutex failed");
		LOG_END;
	}
	int err;
	err = WaitForSingleObject(semEvent, timeout);
	switch (err) {
	case WAIT_TIMEOUT:
		LOG_BEGIN(loggerModuleName, EVENT_LOG | 8);
		LOG("Synchronized: timeout on wait");
		LOG_END;
		timeoutOccurred = true;
		break;
	case WAIT_ABANDONED:
		LOG_BEGIN(loggerModuleName, ERROR_LOG | 2);
		LOG("Synchronized: waiting for event failed");
		LOG_END;
	}
	if (WaitForSingleObject (semMutex, INFINITE) != WAIT_OBJECT_0) {
		LOG_BEGIN(loggerModuleName, WARNING_LOG | 8);
		LOG("Synchronized: waiting for mutex failed");
		LOG_END;
	}
	isLocked = true;
#endif
	return timeoutOccurred;
}

void Synchronized::notify() {
#ifdef POSIX_THREADS
	int result;
	pthread_mutex_lock(&monitor);
	result = pthread_cond_signal(&cond);
	pthread_mutex_unlock(&monitor);
	if (result) {
		LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
		LOG("Synchronized: notify failed (result)");
		LOG(result);
		LOG_END;
	}
#elif defined(WIN32)
	numNotifies = 1;
	if (!SetEvent(semEvent)) {
		LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
		LOG("Synchronized: notify failed");
		LOG_END;
	}
#endif
}


void Synchronized::notify_all() {
#ifdef POSIX_THREADS
	int result;
	pthread_mutex_lock(&monitor);
	result = pthread_cond_broadcast(&cond);
	pthread_mutex_unlock(&monitor);
	if (result) {
		LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
		LOG("Synchronized: notify_all failed (result)");
		LOG(result);
		LOG_END;
	}
#elif defined(WIN32)
	numNotifies = (char)0x80;
	while (numNotifies--)
		if (!SetEvent(semEvent)) {
			LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
			LOG("Synchronized: notify failed");
			LOG_END;
		}
#endif
}

void Synchronized::lock() {
#ifdef POSIX_THREADS
    pthread_mutex_lock(&serializer);
#elif defined(WIN32)
    if (WaitForSingleObject(semMutex, INFINITE) != WAIT_OBJECT_0) {
	LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
	LOG("Synchronized: lock failed");
	LOG_END;
	return;
    }
    if (isLocked) {
	// This thread owns already the lock, but
        // we do not like recursive locking. Thus
        // release it immediately and print a warning!
	if (!ReleaseMutex(semMutex)) {
		LOG_BEGIN(loggerModuleName, WARNING_LOG | 1);
		LOG("Synchronized: unlock failed");
		LOG_END;
	}      
	LOG_BEGIN(loggerModuleName, WARNING_LOG | 0);
	LOG("Synchronized: recursive locking detected!");
	LOG_END;	
    }
    isLocked = true;
#endif
}

void Synchronized::unlock() {
#ifdef POSIX_THREADS
	pthread_mutex_unlock(&serializer);
#elif defined(WIN32)
	isLocked = false;
	if (!ReleaseMutex(semMutex)) {
		LOG_BEGIN(loggerModuleName, WARNING_LOG | 1);
		LOG("Synchronized: unlock failed");
		LOG_END;
		return;
	}
#endif
}

bool Synchronized::trylock() {
#ifdef POSIX_THREADS
	if (pthread_mutex_trylock(&serializer) == 0)
		return true;
	else
		return false;
#elif defined(WIN32)
	int status = WaitForSingleObject(semMutex, 0);
	if (status != WAIT_OBJECT_0) {
		LOG_BEGIN(loggerModuleName, DEBUG_LOG | 9);
		LOG("Synchronized: try lock failed");
		LOG_END;
		return false;
	}
	if (isLocked) {
		if (!ReleaseMutex(semMutex)) {
			LOG_BEGIN(loggerModuleName, WARNING_LOG | 1);
			LOG("Synchronized: unlock failed");
			LOG_END;
		}
		return false;
	}
	else
		isLocked = true;
	return true;
#endif
}


/*------------------------ class Thread ----------------------------*/

ThreadList Thread::threadList; 

#ifdef POSIX_THREADS
void* thread_starter(void* t)
{
	Thread *thread = (Thread*)t;
	Thread::threadList.add(thread);

	LOG_BEGIN(loggerModuleName, DEBUG_LOG | 1);
	LOG("Thread: started (tid)");
	LOG((int)(thread->tid));
	LOG_END;

	thread->get_runnable().run();

	LOG_BEGIN(loggerModuleName, DEBUG_LOG | 1);
	LOG("Thread: ended (tid)");
	LOG((int)(thread->tid));
	LOG_END;

	Thread::threadList.remove(thread);
	thread->status = Thread::FINISHED;

	return t;
}
#elif defined(WIN32)
DWORD thread_starter(LPDWORD lpdwParam)
{
	Thread *thread = (Thread*) lpdwParam;
	Thread::threadList.add(thread);

	LOG_BEGIN(loggerModuleName, DEBUG_LOG | 1);
	LOG("Thread: started (tid)");
	LOG(thread->tid);
	LOG_END;

	thread->get_runnable().run();

	LOG_BEGIN(loggerModuleName, DEBUG_LOG | 1);
	LOG("Thread: ended (tid)");
	LOG(thread->tid);
	LOG_END;

	Thread::threadList.remove(thread);
	thread->status = Thread::FINISHED;

	::SetEvent(thread->threadEndEvent);

	return 0;	
}
#endif

Thread::Thread()
	: runnable(this)
	, status(IDLE)
	, stackSize(AGENTPP_DEFAULT_STACKSIZE)
#if defined(POSIX_THREADS)
	, tid()
#elif defined(WIN32)
	, threadHandle(INVALID_HANDLE_VALUE)
	, tid()
	, threadEndEvent(::CreateEvent(NULL, true, false, NULL))
#endif
{
}

Thread::Thread(Runnable &r)
	: runnable(&r)
	, status(IDLE)
	, stackSize(AGENTPP_DEFAULT_STACKSIZE)
#if defined(POSIX_THREADS)
	, tid()
#elif defined(WIN32)
	, threadHandle(INVALID_HANDLE_VALUE)
	, tid()
	, threadEndEvent(::CreateEvent(NULL, true, false, NULL))
#endif
{
}

void Thread::run()
{
	LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
	LOG("Thread: empty run method!");
	LOG_END;
}

Thread::~Thread() 
{ 
	if (status != IDLE)
	{
		join(); 
	}
#ifdef WIN32
	::CloseHandle(threadEndEvent);
	if(threadHandle != INVALID_HANDLE_VALUE)
	{
		::CloseHandle(threadHandle);
		threadHandle = INVALID_HANDLE_VALUE;
	}
#endif
}


Runnable& Thread::get_runnable()
{
	return *runnable;
}

void Thread::join()
{
#ifdef POSIX_THREADS
	if (status) {
		void* retstat;
		int err = pthread_join(tid, &retstat);
		if (err) {
			LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
			LOG("Thread: join failed (error)");
			LOG(err);
			LOG_END;
		}
		status = IDLE;
		LOG_BEGIN(loggerModuleName, DEBUG_LOG | 4);
		LOG("Thread: joined thread successfully (tid)");
		LOG((int)tid);
		LOG_END;
	}
	else {
		LOG_BEGIN(loggerModuleName, WARNING_LOG | 1);
		LOG("Thread: thread not running (tid)");
		LOG((int)tid);
		LOG_END;
	}
#elif defined(WIN32)
	if (status) {
		if (WaitForSingleObject(threadEndEvent, 
					INFINITE) != WAIT_OBJECT_0) {
			LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
			LOG("Thread: join failed");
			LOG_END;
		}
		status = IDLE;
		LOG_BEGIN(loggerModuleName, DEBUG_LOG | 4);
		LOG("Thread: joined thread successfully");
		LOG_END;
	}
	else {
		LOG_BEGIN(loggerModuleName, WARNING_LOG | 1);
		LOG("Thread: thread not running");
		LOG_END;
	}
#endif
}

void Thread::start()
{
#ifdef POSIX_THREADS
	if (status == IDLE) {
		pthread_attr_t attr;
		pthread_attr_init(&attr);
		pthread_attr_setstacksize(&attr, stackSize);
		int err = pthread_create(&tid, &attr, thread_starter, this);
		if (err) {
			LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
			LOG("Thread: cannot start thread (error)");
			LOG(err);
			LOG_END;
			status = IDLE;
		}
		else 
			status = RUNNING;
		pthread_attr_destroy(&attr);
	}
	else {
		LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
		LOG("Thread: thread already running!");
		LOG_END;
	}
#elif defined(WIN32)
	DWORD *targ = (DWORD*)this;
  
	if (status == IDLE) {

        if(threadHandle != INVALID_HANDLE_VALUE)
		{
			::CloseHandle(threadHandle);
			threadHandle = INVALID_HANDLE_VALUE;
		}

        threadHandle = 
		  CreateThread (0, // no security attributes
				stackSize, 
				(LPTHREAD_START_ROUTINE)thread_starter, 
				targ, 
				0,   
				&tid);   
		
		if (threadHandle == 0) {
			LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
			LOG("Thread: cannot start thread");
			LOG_END;
			status = IDLE;
		}
		else {
			status = RUNNING;
		}
	}
	else {
		LOG_BEGIN(loggerModuleName, ERROR_LOG | 1);
		LOG("Thread: thread already running!");
		LOG_END;
	}
#endif
}

void  Thread::sleep(long millis)
{
#ifdef WIN32
	Sleep (millis);
#else
	nsleep((int)(millis/1000), (millis%1000)*1000000);
#endif
}

void Thread::sleep(long millis, int nanos)
{
#ifdef WIN32
	sleep (millis);
#else
	nsleep((int)(millis/1000), (millis%1000)*1000000 + nanos);
#endif
}

void Thread::nsleep(int secs, long nanos)
{
#ifdef WIN32
	DWORD millis = secs*1000 + nanos/1000000;
	Sleep(millis);
#else
	long s = secs + nanos / 1000000000;
	long n = nanos % 1000000000;

#ifdef _POSIX_TIMERS
	struct timespec interval, remainder;
	interval.tv_sec = (int)s;
	interval.tv_nsec = n;
	if (nanosleep(&interval, &remainder) == -1) {
		if (errno == EINTR) {
			LOG_BEGIN(loggerModuleName, EVENT_LOG | 3);
			LOG("Thread: sleep interrupted");
			LOG_END;
		}
	}
#else
	struct timeval interval;
	interval.tv_sec = s;
	interval.tv_usec = n/1000;
	fd_set writefds, readfds, exceptfds;
	FD_ZERO(&writefds);
	FD_ZERO(&readfds);
	FD_ZERO(&exceptfds);
	if (select(0, &writefds, &readfds, &exceptfds, &interval) == -1) {
		if (errno == EINTR) {
			LOG_BEGIN(loggerModuleName, EVENT_LOG | 3);
			LOG("Thread: sleep interrupted");
			LOG_END;
		}
	}
#endif
#endif
}

#ifdef AGENTPP_USE_THREAD_POOL

static const char *threadPoolLoggerName = "agent++.threads.threadpool";

/*--------------------- class TaskManager --------------------------*/

TaskManager::TaskManager(ThreadPool *tp, int stackSize):thread(*this)
{
	threadPool = tp;
	task       = 0;
	go         = true;
	thread.set_stack_size(stackSize);
	thread.start();
        if( thread.is_alive() )
        {
            LOG_BEGIN(threadPoolLoggerName, DEBUG_LOG | 1);
            LOG("TaskManager: thread started");
            LOG_END;
        }
        else
        {
            LOG_BEGIN(threadPoolLoggerName, ERROR_LOG | 0);
            LOG("TaskManager: failed to start thread");
            LOG_END;
        }
}

TaskManager::~TaskManager() {
	stop();
	thread.join();
	LOG_BEGIN(threadPoolLoggerName, DEBUG_LOG | 1);
	LOG("TaskManager: thread stopped");
	LOG_END;
}

void TaskManager::runTask() {
	Lock guard(*this);
	if( task ) {
		task->run();
		delete task;
		task = 0;
	}
}

void TaskManager::run() {
	bool uglyRaceLikely = false;
	while (go) {
		if (task) {
			LOG_BEGIN(threadPoolLoggerName, DEBUG_LOG | 5);
			LOG("TaskManager::run(): before runTask()");
			LOG_END;

			runTask();

			threadPool->idle_notification();
			uglyRaceLikely = true;

			LOG_BEGIN(threadPoolLoggerName, DEBUG_LOG | 5);
			LOG("TaskManager::run(): after runTask()");
			LOG_END;
		}
		else {
			if( uglyRaceLikely ) {
				wait(1000);
				uglyRaceLikely = false;
			}
			else {
				wait();
			}
		}
	}
}

bool TaskManager::set_task(Runnable *t) {
	bool rc = false;
	if (!task) {
		Lock guard(*this);
		if( !task ) {
			task = t;
			rc = true;
		}
	}

	if (rc) {
		notify();

		LOG_BEGIN(threadPoolLoggerName, DEBUG_LOG | 2);
		LOG("TaskManager::set_task(): after notify");
		LOG_END;
	}
	else {
		LOG_BEGIN(threadPoolLoggerName, DEBUG_LOG | 2);
		LOG("TaskManager::set_task(): got already a task");
		LOG_END;
	}

	return rc;
}

/*--------------------- class ThreadPool --------------------------*/

bool ThreadPool::assign(Runnable* t) 
{
	Lock guard(*this);
	ArrayCursor<TaskManager> cur;
	for (cur.init(&taskList); cur.get(); cur.next()) {
		TaskManager *tm = cur.get();
		if (tm->is_idle()) {
			LOG_BEGIN(threadPoolLoggerName, DEBUG_LOG | 1);
			LOG("ThreadPool::assign(): idle task manager found at (cursor)");
			LOG(cur.get_cursor());
			LOG_END;

			if (tm->set_task(t)) {
				return true;
			}
		}
	}

	return false;
}

void ThreadPool::execute(Runnable *t)
{
	while (!assign(t))
		wait(1000);
}

bool ThreadPool::is_idle() 
{
	Lock guard(*this);
	ArrayCursor<TaskManager> cur;
	for (cur.init(&taskList); cur.get(); cur.next()) {
		if (!cur.get()->is_idle()) {
			return false;
		}
	}
	return true;
}

bool ThreadPool::has_idle() 
{
	bool rc = false;
	Lock guard(*this);
	ArrayCursor<TaskManager> cur;
	// we need to visit all to avoid race conditions
	for (cur.init(&taskList); cur.get(); cur.next()) {
		if (cur.get()->is_idle()) {
			rc = true;
		}
	}
	return rc;
}

ThreadPool::ThreadPool(size_t size)
{
	for (size_t i=0; i<size; i++) {
		taskList.add(new TaskManager(this));
	}
}

ThreadPool::ThreadPool(size_t size, size_t stack_size)
{
	stackSize = stack_size;
	for (size_t i=0; i<size; i++) {
		taskList.add(new TaskManager(this, stackSize));
	}
} 

/*--------------------- class QueuedThreadPool --------------------------*/

QueuedThreadPool::QueuedThreadPool(size_t size)
	: ThreadPool(size)
	, queue()
	, go(false)
{
}

QueuedThreadPool::QueuedThreadPool(size_t size, size_t stack_size)
	: ThreadPool(size, stack_size)
	, queue()
	, go(false)
{
}

QueuedThreadPool::~QueuedThreadPool()
{
	go = false;
	idle_notification();
	join();
}

bool QueuedThreadPool::assign(Runnable* t) 
{
	if (!ThreadPool::assign(t)) {
		LOG_BEGIN(threadPoolLoggerName, DEBUG_LOG | 8);
		LOG("QueuedThreadPool: can't assign job to thread (already pending)");
		LOG(queue.size());
		LOG_END;

		Lock guard(*this);
		queue.add(t);
		/* XXX do not notify run(), 'cause there's no free thread
		       let TaskManager::run() notify us via
		       idle_notification() */
		// notify();
	}
	return true;
}

void QueuedThreadPool::execute(Runnable *t)
{
	if(!is_alive()) {
		LOG_BEGIN(threadPoolLoggerName, WARNING_LOG | 3);
		LOG("QueuedThreadPool: execute() called but job scheduler isn't running");
		LOG_END;
	}
	ThreadPool::execute(t);
}

void QueuedThreadPool::run() 
{
	go = true;
	while (go) {
		LOG_BEGIN(threadPoolLoggerName, DEBUG_LOG | 12);
		LOG("QueuedThreadPool: looking for next event (pending)");
		LOG(queue.size());
		LOG_END;

		/*
		 * it might look inefficient first looping for idle tasks,
		 * but an ugly race condition might prevent assigned tasks
		 * from being executed and is_idle proves such conditions
		 * and resends the action notification
		 */
		while (has_idle() && !queue.empty()) {
			Lock guard(*this);
			Runnable *t = queue.first();
			if (t && ThreadPool::assign(t)) {
				queue.removeFirst();
			}
			else {
				break;
			}
		}

		LOG_BEGIN(threadPoolLoggerName, DEBUG_LOG | 8);
		LOG("QueuedThreadPool: waiting for next event (pending)");
		LOG(queue.size());
		LOG_END;

		// Lock guard(*this);
		wait();
	}
}

#endif // AGENTPP_USE_THREAD_POOL

static const char *mibTaskLoggerName = "agent++.threads.mibtask";

void MibTask::run()
{
	LOG_BEGIN(mibTaskLoggerName, DEBUG_LOG | 8);
	LOG("MibTask::run() for (type) request (id) to (oid)");
	LOG(task->req->get_type());
	LOG(task->req->get_request_id());
	LOG(task->req->get_oid(0).get_printable());
	LOG_END;

	(task->called_class->*task->method)(task->req);       	

	LOG_BEGIN(mibTaskLoggerName, DEBUG_LOG | 8);
	LOG("MibTask::run() finished");
	LOG_END;
}

#ifdef NO_FAST_MUTEXES

LockRequest::LockRequest(Synchronized* s)
{ 
	target = s; 
	lock();
}

LockRequest::~LockRequest()
{
	unlock();
}
 
LockQueue::LockQueue()
{
	go = true;
	start();
}

LockQueue::~LockQueue()
{
	go = false;
	// wake up queue
	notify();
	lock();
	unlock();

	LOG_BEGIN(loggerModuleName, DEBUG_LOG | 1);
	LOG("LockQueue: end queue");
	LOG_END;

	// join thread here, before pending list is deleted 
	if (is_alive()) join();

	LOG_BEGIN(loggerModuleName, DEBUG_LOG | 1);
	LOG("LockQueue: queue stopped");
	LOG_END;

	pendingRelease.clear();
	pendingLock.clear();
}

void LockQueue::run() 
{
	while ((!pendingLock.empty()) || (!pendingRelease.empty()) || (go)) {
		while (!pendingRelease.empty()) {
			lock();
			LockRequest* r = pendingRelease.removeFirst();
			r->target->unlock();
			r->lock();
			r->notify();
			r->unlock();
			unlock();
		}
		int pl = pendingLock.size();
		int pending = pl;
		for (int i=0; i<pl; i++) {
			lock();
			LockRequest* r = pendingLock.removeFirst();
			if (r->target->trylock()) {
				r->lock();
				r->notify();
				r->unlock();
				pending--;
			}
			else {
				pendingLock.addLast(r);
			}
			unlock();
		}
		LOG_BEGIN(loggerModuleName, DEBUG_LOG | 8);
		LOG("LockQueue: waiting for next event (pending)");
		LOG(pending);
		LOG_END;

		// do not wait forever because we cannot 
		// be sure that all instrumentation code notifies
		// us correctly.
		wait(5000);
	}
}

void LockQueue::acquire(LockRequest* r)
{
	lock();
	LOG_BEGIN(loggerModuleName, DEBUG_LOG | 2);
	LOG("LockQueue: adding lock request (ptr)");
	LOG((long)r->target);
	LOG_END;
	pendingLock.addLast(r);
	unlock();
	notify();
}

void LockQueue::release(LockRequest* r)
{
	lock();
	LOG_BEGIN(loggerModuleName, DEBUG_LOG | 2);
	LOG("LockQueue: adding release request (ptr)");
	LOG((long)r->target);
	LOG_END;
	pendingRelease.addLast(r);
	unlock();
	notify();
}


#endif // NO_FAST_MUTEXES

#endif // _THREADS

#ifdef AGENTPP_NAMESPACE
}
#endif
