#!perl -w

use strictures;
use Cwd;
use File::Spec;
use File::Basename;

use Test::More;
use App::Cmd::Tester;

my ( $dist_basedir, $dir, $sqlitedb, $test_config, $cli_config_filename );
my ( $service_config_file, $data_file );

BEGIN
{
    $dist_basedir =
      Cwd::abs_path( File::Spec->catdir( File::Basename::dirname($0), File::Spec->updir() ) );

    my $lib_pl = File::Spec->catfile( $dist_basedir, "t", "lib.pl" );
    do $lib_pl;

    $dir = test_dir();
    $sqlitedb = File::Spec->catfile( $dir, "apdex_$$.sl3" );
    $test_config = {
                     "connect_info" => {
                                         "dsn"      => "dbi:SQLite:dbname=$sqlitedb",
                                         "user"     => "",
                                         "password" => ""
                                      },
                      "log_adapter" => {
                                   name => "Dispatch",
                                   args => [ outputs => [ [ 'Screen', min_level => 'notice' ], ], ],
                      },
                   };

    $cli_config_filename = json_config_file($test_config);

    $ENV{DANCER_CONFDIR} = File::Spec->catdir( $dist_basedir, "test_data" );

    my $cfg = <<EOC;
appname: "Web::App::Perf::Index"
layout: "main"
charset: "UTF-8"
template: "simple"
apdex:
  times:
    out:
      formatter: DateTime::Format::Strptime
      construct:
        pattern: "%s000"
    in:
      formatter: DateTime::Format::Strptime
      construct:
        pattern: "%s"
      separator: "-"
plugins:
  DBIC:
    default:
      dsn: dbi:SQLite:dbname=$sqlitedb
      schema_class: App::Perf::Index::Schema
EOC
    File::Slurp::write_file( File::Spec->catfile( $ENV{DANCER_CONFDIR}, "config.yml" ), $cfg );
}

use CLI::App::Perf::Index;
# the order is important
use Web::App::Perf::Index;
use Dancer::Test;

# we have to repeat the expected to run parts from 010_cli_workflow to get a database we
# can calculate results from
my $result =
  test_app( 'CLI::App::Perf::Index' => [ qw(deploy --configfile), $cli_config_filename ] );
is( $result->output,    '',    'nothing sent to output (deploy)' );
is( $result->error,     undef, 'nothing sent to error (deploy)' );
is( $result->exit_code, 0,     'everything went fine (deploy)' );

$service_config_file = File::Spec->catfile( $dist_basedir, qw(test_data exam1-rules.json) );
$result = test_app(
                    'CLI::App::Perf::Index' => [
                                                 qw(register --configfile), $cli_config_filename,
                                                 '--service-details', $service_config_file
                                               ]
                  );
is( $result->output,    '',    'nothing sent to output registering App One' );
is( $result->error,     undef, 'nothing sent to error registering App One' );
is( $result->exit_code, 0,     'everything went fine registering App One' );

$service_config_file = File::Spec->catfile( $dist_basedir, qw(test_data exam2-rules.json) );
$result = test_app(
                    'CLI::App::Perf::Index' => [
                                                 qw(register --configfile), $cli_config_filename,
                                                 '--service-details', $service_config_file
                                               ]
                  );
is( $result->output, '',    'nothing sent to output registering App Two' );
is( $result->error,  undef, 'nothing sent to error registering App Two' );
is( $result->exit_code, 0, 'everything went fine registering App Two' ) or diag( Dumper($result) );

$data_file = File::Spec->catfile( $dist_basedir, qw(test_data exam1.data) );
$result = test_app(
                   'CLI::App::Perf::Index' => [
                                                qw(import --configfile), $cli_config_filename,
                                                qw(--service ExampleAppOne --input-file), $data_file
                                              ]
                  );
isnt( $result->output, '', 'Got report about imported data for ExampleAppOne' );
if ( $result->output =~ m/Fetched (\d+) rows for ExampleAppOne/ )
{
    is( $1, 20, 'Right numbers of records parsed for ExampleAppOne' );
}
else
{
    like( $result->output,
          qr/Fetched (\d+) rows for ExampleAppOne/,
          'Amount of records parsed for ExampleAppOne' );
}
if ( $result->output =~ m/(\d+) rows: (\d+) converted -- (\d+) skipped -- (\d+) failed/ )
{
    is( $1, 20, 'Right numbers of records processed for ExampleAppOne' );
    is( $2, 18, 'Right numbers of records converted for ExampleAppOne' );
    is( $3, 2,  'Right numbers of records skipped for ExampleAppOne' );
    is( $4, 0,  'Right numbers of records failed for ExampleAppOne' );
}
else
{
    like( $result->output,
          qr/(\d+) rows: (\d+) converted -- (\d+) skipped -- (\d+) failed/,
          'Amount of records processed for ExampleAppOne' );
}

is( $result->error,     undef, 'nothing sent to error importing for App One' );
is( $result->exit_code, 0,     'everything went fine importing for App One' );

$data_file = File::Spec->catfile( $dist_basedir, qw(test_data exam2.data) );
$result = test_app(
                   'CLI::App::Perf::Index' => [
                                                qw(import --configfile), $cli_config_filename,
                                                qw(--service ExampleAppTwo --input-file), $data_file
                                              ]
                  );
isnt( $result->output, '', 'Got report about imported data for ExampleAppTwo' );
if ( $result->output =~ m/Fetched (\d+) rows for ExampleAppTwo/ )
{
    is( $1, 77, 'Right numbers of records parsed for ExampleAppTwo' );
}
else
{
    like( $result->output,
          qr/Fetched (\d+) rows for ExampleAppTwo/,
          'Amount of records parsed for ExampleAppTwo' );
}
if ( $result->output =~ m/(\d+) rows: (\d+) converted -- (\d+) skipped -- (\d+) failed/ )
{
    is( $1, 77, 'Right numbers of records processed for ExampleAppTwo' );
    is( $2, 68, 'Right numbers of records converted for ExampleAppTwo' );
    is( $3, 9,  'Right numbers of records skipped for ExampleAppTwo' );
    is( $4, 0,  'Right numbers of records failed for ExampleAppTwo' );
}
else
{
    like( $result->output,
          qr/(\d+) rows: (\d+) converted -- (\d+) skipped -- (\d+) failed/,
          'Amount of records processed for ExampleAppTwo' );
}

is( $result->error,     undef, 'nothing sent to error importing for App Two' );
is( $result->exit_code, 0,     'everything went fine importing for App Two' );

# now we're beginning with the real Dancer tests ...
foreach my $route (
            qw(/apdex),
            qw(/apdex/ExampleAppOne /apdex/ExampleAppOne/Local /apdex/ExampleAppOne/Local/GB),
            qw(/apdex/ExampleAppTwo /apdex/ExampleAppTwo/Internal /apdex/ExampleAppTwo/Internet/DE),
            qw(/info/buckets/ExampleAppOne /info/buckets/ExampleAppTwo)
  )
{
    route_exists [ GET => $route ], "a route handler is defined for $route";
    response_status_is [ 'GET' => $route ], 200, "response status is 200 for $route";
}

my ( $response, $content );
my $json = JSON->new();
use Data::Dumper;

my %route_responses = (
               '/apdex/ExampleAppOne' => {
                                           ExampleAppOne => {
                                                              Local      => {},
                                                              HereDyna   => {},
                                                              RemoteDyna => {}
                                                            }
                                         },
               '/apdex/ExampleAppOne/Local' => {
                                                 ExampleAppOne => {
                                                                    Local => {
                                                                               '-'  => {},
                                                                               'NL' => {},
                                                                               'GB' => {},
                                                                               'IT' => {}
                                                                             }
                                                                  }
                                               },
               '/apdex/ExampleAppOne/Local/GB' => {
                                                 ExampleAppOne => {
								   Local => { GB => [ 1349049600000, 1349049600000 ] }
								  }
						  },
               '/apdex/ExampleAppTwo' => {
                                           ExampleAppTwo => {
                                                              Internal => {},
                                                              Internet => {}
                                                            }
                                         },
               '/apdex/ExampleAppTwo/Internal' => { ExampleAppTwo => { Internal => { GB => {} } } },
               '/apdex/ExampleAppTwo/*'        => {
                                             ExampleAppTwo => {
                                                                'Internal' => { 'GB' => {} },
                                                                'Internet' => {
                                                                                '-'  => {},
                                                                                'DE' => {},
                                                                                'CZ' => {},
                                                                                'ZA' => {},
                                                                                'GB' => {}
                                                                              }
                                                              }
                                           },
               '/apdex/ExampleAppTwo/*/*' => {
                                         ExampleAppTwo => {
                                             Internal => { GB => [ 1320105600000, 1325635200000 ] },
                                             Internet => {
                                                           '-' => [ 1301616000000, 1302307200000 ],
                                                           DE  => [ 1301616000000, 1302307200000 ],
                                                           CZ  => [ 1301616000000, 1302307200000 ],
                                                           ZA  => [ 1301616000000, 1302307200000 ],
                                                           GB  => [ 1301616000000, 1325656800000 ]
                                                         }
                                         }
               },
);

foreach my $path ( sort keys %route_responses )
{
    $response = Dancer::Test::_req_to_response( [ GET => $path ] );
    is( $response->status, 200, "response status is 200 for $path" );
    $content = $json->decode( $response->{content} );
    is_deeply( $content, $route_responses{$path}, "Content computes for $path" )
      or diag( Dumper( $content, $route_responses{$path} ) );
}

done_testing;
