#!perl -w

use strictures;
use Cwd;
use File::Spec;
use File::Basename;

use Test::More;
use App::Cmd::Tester;
use Data::Dumper;

my $dist_basedir =
  Cwd::abs_path( File::Spec->catdir( File::Basename::dirname($0), File::Spec->updir() ) );

use_ok("CLI::App::Perf::Index");

my $lib_pl = File::Spec->catfile( $dist_basedir, "t", "lib.pl" );
do $lib_pl;

my $dir = test_dir();
my $sqlitedb = File::Spec->catfile( $dir, "apdex_$$.sl3" );
my $test_config = {
                    "connect_info" => {
                                        "dsn"      => "dbi:SQLite:dbname=$sqlitedb",
                                        "user"     => "",
                                        "password" => ""
                                      },
                      "log_adapter" => {
                                   name => "Dispatch",
                                   args => [ outputs => [ [ 'Screen', min_level => 'notice' ], ], ],
                      },
                  };

my $cli_config_filename = json_config_file($test_config);
my ( $service_config_file, $data_file );

my $result =
  test_app( 'CLI::App::Perf::Index' => [ qw(deploy --config-file), $cli_config_filename ] );
like( $result->error, qr/Mandatory parameter 'connect_info' missing in call to/, 'wrong argument' );
isnt( $result->exit_code, 0, 'exit code shows error (no connect_info)' );

$result = test_app( 'CLI::App::Perf::Index' => [ qw(deploy --configfile), $cli_config_filename ] );
is( $result->output,    '',    'nothing sent to output (deploy)' );
is( $result->error,     undef, 'nothing sent to error (deploy)' );
is( $result->exit_code, 0,     'everything went fine (deploy)' );

$result = test_app( 'CLI::App::Perf::Index' => [ qw(deploy --configfile), $cli_config_filename ] );
like( $result->stderr, qr/table \w+ already exists/, 'Tables already exists' );
isnt( $result->exit_code, 0, 'exit code shows error (db exists)' );

$result =
  test_app( 'CLI::App::Perf::Index' => [ qw(register --configfile), $cli_config_filename ] );
like( $result->error,
      qr/Mandatory parameter 'service_details' missing in call to/,
      'missing argument' );
isnt( $result->exit_code, 0, 'exit code shows error (no service_details)' );

$service_config_file = File::Spec->catfile( $dist_basedir, qw(test_data exam1-rules.json) );
$result = test_app(
                    'CLI::App::Perf::Index' => [
                                                 qw(register --configfile), $cli_config_filename,
                                                 '--service-details', $service_config_file
                                               ]
                  );
is( $result->output,    '',    'nothing sent to output registering App One' );
is( $result->error,     undef, 'nothing sent to error registering App One' );
is( $result->exit_code, 0,     'everything went fine registering App One' );

$service_config_file = File::Spec->catfile( $dist_basedir, qw(test_data exam2-rules.json) );
$result = test_app(
                    'CLI::App::Perf::Index' => [
                                                 qw(register --configfile), $cli_config_filename,
                                                 '--service-details', $service_config_file
                                               ]
                  );
is( $result->output, '',    'nothing sent to output registering App Two' );
is( $result->error,  undef, 'nothing sent to error registering App Two' );
is( $result->exit_code, 0, 'everything went fine registering App Two' ) or diag( Dumper($result) );

$data_file = File::Spec->catfile( $dist_basedir, qw(test_data exam1.data) );
$result = test_app(
                   'CLI::App::Perf::Index' => [
                                                qw(import --configfile), $cli_config_filename,
                                                qw(--service ExampleAppOne --input-file), $data_file
                                              ]
                  );
isnt( $result->output, '', 'Got report about imported data for ExampleAppOne' );
if ( $result->output =~ m/Fetched (\d+) rows for ExampleAppOne/ )
{
    is( $1, 20, 'Right numbers of records parsed for ExampleAppOne' );
}
else
{
    like( $result->output,
          qr/Fetched (\d+) rows for ExampleAppOne/,
          'Amount of records parsed for ExampleAppOne' );
}
if ( $result->output =~ m/(\d+) rows: (\d+) converted -- (\d+) skipped -- (\d+) failed/ )
{
    is( $1, 20, 'Right numbers of records processed for ExampleAppOne' );
    is( $2, 18, 'Right numbers of records converted for ExampleAppOne' );
    is( $3, 2,  'Right numbers of records skipped for ExampleAppOne' );
    is( $4, 0,  'Right numbers of records failed for ExampleAppOne' );
}
else
{
    like( $result->output,
          qr/(\d+) rows: (\d+) converted -- (\d+) skipped -- (\d+) failed/,
          'Amount of records processed for ExampleAppOne' );
}

is( $result->error,     undef, 'nothing sent to error importing for App One' );
is( $result->exit_code, 0,     'everything went fine importing for App One' );

$data_file = File::Spec->catfile( $dist_basedir, qw(test_data exam2.data) );
$result = test_app(
                   'CLI::App::Perf::Index' => [
                                                qw(import --configfile), $cli_config_filename,
                                                qw(--service ExampleAppTwo --input-file), $data_file
                                              ]
                  );
isnt( $result->output, '', 'Got report about imported data for ExampleAppTwo' );
if ( $result->output =~ m/Fetched (\d+) rows for ExampleAppTwo/ )
{
    is( $1, 77, 'Right numbers of records parsed for ExampleAppTwo' );
}
else
{
    like( $result->output,
          qr/Fetched (\d+) rows for ExampleAppTwo/,
          'Amount of records parsed for ExampleAppTwo' );
}
if ( $result->output =~ m/(\d+) rows: (\d+) converted -- (\d+) skipped -- (\d+) failed/ )
{
    is( $1, 77, 'Right numbers of records processed for ExampleAppTwo' );
    is( $2, 68, 'Right numbers of records converted for ExampleAppTwo' );
    is( $3, 9,  'Right numbers of records skipped for ExampleAppTwo' );
    is( $4, 0,  'Right numbers of records failed for ExampleAppTwo' );
}
else
{
    like( $result->output,
          qr/(\d+) rows: (\d+) converted -- (\d+) skipped -- (\d+) failed/,
          'Amount of records processed for ExampleAppTwo' );
}

is( $result->error,     undef, 'nothing sent to error importing for App Two' );
is( $result->exit_code, 0,     'everything went fine importing for App Two' );

done_testing;
