package Web::App::Perf::Index;

# ABSTRACT: Web-App to query Application Performance Index

use v5.14;
use strictures;

use Module::Runtime qw(require_module);

use Dancer ':syntax';
use App::Perf::Index;
use List::Util qw(reduce);
use Scalar::Util qw(looks_like_number);
use Params::Util qw(_STRING);

use Dancer::Cookies qw();
use Dancer::Plugin::DBIC 'schema';

use MIME::Base64 qw(encode_base64 decode_base64);

use DateTime;
use Date::Parse;

get '/' => sub {
    template 'index';
};

sub _formatter
{
    my $cfg = shift;
    require_module( $cfg->{formatter} );
    # XXX "HASH" eq ..., "ARRAY" eq ...
    my $fmter = $cfg->{formatter}->new( %{ $cfg->{construct} } );
    return $fmter;
}

sub _apdex_thresholds
{
    my ( $service_name, $category_name, $satisfying, $tolerable ) = @_;
    my $cookie_name = join( "_", "apdex", $service_name, $category_name );
    my $cookies = Dancer::Cookies->cookies();
    if ( defined( $cookies->{$cookie_name} ) )
    {
        my $cat_cookie = from_json( decode_base64( $cookies->{$cookie_name}->value ) );
        ( $satisfying, $tolerable ) =
          ( $cat_cookie->{sat} // $satisfying, $cat_cookie->{tol} // $tolerable );
    }

    return ( $satisfying, $tolerable );
}

sub _get_services
{
    my @services = schema->resultset('Service')->all;
    +{ map { $_->service_name => {} } @services };
}

sub _checked_get_args
{
    defined( $_[0] ) or return [];
    _STRING( $_[0] ) or return "Invalid argument for $_[1]";
    $_[0] eq "*" and return [];
    $_[0] =~ m/[^\w,]/ and return "Invalid characters for $_[1]";
    [ split( ",", $_[0] ) ];
}

sub _check_time_args
{
    _STRING( $_[0] ) or return "Invalid arguments for $_[1]";
    $_[0] eq "*" and return [ [] ];

    my @time_args = split( "-", $_[0] );
    scalar(@time_args) == 2 or return "Time frame must be from-until[,granularity]";
    splice( @time_args, $#time_args, 1, split( ",", $time_args[-1] ) );
    scalar(@time_args) <= 3 or return "Time frame must be from-until[,granularity]";

    my ( $from, $until, @time_frames );
    $from = DateTime->from_epoch(
                                  epoch => (
                                             looks_like_number( $time_args[0] )
                                             ? $time_args[0]
                                             : Date::Parse::str2time( $time_args[0] )
                                           )
                                );
    $until = DateTime->from_epoch(
                                   epoch => (
                                              looks_like_number( $time_args[1] )
                                              ? $time_args[1]
                                              : Date::Parse::str2time( $time_args[1] )
                                            )
                                 );

    my $now = DateTime->now();
    $from > $now  and return "from time is in future";
    $until > $now and return "until time is in future";

    if ( scalar(@time_args) == 3 )
    {
        my ( $dur, $spec );
        my %since;

	no warnings 'experimental';
        given ( $time_args[2] )
        {
            when (/^(year|month|week|hour)ly$/)
            {
                $spec = $1;
                $dur = DateTime::Duration->new( $spec . "s", 1, end_of_month => "limit" );
            }
            when (/^quarterly$/)
            {
                $spec = "month";
                $dur = DateTime::Duration->new( months       => 3,
                                                end_of_month => "limit" );
            }
            when (/^daily/)
            {
                $spec = "day";
                $dur = DateTime::Duration->new( days         => 1,
                                                end_of_month => "limit" );
            }
            default { die "Time frame must be from-until[,granularity]"; }
        }

        foreach my $unit (qw(year month day hour))
        {
            my $offset = $unit eq $spec ? $from->$unit() % $dur->in_units("${spec}s") : 0;
            $since{$unit} = $from->$unit() - $offset;
            $unit eq $spec and last;
        }

        debug( "since: ", \%since );
        $from = DateTime->new( %since, time_zone => "UTC" );

        while ( DateTime->compare( $from, $until ) <= 0 )
        {
            my $begin = $from->epoch();
            $from = $from + $dur;
            my $end = $from->epoch() - 1;
            push( @time_frames, [ $begin, $end ] );
        }
    }
    else
    {
        @time_frames = ( [ $from->epoch(), $until->epoch() ] );
    }

    debug( "time frames: ", \@time_frames );

    \@time_frames;
}

sub _get_categories
{
    my ( $services, $categories ) = @_;
    $services = _checked_get_args( $services, "services" );
    ref $services or return [$services];
    $categories = _checked_get_args( $categories, "categories" );
    ref $categories or return [$categories];
    my %filter;
    scalar(@$services) and $filter{service_name} = $services;
    scalar(@$categories) and $filter{"categories.category_name"} = $categories;
    my $rs = schema->resultset('Service')->search(
                                                   \%filter,
                                                   {
                                                     prefetch => [qw/categories/],
                                                     distinct => 1
                                                   }
                                                 );

    my %categories;
    while ( my $srv_rs = $rs->next )
    {
        my $cats = $srv_rs->categories;
        while ( my $cat_rs = $cats->next )
        {
            $categories{ $srv_rs->service_name }->{ $cat_rs->category_name } = {};
        }
    }

    \%categories;
}

sub _get_regions
{
    my ( $services, $categories ) = @_;
    $services = _checked_get_args( $services, "services" );
    ref $services or return [$services];
    $categories = _checked_get_args( $categories, "categories" );
    ref $categories or return [$categories];

    my %filter;
    scalar(@$services) and $filter{service_name} = $services;
    scalar(@$categories) and $filter{"categories.category_name"} = $categories;
    my $rs = schema->resultset('Service')->search(
                         \%filter,
                         {
                           join    => [ { categories => [ { performance_datas => 'region' } ] }, ],
                           columns => [
                                        'me.service_id',
                                        'me.service_name',
                                        { categories_category_id   => 'categories.category_id' },
                                        { categories_category_name => 'categories.category_name' },
                                        { region_region_id         => 'region.region_id' },
                                        { region_alpha2_code       => 'region.alpha2_code' },
                                      ],
                           distinct => 1,
                           cache    => 1,
                         }
    );

    my %regions;
    while ( my $srv = $rs->next )
    {
        $regions{ $srv->service_name }->{ $srv->get_column('categories_category_name') }
          ->{ $srv->get_column('region_alpha2_code') } = {};
    }

    \%regions;
}

sub _get_time_specs
{
    my ( $services, $categories, $regions ) = @_;

    $services = _checked_get_args( $services, "services" );
    ref $services or return [$services];
    $categories = _checked_get_args( $categories, "categories" );
    ref $categories or return [$categories];
    $regions = _checked_get_args( $regions, "regions" );
    ref $regions or return [$regions];

    my %filter;
    scalar(@$services)   and $filter{service_name}               = $services;
    scalar(@$categories) and $filter{"categories.category_name"} = $categories;
    scalar(@$regions)    and $filter{"region.alpha2_code"}       = $regions;

    my %times;

    my $rs = schema->resultset('Service')->search(
                       \%filter,
                       {
                         join    => [ { categories => [ { performance_datas => 'region' } ] }, ],
                         columns => [
                                      'me.service_id',
                                      'me.service_name',
                                      { categories_category_id    => 'categories.category_id' },
                                      { categories_category_name  => 'categories.category_name' },
                                      { region_region_id          => 'region.region_id' },
                                      { region_alpha2_code        => 'region.alpha2_code' },
                                      { performance_datas_perf_id => 'performance_datas.perf_id' },
                                      {
                                        performance_datas_min_time =>
                                          { min => 'performance_datas.time_slice' }
                                      },
                                      {
                                        performance_datas_max_time =>
                                          { max => 'performance_datas.time_slice' }
                                      },
                                    ],
                         group_by =>
                           [qw(me.service_name categories.category_name region.alpha2_code)],
                         cache => 1,
                       }
    );

    my $cookies = Dancer::Cookies->cookies();
    my $fmter =
      _formatter( from_json( decode_base64( $cookies->{apdex}->value ) )->{times}->{out} );
    while ( my $srv = $rs->next )
    {
        my $min_dt = DateTime->from_epoch( epoch => $srv->get_column('performance_datas_min_time'),
                                           formatter => $fmter );
        my $max_dt = DateTime->from_epoch( epoch => $srv->get_column('performance_datas_max_time'),
                                           formatter => $fmter );
        # on wish of customer deliver interger formats as number but as string
        my ( $fmt_min_dt, $fmt_max_dt ) = ( "" . $min_dt, "" . $max_dt );    # apply format
        looks_like_number($fmt_min_dt)
          and ( 0 + $fmt_min_dt ) == $fmt_min_dt
          and $fmt_min_dt = 0 + $fmt_min_dt;
        looks_like_number($fmt_max_dt)
          and ( 0 + $fmt_max_dt ) == $fmt_max_dt
          and $fmt_max_dt = 0 + $fmt_max_dt;

        $times{ $srv->service_name }->{ $srv->get_column('categories_category_name') }
          ->{ $srv->get_column('region_alpha2_code') } = [ $fmt_min_dt, $fmt_max_dt ];
    }

    \%times;
}

sub _get_apdex
{
    my ( $services, $categories, $regions, $time_spec ) = @_;

    $services = _checked_get_args( $services, "services" );
    ref $services or return [$services];
    $categories = _checked_get_args( $categories, "categories" );
    ref $categories or return [$categories];
    $regions = _checked_get_args( $regions, "regions" );
    ref $regions or return [$regions];
    $time_spec = _check_time_args($time_spec);
    ref $time_spec or return [$time_spec];

    my %filter;
    scalar(@$services)   and $filter{service_name}               = $services;
    scalar(@$categories) and $filter{"categories.category_name"} = $categories;
    scalar(@$regions)    and $filter{"region.alpha2_code"}       = $regions;

    my @time_where;
    no warnings 'experimental';
    given ( scalar(@$time_spec) )
    {
        when (0) { }
        when (1)
        {
            scalar( @{ $time_spec->[0] } )
              and @time_where =
              ("performance_datas.time_slice BETWEEN ? AND ?")
        }
        default { @time_where = ("performance_datas.time_slice BETWEEN ? AND ?") }
    }

    my $where = join( " AND ",
                      @time_where,
                      map { $_ . " IN (" . join( ",", ("?") x scalar( @{ $filter{$_} } ) ) . ")" }
                        sort keys %filter );
    $where and $where = "WHERE ( $where )";
    my $buckets =
      join( ", ", ( map +("SUM(performance_datas.$_)"), ( "bucket00" .. "bucket19" ) ) );
    my $sql = <<EOQ;
SELECT me.service_name, categories.category_name,
       region.alpha2_code,
       categories.satisfy_threshold, categories.tolerable_threshold,
       $buckets
FROM SERVICES me
     LEFT JOIN CATEGORIES categories ON
	       categories.service_id = me.service_id
     LEFT JOIN PERFORMANCE_DATA performance_datas ON
	       performance_datas.category_id =
	       categories.category_id
     LEFT JOIN REGIONS region ON
	       region.region_id = performance_datas.region_id
$where
GROUP BY me.service_name, categories.category_name,
	 region.alpha2_code
EOQ
    my $sth = schema->storage->dbh_do( sub { $_[1]->prepare( $_[2] ) }, $sql );
    my @params = map { @{ $filter{$_} } } sort keys %filter;

    my %apdex;
    my %rollups;
    my $cookies = Dancer::Cookies->cookies();
    my $fmter =
      _formatter( from_json( decode_base64( $cookies->{apdex}->value ) )->{times}->{out} );
    my %thres;
    foreach my $time_frame (@$time_spec)
    {
        debug( $sql, ": ", join( " -- ", @time_where ? ( @$time_frame, @params ) : @params ) );
        $sth->execute( @time_where ? ( @$time_frame, @params ) : @params );
        my $service_buckets = $sth->fetchall_arrayref();

        foreach my $row (@$service_buckets)
        {
            my ( $sat_thres, $tol_thres ) =
              @{ $thres{ $row->[0] . '_' . $row->[1] } //=
                  [ _apdex_thresholds( $row->[0], $row->[1], $row->[3], $row->[4] ) ] };
            my @buckets = @{$row}[ 5 .. $#$row ];

            my @sat_sums = grep { $_ } @buckets[ 0 .. $sat_thres ];
            debug( "sat_sums: ", \@sat_sums );
            my $sat = reduce { $a + $b } @sat_sums;
            $sat //= 0;

            my @tol_sums = grep { $_ } @buckets[ $sat_thres + 1 .. $tol_thres ];
            debug( "tol_sums: ", \@tol_sums );
            my $tol = reduce { $a + $b } @tol_sums;
            $tol //= 0;

            my @all_sums = grep { $_ } @buckets;
            debug( "all_sums: ", \@all_sums );
            my $all = reduce { $a + $b } grep { $_ } @buckets;

            $apdex{ $row->[0] }->{ $row->[1] }->{ $row->[2] } //= [];
            my $apdex_val = $all ? ( $sat + ( $tol / 2 ) ) / $all : 0;
            my $dt = DateTime->from_epoch( epoch => $time_frame->[0] // 0,
                                           formatter => $fmter );
            my $fmt_dt = "" . $dt;    # apply format
            looks_like_number($fmt_dt) and ( 0 + $fmt_dt ) == $fmt_dt and $fmt_dt = 0 + $fmt_dt;
            push( @{ $apdex{ $row->[0] }->{ $row->[1] }->{ $row->[2] } }, [ $fmt_dt, $apdex_val, 0 + $sat, 0 + $tol, $all - $sat - $tol ] );

            my $rollup_frames = $rollups{ $row->[0] }->{ $row->[1] }->{ $row->[2] }->{ROLLUP} //= {
                                                                                           sat => 0,
                                                                                           tol => 0,
                                                                                           tot => 0
            };
            my $rollup_reg = $rollups{ $row->[0] }->{ $row->[1] }->{ROLLUP} //= {
                                                                                  sat => 0,
                                                                                  tol => 0,
                                                                                  tot => 0
                                                                                };
            my $rollup_cat = $rollups{ $row->[0] }->{ROLLUP} //= {
                                                                   sat => 0,
                                                                   tol => 0,
                                                                   tot => 0
                                                                 };

            foreach my $rollup ( $rollup_frames, $rollup_reg, $rollup_cat )
            {
                $rollup->{sat} += $sat;
                $rollup->{tol} += $tol;
                $rollup->{tot} += $all;
            }
        }
    }

    foreach my $serv_nm ( keys %rollups )
    {
        my ( $sat, $tol, $all ) = @{ $rollups{$serv_nm}->{ROLLUP} }{qw(sat tol tot)};
        my $apdex_val = $all ? ( $sat + ( $tol / 2 ) ) / $all : 0;
        $apdex{$serv_nm}->{ROLLUP} = [ undef, $apdex_val, 0 + $sat, 0 + $tol, $all - $sat - $tol ];

        foreach my $cat_nm ( keys %{ $rollups{$serv_nm} } )
        {
            $cat_nm eq "ROLLUP" and next;
            ( $sat, $tol, $all ) = @{ $rollups{$serv_nm}->{$cat_nm}->{ROLLUP} }{qw(sat tol tot)};
            $apdex_val = $all ? ( $sat + ( $tol / 2 ) ) / $all : 0;
            $apdex{$serv_nm}->{$cat_nm}->{ROLLUP} = [ undef, $apdex_val, 0 + $sat, 0 + $tol, $all - $sat - $tol ];

            foreach my $reg_nm ( keys %{ $rollups{$serv_nm}->{$cat_nm} } )
            {
                $reg_nm eq "ROLLUP" and next;
                ( $sat, $tol, $all ) =
                  @{ $rollups{$serv_nm}->{$cat_nm}->{$reg_nm}->{ROLLUP} }{qw(sat tol tot)};
                $apdex_val = $all ? ( $sat + ( $tol / 2 ) ) / $all : 0;
                push( @{ $apdex{$serv_nm}->{$cat_nm}->{$reg_nm} }, [ "ROLLUP", $apdex_val, 0 + $sat, 0 + $tol, $all - $sat - $tol ] );
            }
        }
    }

    \%apdex;
}

sub _get_service_buckets
{
    my $services = shift;

    $services = _checked_get_args( $services, "services" );
    ref $services or return [$services];

    my %filter;
    scalar(@$services) and $filter{service_name} = $services;
    my $rs = schema->resultset('Service')->search(
                                                   \%filter,
                                                   {
                                                     prefetch => [qw/transform/],
                                                     distinct => 1
                                                   }
                                                 );

    my %buckets;
    while ( my $srv_rs = $rs->next )
    {
        my %tr = %{ $srv_rs->transform->{_column_data} };    # transform rules
        App::Perf::Index->_replace_vars( { BUCKET => '$_' }, \%tr );
        my $buck2time_str = <<"EOC";
sub {
    map { [ \$_, $tr{revbuck} ] } (1..20)
};
EOC
        my $buck2time = eval $buck2time_str;
        $buckets{ $srv_rs->service_name } = [ &{$buck2time} ];
    }

    \%buckets;
}

sub _never_expires
{
    my $now    = DateTime->now;
    my $dur    = DateTime::Duration->new( years => 5 );    # I define this is 'never'
    my $future = $now + $dur;
    $future->epoch - $now->epoch;
}

sub _default_cookies
{
    my $cookies = Dancer::Cookies->cookies();
    my $delta_secs;

    if (@_)
    {
        my ( $services, $categories ) = @_;
        $services = _checked_get_args( $services, "services" );
        ref $services or return [$services];
        $categories = _checked_get_args( $categories, "categories" );
        ref $categories or return [$categories];
        my %filter;
        scalar(@$services) and $filter{service_name} = $services;
        scalar(@$categories) and $filter{"categories.category_name"} = $categories;
        my $rs = schema->resultset('Service')->search(
                                                       \%filter,
                                                       {
                                                         prefetch => [qw/categories/],
                                                         distinct => 1
                                                       }
                                                     );

        my %categories;
        while ( my $srv_rs = $rs->next )
        {
            my $cats = $srv_rs->categories;
            while ( my $cat_rs = $cats->next )
            {
                my $cookie_name =
                  join( "_", "apdex", $srv_rs->service_name, $cat_rs->category_name );
                unless ( defined( $cookies->{$cookie_name} ) )
                {
                    $delta_secs //= _never_expires;

                    my $cookie_value =
                      encode_base64(
                                     to_json(
                                              {
                                                sat => $cat_rs->satisfy_threshold,
                                                tol => $cat_rs->tolerable_threshold,
                                              }
                                            )
                                   );
                    cookie $cookie_name => $cookie_value,
                      expire            => $delta_secs;
                }
            }
        }
    }

    defined $cookies->{apdex}
      or cookie
      apdex  => encode_base64( to_json( config->{apdex} ) ),
      expire => $delta_secs //= _never_expires;

    return;
}

get '/apdex' => sub {
    to_json(_get_services);
};

get '/info/buckets/:services' => sub {
    my $services = param('services');

    return to_json( _get_service_buckets($services) );
};

get '/apdex/**' => sub {
    my @tags = splat;
    my ( $services, $categories, $regions, $time_spec ) = @{ $tags[0] };

    defined $services   or return to_json(_get_services);
    defined $categories or return to_json( _get_categories($services) );

    _default_cookies( $services, $categories );

    defined $regions or return to_json( _get_regions( $services, $categories ) );
    defined $time_spec or return to_json( _get_time_specs( $services, $categories, $regions ) );

    return to_json( _get_apdex( $services, $categories, $regions, $time_spec ) );
};

true;
