package CLI::App::Perf::Index::Role::Register;

use strictures;
use Moose::Role;

# ABSTRACT: registers new service

with "CLI::App::Perf::Index::Role::ServiceDB", "CLI::App::Perf::Index::Role::Log";

use App::Perf::Index::Schema;

# ABSTRACT: register new service to APDex

use JSON;
use File::Slurp;
use Params::Util qw(_ARRAY _HASH);

# XXX add constrain to validate file exists ...
# service-details must point to a valid filename
# $opt->{"service_details"}
#   or !-f $opt->{"service_details"}
#   or $self->usage_error("Nothing to register");
# e.g. by using MX::Type::File

has 'service_details' => (
                           traits        => [qw(Getopt)],
                           isa           => 'Str',
                           is            => 'rw',
                           cmd_flag      => 'service-details',
                           cmd_aliases   => 's',
                           documentation => 'file containing the service details',
                           required      => 1,
                         );

has 'service_info' => (
                        traits   => [qw(NoGetopt)],
                        is       => 'ro',
                        builder  => 'load_service',
                        init_arg => undef,
                        lazy     => 1
                      );

sub load_service
{
    my ($self) = @_;
    my ( $json, $obj, $logger );

    $logger = $self->logger;
    eval {
        my $service_details = read_file( $self->service_details );
        $json = JSON->new();
        $obj  = $json->decode($service_details);
    };
    $@ and $logger->emergency($@);

    defined( $obj->{SERVICE} )
      or $logger->emergency("Error: service details without 'SERVICE' description");
    defined( $obj->{SERVICE}->{NAME} )
      or $logger->emergency("Error: service description without 'NAME'");

    my $i = 0;
    _ARRAY( $obj->{SERVICE}->{CATEGORIES} )
      or $logger->emergency("Error: service description without 'CATEGORIES'");
    foreach my $cat ( @{ $obj->{SERVICE}->{CATEGORIES} } )
    {
        ++$i;
        defined( $cat->{NAME} ) or $logger->emergency("Error: category $i without 'NAME'");
        defined( $cat->{SATISFY_THRESHOLD} )
          or $logger->emergency(
                          "Error: category $i (" . $cat->{NAME} . ") without 'SATISFY_THRESHOLD'" );
        defined( $cat->{TOLERABLE_THRESHOLD} )
          or $logger->emergency(
                        "Error: category $i (" . $cat->{NAME} . ") without 'TOLERABLE_THRESHOLD'" );
    }

    $obj->{VARS} //= {};
    defined( $obj->{TRANSFORM} )
      or $logger->emergency("Error: service details without 'TRANSFORM' rules");
    for my $ensure (qw(BUCKET REVBUCK TIME_SLICE REGION CATEGORY))
    {
        defined( $obj->{TRANSFORM}->{$ensure} )
          or $logger->emergency("Error: service details without 'transform' rule for '$ensure'");
    }

    defined( $obj->{FETCH} )
      or $logger->emergency("Error: service details without 'FETCH' parameters");
    defined( $obj->{FETCH}->{DRIVER} )
      or $logger->emergency("Error: service details without 'fetch' parameter for 'DRIVER' to use");
    defined( $obj->{FETCH}->{QUERY} )
      or $logger->emergency("Error: service details without 'fetch' parameter for 'QUERY' to use");

    $obj->{FETCH}->{USERNAME} //= "";
    $obj->{FETCH}->{PASSWORD} //= "";
    $obj->{FETCH}->{ATTRS}    //= {};

    App::Perf::Index->_replace_vars( $obj->{VARS}, $obj->{TRANSFORM} );

    $obj->{FETCH}->{ATTRS} = $json->encode( $obj->{FETCH}->{ATTRS} );

    return $obj;
}

sub register
{
    my $self = shift;
    my $logger = $self->logger;

    $logger->trace("going to load service details") if $logger->is_trace();
    my $service = $self->service_info();
    $logger->trace("service details loaded") if $logger->is_trace();

    my $cr_vals = {
                    service_name => $service->{SERVICE}->{NAME},
                    fetch        => {
                               driver        => $service->{FETCH}->{DRIVER},
                               username      => $service->{FETCH}->{USERNAME},
                               password      => $service->{FETCH}->{PASSWORD},
                               attrs         => $service->{FETCH}->{ATTRS},
                               query         => $service->{FETCH}->{QUERY},
                               required_mods => $service->{FETCH}->{REQUIRE},
                             },
                    transform => {
                                   skip          => $service->{TRANSFORM}->{SKIP},
                                   assert        => $service->{TRANSFORM}->{ASSERT},
                                   bucket        => $service->{TRANSFORM}->{BUCKET},
                                   revbuck       => $service->{TRANSFORM}->{REVBUCK},
                                   time_slice    => $service->{TRANSFORM}->{TIME_SLICE},
                                   region        => $service->{TRANSFORM}->{REGION},
                                   category      => $service->{TRANSFORM}->{CATEGORY},
                                   used_mods     => $service->{TRANSFORM}->{USE},
                                   required_mods => $service->{TRANSFORM}->{REQUIRE},
                                 },
                  };

    foreach my $cat ( @{ $service->{SERVICE}->{CATEGORIES} } )
    {
        push(
              @{ $cr_vals->{categories} } => {
                                               category_name       => $cat->{NAME},
                                               satisfy_threshold   => $cat->{SATISFY_THRESHOLD},
                                               tolerable_threshold => $cat->{TOLERABLE_THRESHOLD}
                                             }
            );
    }

    my $schema = $self->schema;
    $logger->tracef( "going to register service %s", $service->{SERVICE}->{NAME} )
      if $logger->is_trace();
    my $rs = $schema->resultset('Service');
    $rs->create($cr_vals);
    $logger->infof( "service %s registered", $service->{SERVICE}->{NAME} ) if $logger->is_info();

    return 0;
}

1;
