package CLI::App::Perf::Index::Role::Import;

use strictures;
use Moose::Role;

# ABSTRACT: imports new data into a service

with "CLI::App::Perf::Index::Role::ServiceDB", "CLI::App::Perf::Index::Role::Log";

use App::Perf::Index::Schema;
use App::Perf::Index;

use B::Deparse;
use Data::Dumper;

use Module::Runtime qw(require_module);

# ABSTRACT: import new performance data for service

use DBI;

has service => (
                 traits        => [qw(Getopt)],
                 isa           => 'Str',
                 is            => 'rw',
                 cmd_aliases   => 'S',
                 documentation => 'choose the service for importing',
                 required      => 1,
               );

# XXX add contrain to validate file exists ...
has 'input_file' => (
                      traits        => [qw(Getopt)],
                      isa           => 'Str',
                      is            => 'rw',
                      cmd_flag      => 'input-file',
                      cmd_aliases   => 'f',
                      documentation => 'File to be imported',
                      required      => 1,
                    );

sub service_config
{
    my ($self) = @_;

    my $logger = $self->logger;
    my $schema = $self->schema;

    $logger->trace("fetching service configuration from database") if $logger->is_trace;
    my @services =
      $schema->resultset('Service')
      ->search( { service_name => $self->service }, { join => [qw/fetch transform/] } );
    scalar(@services) == 1
      or die "Searching for service '"
      . $self->service
      . "' delivers "
      . scalar(@services)
      . "rows, expected 1";
    my %tr = %{ $services[0]->transform->{_column_data} };    # transform rules
    my %fr = %{ $services[0]->fetch->{_column_data} };        # fetch rules
    $logger->trace("fetched service configuration from database") if $logger->is_trace;

    my $json = JSON->new();
    $fr{attrs} = $json->decode( $fr{attrs} );

    my %opts;
    $opts{input_file} = $self->input_file;
    $opts{input_file} eq "-" and $opts{input_file} = \*STDIN;
    App::Perf::Index->_replace_vars( \%opts, $fr{attrs} );

    my @categories =
      $schema->resultset('Category')->search( { service_id => $services[0]->service_id } );
    $logger->trace("fetched categories from database") if $logger->is_trace;

    my %category_map = map { $_->category_name, $_->category_id } @categories;

    # find region data
    my @regions = $schema->resultset('Region')->all;
    $logger->trace("fetched regions from database") if $logger->is_trace;

    my %alpha2_codes  = map { $_->alpha2_code,  $_->region_id } @regions;
    my %alpha3_codes  = map { $_->alpha3_code,  $_->region_id } @regions;
    my %numeric_codes = map { $_->numeric_code, $_->region_id } @regions;

    return {
             tr            => \%tr,
             fr            => \%fr,
             category_map  => \%category_map,
             alpha2_codes  => \%alpha2_codes,
             alpha3_codes  => \%alpha3_codes,
             numeric_codes => \%numeric_codes,
           };
}

sub import_cfg
{
    my ( $self, $fr ) = @_;
    my ( $src, $sql, $sth, $rows, $i, %col_names );

    my $logger = $self->logger;
    $src = DBI->connect( @{$fr}{qw(driver username password attrs)} );
    $sql = $fr->{query};
    $sth = $src->prepare($sql)
      or $logger->emergencyf( "Can't prepare '%s': %s", $sql, $src->errstr() );
    $sth->execute() or $logger->emergencyf( "Can't execute '%s': %s", $sql, $src->errstr() );
    $rows      = $sth->fetchall_arrayref();
    $i         = 0;
    %col_names = map +( $_ => $i++ ), @{ $sth->{NAME_uc} };
    $i         = 0;
    %col_names = ( %col_names, map +( $_ => $i++ ), @{ $sth->{NAME_lc} } );
    $i         = 0;
    %col_names = ( %col_names, map +( $_ => $i++ ), @{ $sth->{NAME} } );

    foreach my $cn ( keys %col_names )
    {
        $col_names{$cn} = '$row->[' . $col_names{$cn} . ']';
    }

    $logger->noticef( "Fetched %d rows for %s\n", scalar(@$rows), $self->service )
      if ( $logger->is_notice );

    return ( $rows, \%col_names );
}

sub transformer
{
    my ( $self, $tr, $min_valid_reg_code ) = @_;

    my $logger       = $self->logger;
    my $agg_code_str = "";
    $tr->{use}
      and $agg_code_str .=
      join( ";\n", map { $_ =~ s/^\s+//; $_ =~ s/\s+$//; "use $_" } split( ";", $tr->{used_mods} ) )
      . ";\n";
    $tr->{'required_mods'}
      and $agg_code_str .= join(
        ";\n",
        map {
            $_ =~ s/^\s+//;
            $_ =~ s/\s+$//;
            "require_module(\"$_\")"
          }
          split( ";", $tr->{'required_mods'} )
      )
      . ";\n";
    $tr->{skip} //= 0;
    $tr->{assert} //= 1;
    $agg_code_str = <<EOC;
$agg_code_str
return sub {
    my (\$logger,\$rows,\$cats,\$al2_c,\$al3_c,\$num_c) = \@_;
    my \@result = ( {} );
    my \$row_asserts_failed = 0;
    my \$rows_converted = 0;

    foreach my \$row (\@\$rows) {
	if( $tr->{skip} ) {
	    \$logger->debugf("skipping [%s]", join(", ", \@{\$row})) if(\$logger->is_debug);
	    next;
	}
	elsif( $tr->{assert} ) {
	    my \$bucket = int($tr->{bucket});
	    my \$time_slice = $tr->{time_slice};
	    my \$region = $tr->{region};
	    my \$category = $tr->{category};

	    \$bucket >= 19 and \$bucket = 19;
	    \$bucket < 0 and \$bucket = 19;

	    my \$region_id = \$al2_c->{\$region} // \$al3_c->{\$region} // \$num_c->{\$region};
	    defined \$region_id or die "No region id for \$region row " . (\$rows_converted + \$row_asserts_failed +1);
	    \$region_id < $min_valid_reg_code and next;

	    defined \$cats->{\$category} or die "Invalid category: " . \$category;
	    \$category = \$cats->{\$category};

	    my \$line_id = \$result[0]->{\$time_slice}->{\$region_id}->{\$category} //= scalar(\@result);
	    \$result[\$line_id][0] //= \$time_slice;
	    \$result[\$line_id][1] //= \$region_id;
	    \$result[\$line_id][2] //= \$category;
	    \$result[\$line_id][ \$bucket + 3 ] //= 0;
	    ++\$result[\$line_id][ \$bucket + 3 ];
	    ++\$rows_converted;
	}
	else {
	    \$logger->errorf("Row assertion failed for '%s'\\n", Data::Dumper->new( [\$row] )->Indent(0)->Sortkeys(1)->Quotekeys(0)->Terse(1)->Dump());
	    ++\$row_asserts_failed;
	}
    }

    my \$n_rows = scalar(\@\$rows);
    \$result[0]->{statistics} = {
	rows => \$n_rows,
	converted => \$rows_converted,
	failed => \$row_asserts_failed,
	skipped => \$n_rows - (\$rows_converted + \$row_asserts_failed)
    };

    return \\\@result;
};
EOC

    $logger->trace( Dumper($agg_code_str) ) if ( $logger->is_trace );
    my $agg_code = eval $agg_code_str;
    $@ and $logger->emergency($@);
    $logger->trace( B::Deparse->new()->coderef2text($agg_code) ) if ( $logger->is_trace );

    return $agg_code;
}

sub import_data
{
    my $self = shift;

    #$opt->input_file eq "-" and do
    #{
    #    open( $opt->{input_file},  ) or die "Can't dup STDIN: $!";
    #};

    # service/import dbd related ..
    my $logger  = $self->logger;
    my $schema  = $self->schema;
    my $srv_cfg = $self->service_config;

    my $agg_data;
    {
        my ( $rows, $col_names ) = $self->import_cfg( $srv_cfg->{fr} );
        App::Perf::Index->_replace_vars( $col_names, $srv_cfg->{tr} );
        my $transformer = $self->transformer( $srv_cfg->{tr}, $srv_cfg->{alpha2_codes}->{"-"} );
        $agg_data = &{$transformer}(
                                     $logger, $rows,
                                     $srv_cfg->{category_map},
                                     $srv_cfg->{alpha2_codes},
                                     $srv_cfg->{alpha3_codes},
                                     $srv_cfg->{numeric_codes}
                                   );
    }
    my $agg_struct = shift @$agg_data;

    $logger->noticef( "%d rows: %d converted -- %d skipped -- %d failed\n",
                      @{ $agg_struct->{statistics} }{qw(rows converted skipped failed)} )
      if ( $logger->is_notice );

    my $prepare_sub = sub {
        my ( $storage, $dbh, $sql ) = @_;
        my $sth = $dbh->prepare($sql) or die "Can't prepare '$sql': " . $dbh->errstr();
        $sth->{PrintError} = 0;
        $sth->{RaiseError} = 0;
        $sth;
    };
    my $sql_c =
      "SELECT BUCKET00, BUCKET01, BUCKET02, BUCKET03, BUCKET04, BUCKET05, BUCKET06, BUCKET07, BUCKET08, BUCKET09, BUCKET10, BUCKET11, BUCKET12, BUCKET13, BUCKET14, BUCKET15, BUCKET16, BUCKET17, BUCKET18, BUCKET19 FROM PERFORMANCE_DATA WHERE TIME_SLICE=? AND REGION_ID=? AND CATEGORY_ID=?";
    my $sth_c = $schema->storage->dbh_do( $prepare_sub, $sql_c );

    my $sql_u =
      "UPDATE PERFORMANCE_DATA SET BUCKET00=?, BUCKET01=?, BUCKET02=?, BUCKET03=?, BUCKET04=?, BUCKET05=?, BUCKET06=?, BUCKET07=?, BUCKET08=?, BUCKET09=?, BUCKET10=?, BUCKET11=?, BUCKET12=?, BUCKET13=?, BUCKET14=?, BUCKET15=?, BUCKET16=?, BUCKET17=?, BUCKET18=?, BUCKET19=? WHERE TIME_SLICE=? AND REGION_ID=? AND CATEGORY_ID=?";
    my $sth_u = $schema->storage->dbh_do( $prepare_sub, $sql_u );

    my $sql_i =
      "INSERT INTO PERFORMANCE_DATA (TIME_SLICE, REGION_ID, CATEGORY_ID, BUCKET00, BUCKET01, BUCKET02, BUCKET03, BUCKET04, BUCKET05, BUCKET06, BUCKET07, BUCKET08, BUCKET09, BUCKET10, BUCKET11, BUCKET12, BUCKET13, BUCKET14, BUCKET15, BUCKET16, BUCKET17, BUCKET18, BUCKET19) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";
    my $sth_i = $schema->storage->dbh_do( $prepare_sub, $sql_i );

    foreach my $result (@$agg_data)
    {
        scalar @$result < 23 and push @$result, (undef) x ( 23 - scalar(@$result) );
        my $n = eval { $sth_i->execute(@$result) };
        unless ( not $@ || !$n )
        {
            $schema->txn_begin();
            $sth_c->execute( @$result[ 0 .. 2 ] )
              or die "Can't execute '$sql_c': " . $sth_c->errstr();
            my $rows = $sth_c->fetchall_arrayref();
            scalar(@$rows) != 1
              and $logger->emergencyf(
                "Database error - TIME_SLICE: %d -- REGION: %d -- CATEGORY: %s must be unique but delivered %d rows",
                @$result[ 0 .. 2 ],
                scalar @$rows
              );
            my @row = @{ $rows->[0] };
            for my $i ( 0 .. $#row )
            {
                defined( $row[$i] ) or defined( $result->[ $i + 3 ] ) or next;
                $row[$i] //= 0;
                $result->[ $i + 3 ] //= 0;
                $row[$i] += $result->[ $i + 3 ];
            }
            $sth_u->execute( @row, @$result[ 0 .. 2 ] )
              or die "Can't execute '$sql_u': " . $sth_u->errstr();
            $schema->txn_commit();
        }
    }

    return 0;
}

1;
