/* common.h - prototypes for the common useful functions

   Copyright (C)
    2000  Russell Kroll <rkroll@exploits.org>
    2020-2025 Jim Klimov <jimklimov+nut@gmail.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifndef NUT_COMMON_H_SEEN
#define NUT_COMMON_H_SEEN 1

/* "config.h" is generated by autotools and lacks a header guard, so
 * we use an unambiguously named macro we know we must have, as one.
 * It must be the first header: be sure to know all about system config.
 */
#ifndef NUT_NETVERSION
# include "config.h"
#endif

#ifdef WIN32
# ifndef __POSIX_VISIBLE
/* for fcntl() and its flags in MSYS2 */
#  define __POSIX_VISIBLE 200809
# endif
#endif	/* WIN32 */

/* Need this on AIX when using xlc to get alloca */
#ifdef _AIX
#pragma alloca
#endif /* _AIX */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>	/* suseconds_t among other things */
#include <sys/stat.h>

#ifdef HAVE_SYS_SIGNAL_H
#include <sys/signal.h>
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#include <stdlib.h>

#ifdef HAVE_STRINGS_H
#include <strings.h>	/* for strncasecmp() and strcasecmp() */
#endif
#ifdef HAVE_STRING_H
#include <string.h>	/* for strdup() and many others */
#endif

#ifndef WIN32
# include <syslog.h>
#else	/* WIN32 */
# include <winsock2.h>
# include <windows.h>
# include <ws2tcpip.h>
#endif	/* WIN32 */

#ifdef NUT_WANT_INET_NTOP_XX
/* We currently have a few consumers who should define this macro before
 * including common.h, while we do not want to encumber preprocessing the
 * majority of codebase with these headers and our (thread-unsafe) methods.
 */
# ifndef WIN32
#  include <netdb.h>
#  include <sys/socket.h>
#  include <arpa/inet.h>
#  include <netinet/in.h>
# endif	/* WIN32 */
#endif	/* NUT_WANT_INET_NTOP_XX */

#include <unistd.h>	/* useconds_t */
#ifndef HAVE_USECONDS_T
# define useconds_t	unsigned long int
#endif
#ifndef HAVE_SUSECONDS_T
/* Note: WIN32 may have this defined as just "long" which should
 * hopefully be identical to the definition below, which we test
 * in our configure script. See also struct timeval fields for a
 * platform, if in doubt.
 */
# define suseconds_t	signed long int
#endif

#include <assert.h>

#include "timehead.h"
#include "attribute.h"
#include "proto.h"
#include "str.h"
#include "nut_stdint.h"

#if (defined HAVE_LIBREGEX && HAVE_LIBREGEX)
# include <regex.h>
#endif

#ifdef __cplusplus
/* *INDENT-OFF* */
extern "C" {
/* *INDENT-ON* */
#endif

#ifndef MAX
# define	MAX(p,q)	(((p) >= (q)) ? (p) : (q))
#endif
#ifndef MIN
# define	MIN(p,q)	(((p) <= (q)) ? (p) : (q))
#endif

/* POSIX requires these, and most but not all systems use same
 * magical numbers for the file descriptors... yep, not all do!
 */
#ifndef STDIN_FILENO
# define STDIN_FILENO  0	/* standard input file descriptor */
#endif
#ifndef STDOUT_FILENO
# define STDOUT_FILENO 1	/* standard output file descriptor */
#endif
#ifndef STDERR_FILENO
# define STDERR_FILENO 2	/* standard error file descriptor */
#endif

/* porting stuff for WIN32, used by serial and SHUT codebases */
#ifndef WIN32
/* Just match three macro groups defined for WIN32 */

/* Type of what file open() and close() use,
 * including pipes for driver-upsd communications: */
# define TYPE_FD int
# define ERROR_FD (-1)
# define VALID_FD(a) ((a)>=0)

/* Type of what NUT serial/SHUT methods juggle: */
# define TYPE_FD_SER TYPE_FD
# define ERROR_FD_SER ERROR_FD
# define VALID_FD_SER(a) VALID_FD(a)

/* Type of what socket() returns, mostly for networked code: */
# define TYPE_FD_SOCK TYPE_FD
# define ERROR_FD_SOCK ERROR_FD
# define VALID_FD_SOCK(a) VALID_FD(a)

#else	/* WIN32 */

/* Separate definitions of TYPE_FD, ERROR_FD, VALID_FD() macros
 * for usual file descriptors vs. types needed for serial port
 * work or for networking sockets.
 */
# define TYPE_FD HANDLE
# define ERROR_FD (INVALID_HANDLE_VALUE)
# define VALID_FD(a) ((a)!=INVALID_HANDLE_VALUE)

# ifndef INVALID_SOCKET
#  define INVALID_SOCKET ((SOCKET)(-1))
# endif

/* Bitness-dependent "pointer-sized unsigned integer" (usually 32 or 64 bits) */
# define TYPE_FD_SOCK SOCKET
# define ERROR_FD_SOCK INVALID_SOCKET
/* Valid range for SOCKET is 0..(INVALID_SOCKET-1) and there is no special
 * check for "-1" (may be or not be coincidental by casting and/or definition
 * in existing headers) nor generally negative values, as in Unix socket API.
 */
# define VALID_FD_SOCK(a) ((a)!=INVALID_SOCKET)

typedef struct serial_handler_s {
	HANDLE handle;
	OVERLAPPED io_status;
	int     overlapped_armed;

	unsigned int vmin_;
	unsigned int vtime_;
	unsigned int r_binary;
	unsigned int w_binary;
} serial_handler_t;

# define TYPE_FD_SER serial_handler_t *
# define ERROR_FD_SER (NULL)
# define VALID_FD_SER(a) ((a)!=NULL)

/* difftime returns erroneous value so we use this macro */
# undef difftime
# define difftime(t1,t0) (double)(t1 - t0)
#endif	/* WIN32 */

/* Two uppercase letters are more readable than one exclamation */
#define INVALID_FD_SER(a) (!VALID_FD_SER(a))
#define INVALID_FD_SOCK(a) (!VALID_FD_SOCK(a))
#define INVALID_FD(a) (!VALID_FD(a))

#define SIZEOF_ARRAY(a) (sizeof(a) / sizeof(a[0]))

extern const char *UPS_VERSION;

/* Use in code to notify the developers and quiesce the compiler that
 * (for this codepath) the argument or variable is unused intentionally.
 * void f(int x) {
 *   NUT_UNUSED_VARIABLE(x);
 *   ...
 * }
 *
 * Note that solutions which mark up function arguments or employ this or
 * that __attribute__ proved not portable enough for wherever NUT builds.
 */
#define NUT_UNUSED_VARIABLE(x) (void)(x)

/** @brief Default timeout (in seconds) for network operations, as used by `upsclient` and `nut-scanner`. */
#define DEFAULT_NETWORK_TIMEOUT		5

/** @brief Default timeout (in seconds) for retrieving the result of a `TRACKING`-enabled operation (e.g. `INSTCMD`, `SET VAR`). */
#define DEFAULT_TRACKING_TIMEOUT	10

/* Returns a pointer to static internal char[] buffer with current value
 * of NUT_VERSION_MACRO (aka char* UPS_VERSION) and its layman description
 * (e.g. a "release" or "development iteration after" a certain semantically
 * versioned release). Returns UPS_VERSION if failed to construct a better
 * description. Either way, should not be free()'d by caller and does not
 * have an end-of-line char of its own. */
const char *describe_NUT_VERSION_once(void);

/* Return a buffer with a (possibly multiline) string that can be printed
 * as part of program help/usage message. Caller should not free() the buffer.
 * Optional "progconf" allows to suggest config file(s) to study as well.
 * NOTE: the string in buffer starts with text and ends with one EOL char.
 */
const char *suggest_doc_links(const char *progname, const char *progconf);

/* For drivers that failed to start because they could not hold on to a
 * device, on systems where the nut-driver-enumerator could produce units
 * that conflict with a manually-launched driver program, suggest that
 * this may be the case. Has some work on systems where NDE can be used
 * (currently where SMF or SystemD were considered during build), no-op
 * on others.
 * We define this in one spot, to change conditions or wording easily.
 */
void suggest_NDE_conflict(void);

/* Based on NUT_QUIET_INIT_BANNER envvar (present and empty or "true")
 * hide the NUT tool name+version banners; show them by default */
int banner_is_disabled(void);

/* Some NUT programs have historically printed their banner at start-up
 * always, and so did not print one in help()/usage() or handling `-V`
 * like others did. Now that we have NUT_QUIET_INIT_BANNER, we need a
 * way to print that banner (regardless of the flag in some cases).
 * The "even_if_disabled" should be 0 for initial banner of those
 * programs (so the envvar would hide it), 1 -V case and 2 in -h case
 * (for a blank line after). As before, the banner is printed to stdout.
 * Returns the result of printf() involved. Remembers to not print again
 * if the earlier printf() was successful.
 */
int print_banner_once(const char *prog, int even_if_disabled);

/* Normally we can (attempt to) use the syslog or Event Log (WIN32),
 * but environment variable NUT_DEBUG_SYSLOG allows to bypass it, and
 * perhaps keep daemons logging to stderr (e.g. in NUT Integration Test
 * suite to not pollute the OS logs, or in systemd where stderr and
 * syslog both go into the same journal). Returns:
 *  0  Not disabled (NUT_DEBUG_SYSLOG not set to a value below; unset or "default"
 *     values are handled quietly, but others emit a warning)
 *  1  Disabled and background() keeps stderr attached (NUT_DEBUG_SYSLOG="stderr")
 *  2  Disabled and background() detaches stderr as usual (NUT_DEBUG_SYSLOG="none")
 */
int syslog_is_disabled(void);

/* get the syslog ready for us */
void open_syslog(const char *progname);

/* close ttys and become a daemon */
void background(void);

/* allow tagging the (forked) process in logs to ease debugging */
const char *getproctag(void);
/* save a copy of tag, or call with NULL to clean and free the internal buffer;
 * if using this feature in a particular NUT program at all - it automatically
 * registers with atexit() to do such clean-up in exit handling.
 *
 * WARNING: first call to this method also caches the getprocname(getpid())
 * so if you want to see debug logs from that - only call this after setting
 * the nut_debug_level (by parsing CLI arguments and/or NUT_DEBUG_LEVEL envvar).
 */
void setproctag(const char *tag);

/* do this here to keep pwd/grp stuff out of the main files */
struct passwd *get_user_pwent(const char *name);

/* change to the user defined in the struct */
void become_user(struct passwd *pw);

/* drop down into a directory and throw away pointers to the old path */
void chroot_start(const char *path);

/* Try to identify process (program) name for the given PID,
 * return NULL if we can not for any reason (does not run,
 * no rights, do not know how to get it on current OS, etc.)
 * If the returned value is not NULL, caller should free() it.
 * Some implementation pieces borrowed from
 * https://man7.org/linux/man-pages/man2/readlink.2.html and
 * https://github.com/openbsd/src/blob/master/bin/ps/ps.c
 * NOTE: Very much platform-dependent! */
char * getprocname(pid_t pid);

/* Determine the base name of specified progname (may be full path)
 * and the location of the last "." dot character in it for extension
 * (caller's len and dot populated only if pointers are not NULL).
 */
size_t parseprogbasename(char *buf, size_t buflen, const char *progname, size_t *pprogbasenamelen, size_t *pprogbasenamedot);

/* If we can determine the binary path name of the specified "pid",
 * check if it matches the assumed name of the current program.
 * Returns:
 *	-3	Skipped because NUT_IGNORE_CHECKPROCNAME is set
 *	-2	Could not parse a program name (ok to proceed,
 *		risky - but matches legacy behavior)
 *	-1	Could not identify a program name (ok to proceed,
 *		risky - but matches legacy behavior)
 *	0	Process name identified, does not seem to match
 *	1+	Process name identified, and seems to match with
 *		varying precision
 * Generally speaking, if (checkprocname(...)) then ok to proceed.
 * Singular for programs with a single expected executable name,
 * plural for programs with expected aliases (e.g. "old"/new" migrations).
 */
int checkprocname(pid_t pid, const char *progname);
int checkprocnames(pid_t pid, const char **prognames);
/* compareprocname*() methods do the bulk of work for checkprocname*()
 * and return same values. The "pid" argument is used for logging.
 * Generally speaking, if (compareprocname(...)) then ok to proceed.
 * Singular for programs with a single expected executable name,
 * plural for programs with expected aliases (e.g. "old"/new" migrations).
 */
int compareprocname(pid_t pid, const char *procname, const char *progname);
int compareprocnames(pid_t pid, const char *procname, const char **prognames);
/* Helper for the above methods and some others. If it returns true (1),
 * work about PID-name comparison should be quickly skipped.
 */
int checkprocname_ignored(const char *caller);

/* write a pid file - <name> is a full pathname *or* just the program name */
void writepid(const char *name);

/* parses string buffer into a pid_t if it passes
 * a few sanity checks; returns -1 on error */
pid_t parsepid(const char *buf);

/* send a signal to another running NUT process */
#ifndef WIN32
int sendsignal(const char *progname, int sig, int check_current_progname);
#else	/* WIN32 */
int sendsignal(const char *progname, const char * sig, int check_current_progname);
#endif	/* WIN32 */

int snprintfcat(char *dst, size_t size, const char *fmt, ...)
	__attribute__ ((__format__ (__printf__, 3, 4)));

/*****************************************************************************
 * String methods for space-separated token lists, used originally in dstate *
 * NOTE: These methods are also exposed by external API (via libupsclient.h) *
 * with the `upscli_` prefix, to ease third-party C NUT clients' parsing of  *
 * `ups.status` et al.                                                       *
 *****************************************************************************/

/* Return non-zero if "string" contains "token" (case-sensitive),
 * either surrounded by space character(s) or start/end of "string",
 * or 0 if that token is not there, or if either string is NULL or empty.
 */
int	str_contains_token(const char *string, const char *token);

/* Add "token" to end of string "tgt", if it is not yet there
 * (prefix it with a space character if "tgt" is not empty).
 * Return 0 if already there, 1 if token was added successfully,
 * -1 if we needed to add it but it did not fit under the tgtsize limit,
 * -2 if either string was NULL or "token" was empty.
 * NOTE: If token contains space(s) inside, recurse to treat it
 * as several tokens to add independently.
 * Optionally calls "callback_always" (if not NULL) after checking
 * for spaces (and maybe recursing) and before checking if the token
 * is already there, and/or "callback_unique" (if not NULL) after
 * checking for uniqueness and going to add a newly seen token.
 * If such callback returns 0, abort the addition of token and return -3.
 */
int	str_add_unique_token(char *tgt, size_t tgtsize, const char *token,
			    int (*callback_always)(char *, size_t, const char *),
			    int (*callback_unique)(char *, size_t, const char *)
);

/* Report maximum platform value for the pid_t */
pid_t get_max_pid_t(void);

/* send sig to pid after some sanity checks, returns
 * -1 for error, or zero for a successfully sent signal */
int sendsignalpid(pid_t pid, int sig, const char *progname, int check_current_progname);
int sendsignalpidaliases(pid_t pid, int sig, const char **prognames, int check_current_progname);

/* open <pidfn> and get the pid
 * returns zero or more for successfully retrieved value,
 * negative for errors:
 * -3   PID file not found
 * -2   PID file not parsable
 */
pid_t parsepidfile(const char *pidfn);

#ifndef WIN32
/* use parsepidfile() to get the pid, then send it <sig>
 * returns zero for successfully sent signal,
 * negative for errors:
 * -3   PID file not found
 * -2   PID file not parsable
 * -1   Error sending signal
 *
 * if executable process with that pid has suitable progname
 * (specified or that of the current process, depending on args:
 * most daemons request to check_current_progname for their other
 * process instancees, but upsdrvctl which manages differently
 * named driver programs does not request it)
 */
int sendsignalfn(const char *pidfn, int sig, const char *progname, int check_current_progname);
int sendsignalfnaliases(const char *pidfn, int sig, const char **prognames, int check_current_progname);
#else	/* WIN32 */
/* No progname here - communications via named pipe */
int sendsignalfn(const char *pidfn, const char * sig, const char *progname_ignored, int check_current_progname_ignored);
int sendsignalfnaliases(const char *pidfn, const char * sig, const char **prognames_ignored, int check_current_progname_ignored);
#endif	/* WIN32 */

/* return a pointer to character inside the file that starts a basename
 * caller should strdup() a copy to retain beyond the lifetime of "file" */
const char *xbasename(const char *file);

/* enable writing upslog_with_errno() and upslogx() type messages to
   the syslog */
void syslogbit_set(void);

/* Return the default path for the directory containing configuration files */
const char * confpath(void);

/* Return the default path for the directory containing state files */
const char * dflt_statepath(void);

/* Return the alternate path for pid files (non-root daemons) */
const char * altpidpath(void);

/* Return the main path for pid files (root daemons like upsmon) */
const char * rootpidpath(void);

/* Die with a standard message if socket filename is too long */
void check_unix_socket_filename(const char *fn);

#ifdef NUT_WANT_INET_NTOP_XX
/* Helpers to convert one IP address to string from different structure types
 * Return pointer to internal buffer (in NOT THREAD SAFE! methods named as such)
 * or caller-provided buffer or an allocated buffer that caller must free (in
 * the "x" methods), or NULL and errno upon errors */
const char *inet_ntopSS(struct sockaddr_storage *s, char *addrstr, size_t addrstrsz);
const char *inet_ntopSS_thread_unsafe(struct sockaddr_storage *s);
const char *xinet_ntopSS(struct sockaddr_storage *s);

const char *inet_ntopAI(struct addrinfo *ai, char *addrstr, size_t addrstrsz);
const char *inet_ntopAI_thread_unsafe(struct addrinfo *ai);
const char *xinet_ntopAI(struct addrinfo *ai);
#endif	/* NUT_WANT_INET_NTOP_XX */

/* Provide integration for systemd inhibitor interface (where available,
 * dummy code otherwise) implementing the pseudo-code example from
 * https://systemd.io/INHIBITOR_LOCKS/
 * TODO: Potentially extensible to other frameworks with similar concepts?..
 */
TYPE_FD Inhibit(const char *arg_what, const char *arg_who, const char *arg_why, const char *arg_mode);
/* Let the service management framework proceed with its sleep mode */
void Uninhibit(TYPE_FD *fd_ptr);
/* Check once if the system plans to sleep or is waking up:
 *  -1	Same reply as before, whatever it was
 *   0	(false) Just after the sleep, at least as a bus signal
 *   1	(true) Before the sleep - we must process and un-block it
 */
int isPreparingForSleep(void);

/* A couple of methods to reflect built-in (absent) or run-time (it depends)
 * support for monitoring that the OS goes to sleep and wakes up, and if we
 * can "inhibit" that going to sleep in order to do some house-keeping first.
 * -1 = do not know yet
 *  0 = not supported, do not bother asking in daemon loops
 *  1 = seems supported
 */
int isInhibitSupported(void);
int isPreparingForSleepSupported(void);

/* Send (daemon) state-change notifications to an
 * external service management framework such as systemd.
 * State types below are initially loosely modeled after
 *   https://www.freedesktop.org/software/systemd/man/sd_notify.html
 */
typedef enum eupsnotify_state {
	NOTIFY_STATE_READY = 1,
	NOTIFY_STATE_READY_WITH_PID,
	NOTIFY_STATE_RELOADING,
	NOTIFY_STATE_STOPPING,
	NOTIFY_STATE_STATUS,	/* Send a text message per "fmt" below */
	NOTIFY_STATE_WATCHDOG,	/* Ping the framework that we are still alive */
	NOTIFY_STATE_EXTEND_TIMEOUT	/* Ping the framework that we are still alive when starting/stopping */
} upsnotify_state_t;
const char *str_upsnotify_state(upsnotify_state_t state);
/* Note: here fmt may be null, then the STATUS message would not be sent/added */
int upsnotify(upsnotify_state_t state, const char *fmt, ...)
	__attribute__ ((__format__ (__printf__, 2, 3)));

/* Exposed to code consumers (NUT daemons) for NOTIFY_STATE_EXTEND_TIMEOUT
 * By default upsnotify_extend_timeout_usec == 0 so the
 * upsnotify() method would fall back to current WATCHDOG_USEC
 * if available, or to upsnotify_extend_timeout_usec_default.
 * NOTE: It seems that internally in systemd, UINT64_MAX or
 *  ((uint64_t)-1) means "infinity". Internally systemd uses
 *  uint64_t as their usec_t (at least currently) but this
 *  does not seem to be a public API/contract. De-facto the
 *  value did not have any effect; however INT64_MAX did work
 *  (presumably as almost 300K years, did not check that long).
 *  More at https://github.com/systemd/systemd/issues/39535
 * Whatever value gets applied, it should exceed the relevant
 * loop cycle duration at that point in daemon life time.
 */
#define UPSNOTIFY_EXTEND_TIMEOUT_USEC_INFINITY	((uint64_t)INT64_MAX)
extern uint64_t upsnotify_extend_timeout_usec_default, upsnotify_extend_timeout_usec;

/* upslog*() messages are sent to syslog always;
 * their life after that is out of NUT's control */
void upslog_with_errno(int priority, const char *fmt, ...)
	__attribute__ ((__format__ (__printf__, 2, 3)));
void upslogx(int priority, const char *fmt, ...)
	__attribute__ ((__format__ (__printf__, 2, 3)));

/* upsdebug*() messages are only logged if debugging
 * level is high enough. To speed up a bit (minimize
 * passing of ultimately ignored data trough the stack)
 * these are "hidden" implementations wrapped into
 * macros for earlier routine names spread around the
 * codebase, they would check debug level first and
 * only if logging should happen - call the routine
 * and pass around pointers and other data.
 */
void s_upsdebug_with_errno(int level, const char *fmt, ...)
	__attribute__ ((__format__ (__printf__, 2, 3)));
void s_upsdebugx(int level, const char *fmt, ...)
	__attribute__ ((__format__ (__printf__, 2, 3)));
void s_upsdebug_hex(int level, const char *msg, const void *buf, size_t len);
void s_upsdebug_ascii(int level, const char *msg, const void *buf, size_t len);
/* These macros should help avoid run-time overheads
 * passing data for messages nobody would ever see.
 *
 * Also NOTE: the "level" may be specified by callers in various ways,
 * e.g. as a "X ? Y : Z" style expression; to catch those expansions
 * transparently we hide them into parentheses as "(label)".
 *
 * For stricter C99 compatibility, all parameters specified to a macro
 * must be populated by caller (so we do not handle "fmt, args..." where
 * the args part may be skipped by caller because fmt is a fixed string).
 * Note it is then up to the caller (and compiler checks) that at least
 * one argument is provided, the format string (maybe fixed) -- as would
 * be required by the actual s_upsdebugx*() method after macro evaluation.
 */
#define upsdebug_with_errno(level, ...) \
	do { if (nut_debug_level >= (level)) { s_upsdebug_with_errno((level), __VA_ARGS__); } } while(0)
#define upsdebugx(level, ...) \
	do { if (nut_debug_level >= (level)) { s_upsdebugx((level), __VA_ARGS__); } } while(0)
#define upsdebug_hex(level, msg, buf, len) \
	do { if (nut_debug_level >= (level)) { s_upsdebug_hex((level), msg, buf, len); } } while(0)
#define upsdebug_ascii(level, msg, buf, len) \
	do { if (nut_debug_level >= (level)) { s_upsdebug_ascii((level), msg, buf, len); } } while(0)

void fatal_with_errno(int status, const char *fmt, ...)
	__attribute__ ((__format__ (__printf__, 2, 3))) __attribute__((noreturn));
void fatalx(int status, const char *fmt, ...)
	__attribute__ ((__format__ (__printf__, 2, 3))) __attribute__((noreturn));

/* Report CONFIG_FLAGS used for this build of NUT similarly to how
 * upsdebugx(1, ...) would do it, but not limiting the string length
 */
void nut_report_config_flags(void);

/* Report search paths used by ltdl-augmented code to discover and
 * load shared binary object files at run-time (nut-scanner, DMF...) */
void upsdebugx_report_search_paths(int level, int report_search_paths_builtin);
void nut_prepare_search_paths(void);

/* Internal toggle for some NUT programs that deal with signal sending.
 * For a detailed rationale comment see common.c */
extern int nut_sendsignal_debug_level;
#define NUT_SENDSIGNAL_DEBUG_LEVEL_DEFAULT 6
#define NUT_SENDSIGNAL_DEBUG_LEVEL_FOPEN_PIDFILE 5
#define NUT_SENDSIGNAL_DEBUG_LEVEL_KILL_SIG0PING 4

extern int nut_debug_level;
extern int nut_log_level;

void *xmalloc(size_t size);
void *xcalloc(size_t number, size_t size);
void *xrealloc(void *ptr, size_t size);
char *xstrdup(const char *string);

int vsnprintfcat(char *dst, size_t size, const char *fmt, va_list ap);
int snprintfcat(char *dst, size_t size, const char *fmt, ...)
	__attribute__ ((__format__ (__printf__, 3, 4)));

/* Define a missing va_copy using __va_copy, if available: */
#ifndef HAVE_VA_COPY
# ifdef HAVE___VA_COPY
#  define va_copy(dest, src) __va_copy(dest, src)
# endif
#endif

/* Mitigate the inherent insecurity of dynamically constructed formatting
 * strings vs. a fixed vararg list with its amounts and types of variables
 * printed by this or that method and pre-compiled in the program.
 * Verbosity is passed to upsdebugx(); a negative value should keep it quiet
 * (e.g. when code deliberately checks for suitable formatting constraints).
 * Consumers like the *_dynamic() methods here and in dstate typically use
 * "1" to make errors in code visible with any effort to troubleshoot them.
 */
/* Verbosity built into the methods which call the *_formatting_*() and
 * pass this value as the verbosity variable argument. It is anticipated
 * that some custom builds can define it to e.g. 0 to see discrepancies
 * at run-time without enabling any debug verbosity: */
#ifndef NUT_DYNAMICFORMATTING_DEBUG_LEVEL
# define NUT_DYNAMICFORMATTING_DEBUG_LEVEL 1
#endif
/* Verbosity built into consumers that deliberately check the formatting
 * strings for this or that outcome and do not want noise in the log: */
#define NUT_DYNAMICFORMATTING_DEBUG_LEVEL_SILENT -1
char *minimize_formatting_string(char *buf, size_t buflen, const char *fmt, int verbosity);
char *minimize_formatting_string_staticbuf(const char *fmt, int verbosity);
int validate_formatting_string(const char *fmt_dynamic, const char *fmt_reference, int verbosity);

int vsnprintfcat_dynamic(char *dst, size_t size, const char *fmt_dynamic, const char *fmt_reference, va_list va);
int snprintfcat_dynamic(char *dst, size_t size, const char *fmt_dynamic, const char *fmt_reference, ...)
	__attribute__ ((__format__ (__printf__, 4, 5)));

int vsnprintf_dynamic(char *dst, size_t size, const char *fmt_dynamic, const char *fmt_reference, va_list va);
int snprintf_dynamic(char *dst, size_t size, const char *fmt_dynamic, const char *fmt_reference, ...)
	__attribute__ ((__format__ (__printf__, 4, 5)));

/* Practical helper with a static buffer which we can use for setting
 * values as a "%s" string e.g. in calls to dstate_setinfo(), etc.
 * Sets buffer to empty string in case of errors.
 */
char * mkstr_dynamic(const char *fmt_dynamic, const char *fmt_reference, ...)
	__attribute__ ((__format__ (__printf__, 2, 3)));

/**** REGEX helper methods ****/

/* helper function: version of strcmp that tolerates NULL
 * pointers. NULL is considered to come before all other strings
 * alphabetically.
 */
int strcmp_null(const char *s1, const char *s2);

#if (defined HAVE_LIBREGEX && HAVE_LIBREGEX)
/* Helper function for compiling a regular expression. On success,
 * store the compiled regular expression (or NULL) in *compiled, and
 * return 0. On error with errno set, return -1. If the supplied
 * regular expression is unparseable, return -2 (an error message can
 * then be retrieved with regerror(3)). Note that *compiled will be an
 * allocated value, and must be freed with regfree(), then free(), see
 * regex(3). As a special case, if regex==NULL, then set
 * *compiled=NULL (regular expression NULL is intended to match
 * anything).
 */
int compile_regex(regex_t **compiled, const char *regex, const int cflags);

/* Helper function for regular expression matching. Check if the
 * entire string str (minus any initial and trailing whitespace)
 * matches the compiled regular expression preg. Return 1 if it
 * matches, 0 if not. Return -1 on error with errno set. Special
 * cases: if preg==NULL, it matches everything (no contraint).  If
 * str==NULL, then it is treated as "".
 */
int match_regex(const regex_t *preg, const char *str);

/* Helper function, similar to match_regex, but the argument being
 * matched is a (hexadecimal) number, rather than a string. It is
 * converted to a 4-digit hexadecimal string. */
int match_regex_hex(const regex_t *preg, const int n);
#endif	/* HAVE_LIBREGEX */

/* Note: different method signatures instead of TYPE_FD_SER due to "const" */
#ifndef WIN32
ssize_t select_read(const int fd, void *buf, const size_t buflen, const time_t d_sec, const suseconds_t d_usec);
ssize_t select_write(const int fd, const void *buf, const size_t buflen, const time_t d_sec, const suseconds_t d_usec);
#else	/* WIN32 */
ssize_t select_read(serial_handler_t *fd, void *buf, const size_t buflen, const time_t d_sec, const suseconds_t d_usec);
/* Note: currently not implemented de-facto for Win32 */
ssize_t select_write(serial_handler_t * fd, const void *buf, const size_t buflen, const time_t d_sec, const suseconds_t d_usec);
#endif	/* WIN32 */

char * get_libname(const char* base_libname);

/* Buffer sizes used for various functions */
#define SMALLBUF	512
#define LARGEBUF	1024

/** @brief (Minimum) Size that a string must have to hold a UUID4 (i.e. UUID4 length + the terminating null character). */
#define UUID4_LEN	37

#define NUT_PATH_MAX	SMALLBUF
#if (defined(PATH_MAX)) && PATH_MAX > NUT_PATH_MAX
# undef NUT_PATH_MAX
# define NUT_PATH_MAX	PATH_MAX
#endif
#if (defined(MAX_PATH)) && MAX_PATH > NUT_PATH_MAX
/* PATH_MAX is the POSIX equivalent for Microsoft's MAX_PATH */
# undef NUT_PATH_MAX
# define NUT_PATH_MAX	MAX_PATH
#endif
#if (defined(UNIX_PATH_MAX)) && UNIX_PATH_MAX > NUT_PATH_MAX
# undef NUT_PATH_MAX
# define NUT_PATH_MAX	UNIX_PATH_MAX
#endif
#if (defined(MAXPATHLEN)) && MAXPATHLEN > NUT_PATH_MAX
# undef NUT_PATH_MAX
# define NUT_PATH_MAX	MAXPATHLEN
#endif

/* Provide declarations for getopt() global variables */

#ifdef NEED_GETOPT_H
#include <getopt.h>
#else
#ifdef NEED_GETOPT_DECLS
extern char *optarg;
extern int optind;
#endif /* NEED_GETOPT_DECLS */
#endif /* HAVE_GETOPT_H */

/* logging flags: bitmask! */

#define UPSLOG_STDERR		0x0001
#define UPSLOG_SYSLOG		0x0002
#define UPSLOG_STDERR_ON_FATAL	0x0004
#define UPSLOG_SYSLOG_ON_FATAL	0x0008

#ifndef HAVE_SETEUID
#	define seteuid(x) setresuid(-1,x,-1)    /* Works for HP-UX 10.20 */
#	define setegid(x) setresgid(-1,x,-1)    /* Works for HP-UX 10.20 */
#endif

/* Several NUT programs define their set_exit_flag(int) methods
 * which accept a signal code or similar parameter. Commonly they
 * also accept a few negative values summarized below, to just
 * exit (typically after completing a processing loop) with one
 * of C standard exit codes.
 */
#define EF_EXIT_FAILURE	-1	/* eventually exit(EXIT_FAILURE); */
#define EF_EXIT_SUCCESS	-2	/* eventually exit(EXIT_SUCCESS); */

#ifdef WIN32
/* FIXME : this might not be the optimal mapping between syslog and ReportEvent*/
#define LOG_ERR 	EVENTLOG_ERROR_TYPE
#define LOG_INFO 	EVENTLOG_INFORMATION_TYPE
#define LOG_DEBUG	EVENTLOG_WARNING_TYPE
#define LOG_NOTICE	EVENTLOG_INFORMATION_TYPE
#define LOG_ALERT	EVENTLOG_ERROR_TYPE
#define LOG_WARNING	EVENTLOG_WARNING_TYPE
#define LOG_CRIT	EVENTLOG_ERROR_TYPE
#define LOG_EMERG	EVENTLOG_ERROR_TYPE

#define closelog()

#define SVCNAME TEXT("Network UPS Tools")
#define EVENTLOG_PIPE_NAME TEXT("nut")
#define UPSMON_PIPE_NAME TEXT("upsmon")
#define UPSD_PIPE_NAME TEXT("upsd")

char * getfullpath(char * relative_path);

/* e.g. getfullpath2(CONFPATH, PATH_ETC) */
char * getfullpath2(char * cfg_path, char * fallback_path);

#define PATH_ETC	"\\..\\etc"
#define PATH_VAR_RUN "\\..\\var\\run"
#define PATH_SHARE "\\..\\share"
#define PATH_BIN "\\..\\bin"
#define PATH_SBIN "\\..\\sbin"
#define PATH_LIB "\\..\\lib"
#endif	/* WIN32*/

/* Return a difference of two timevals as a floating-point number */
double difftimeval(struct timeval x, struct timeval y);
#if defined(HAVE_CLOCK_GETTIME) && defined(HAVE_CLOCK_MONOTONIC) && HAVE_CLOCK_GETTIME && HAVE_CLOCK_MONOTONIC
double difftimespec(struct timespec x, struct timespec y);
#endif

#ifndef HAVE_USLEEP
/* int __cdecl usleep(unsigned int useconds); */
/* Note: if we'd need to define an useconds_t for obscure systems,
 * it should be an int capable of string 0..1000000 value range,
 * so probably unsigned long int */
int __cdecl usleep(useconds_t useconds);
#endif /* HAVE_USLEEP */

#ifndef HAVE_STRNLEN
size_t strnlen(const char *s, size_t maxlen);
#endif

/* Not all platforms support the flag; this method abstracts
 * its use (or not) to simplify calls in the actual codebase */
/* TODO NUT_WIN32_INCOMPLETE? : Extend for TYPE_FD and WIN32 eventually? */
void set_close_on_exec(int fd);

#ifdef __cplusplus
/* *INDENT-OFF* */
}
/* *INDENT-ON* */
#endif

#endif /* NUT_COMMON_H_SEEN */
