/*
 * Professional Galactic Empires Client
 * Copyright 1996-1999 David A. Holland
 * Version 0.1, 6/11/99
 * All rights reserved.
 *
 * See the included file "COPYING" for license information.
 */

/*
 * pge client, take 2.
 */

#include <curses.h>
#include <unistd.h>
#include <ctype.h>
#include <stdarg.h>
#include <string.h>

#include "ptrarray.h"
#include "pgeclient.h"
#include "version.h"
#include "util.h"

typedef long chartype;

static ptrarray<chartype> mainlog;
static int mainscrollup = 0;
static int playable = 0;
static int queuepos = -1;
static int playing = 0;
static int attrib = 0;
static chartype tmpbuf[4096];
static size_t tmpbufpos = 0;

void display_update() {
    int cx=-1, cy=-1;
    int nlines = mainlog.num();
    if (playing) nlines++;   /* count tmpbuf */
    int xtopline = nlines - (LINES-2);
    if (xtopline<0) xtopline = 0;
    int topline = xtopline - mainscrollup;
    for (int i=0; i<LINES-2; i++) {
	move(i,0);
	int line = i+topline;
	if (line>=0 && line<mainlog.num()) {
	    /* blah */
	    chartype *x = mainlog[line];
	    for (int j=0; x[j]; j++) addch(x[j]);
	    //if (playing && tmpbufpos==0 && line==mainlog.num()-1) {
	    //	getyx(stdscr, cy, cx);
	    //}
	}
	else if (line==mainlog.num()) {
	    for (int j=0; tmpbuf[j]; j++) addch(tmpbuf[j]);
	    getyx(stdscr, cy, cx);
	}
	clrtoeol();
    }

    move(LINES-2, 0);
    attron(A_REVERSE);
    char tmp[128];
    if (playing) {
	snprintf(tmp, sizeof(tmp), " Playing.");
    }
    else if (playable) {
	snprintf(tmp, sizeof(tmp), " Ready to play.");
    }
    else if (queuepos>1) {
	snprintf(tmp, sizeof(tmp), " %d people ahead of you.", queuepos);
    }
    else if (queuepos==1) {
	snprintf(tmp, sizeof(tmp), " 1 person ahead of you.");
    }
    else {
	snprintf(tmp, sizeof(tmp), " Not in wait queue.");
    }
    addstr(tmp);
    for (int i=strlen(tmp); i<COLS; i++) addch(' ');
    attroff(A_REVERSE);

    move(LINES-1, 0);
    if (!playing) {
	if (playable) addstr("[P]lay, or view ");
	else addstr("View ");
	addstr("[M]OTD, [S]cores, [N]ews, or [H]istory? [Q to quit] ");
	clrtoeol();
    }
    else {
	addstr("Ctrl-P/N to scroll");
	clrtoeol();
	if (cx>=0 && cy>=0) move(cy, cx);
    }

    refresh();
}

void say(const char *format, ...) {
    char buf[4096];
    va_list ap;
    va_start(ap, format);
    vsnprintf(buf, sizeof(buf), format, ap);
    va_end(ap);

    chartype *x = new chartype[strlen(buf)+1];
    for (int i=0; (x[i] = buf[i]) != 0; i++);
    mainlog.add(x);
}

void display_setplayable(int state) {
    playable = state;
}

void display_setqueuepos(int pos) {
    queuepos = pos;
}

void display_setplaying(int state) {
    playing = state;
}

void display_text() {
    attrib = 0;
}

void display_bold() {
    attrib = A_BOLD;
}

void display_reverse() {
    attrib = A_REVERSE;
}

void display_char(int ch) {
    if (ch=='\r' || ch=='\n') {
	mainlog.add(xstrdup(tmpbuf));
	tmpbufpos = 0;
	tmpbuf[tmpbufpos] = 0;
    }
    else if (ch=='\a') {
	beep();
    }
    else if (tmpbufpos < sizeof(tmpbuf)-1) {
	tmpbuf[tmpbufpos++] = (ch | attrib);
	tmpbuf[tmpbufpos] = 0;
    }
}

static int post_key(int ch) {
    if (ch==12) {
	clear();
	display_update();
    }
    else if (ch==14) {
	mainscrollup -= 1;
	if (mainscrollup<0) mainscrollup = 0;
    }
    else if (ch==16) {
	mainscrollup += 1;
	if (mainscrollup>mainlog.num()-1) mainscrollup = mainlog.num()-1;
    }
    else if (ch==22) {
	mainscrollup -= (LINES-3);
	if (mainscrollup<0) mainscrollup = 0;
    }
    else if (ch==('v'|128)) {
	mainscrollup += (LINES-3);
	if (mainscrollup>mainlog.num()-1) mainscrollup = mainlog.num()-1;
    }
    else if (playing) {
	if ((ch<=127 && ch>=32) || ch=='\b' || ch=='\t') tell("%c", ch);
	else if (ch=='\r' || ch=='\n') tell("\r");
    }
    else {
	ch = toupper(ch);
	if (playable && ch=='P') tell("PLAY\r\n");
	else if (ch=='M') tell("INFO MOTD\r\n");
	else if (ch=='S') tell("INFO SCORES\r\n");
	else if (ch=='N') tell("INFO NEWS\r\n");
	else if (ch=='H') tell("INFO HISTORY\r\n");
	else if (ch=='Q') { tell("QUIT\r\n"); return -1; }
    }
    return 0;
}

static int pre_post_key(int ch) {
    static int gotesc = 0;
    if (ch==27 && !gotesc) gotesc = 1;
    else if (gotesc) {
	gotesc = 0;
	return post_key(ch|128);
    }
    else {
	return post_key(ch);
    }
    return 0;
}

int trigger_display() {
    char buf[128];
    int len = read(0, buf, sizeof(buf));
    if (len<=0) {
	tell("QUIT\r\n");
	return -1;
    }
    int rv = 0;
    for (int i=0; i<len; i++) {
	if (pre_post_key(buf[i])) rv = -1;
    }
    return rv;
}

void opendisplay(int /*usecolor*/) {
    initscr();
    noecho();
    cbreak();
    nonl();
    idlok(stdscr, TRUE);
    //scrollok(stdscr, TRUE);  /* ? */
    say("");
    say("Professional Galactic Empires");
    say("Client version %s, built %s", VERSION, __DATE__);
    say("");
    set_displayfd(0);
    display_update();
}

void closedisplay() {
    endwin();
    printf("\n");
}
