/*
 * Professional Galactic Empires Client
 * Copyright 1996-1999 David A. Holland
 * Version 0.1, 6/11/99
 * All rights reserved.
 *
 * See the included file "COPYING" for license information.
 */

/*
 * pge client. take 2.
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <assert.h>

#include "pgeclient.h"
#include "defs.h"
#include "util.h"

static int serversock = -1;        /* fd for network connection */
static int displayfd = -1;         /* fd for display connection */

static int linemode = 1;           /* true == do network line buffering */

static char netbuf[4096];          /* network line buffer */
static size_t netbufpos;

void set_displayfd(int fd) {
    displayfd = fd;
}

void set_linemode(int mode) {
    linemode = mode;
}

void tell(const char *msg, ...) {
    char buf[4096];
    va_list ap;
    va_start(ap, msg);
    vsnprintf(buf, sizeof(buf), msg, ap);
    va_end(ap);
    write(serversock, buf, strlen(buf));
}

static void trigger_net() {
    size_t max = sizeof(netbuf)-netbufpos;
    int len = read(serversock, netbuf+netbufpos, max);
    if (len<=0) {
	if (len<0) say("server: read: %s", strerror(errno));
	close(serversock);
	serversock = -1;
    }
    else if (linemode) {
	netbufpos += len;
	char *x;
	while ((x = (char *)memchr(netbuf, '\n', netbufpos))!=NULL) {
	    if (x[-1]=='\r') x[-1] = 0;
	    x[0] = 0;
	    x++;
	    process_net_line(netbuf);
	    netbufpos -= (x-netbuf);
	    memmove(netbuf, x, netbufpos);
	    
	    /* ew. this will hopefully go away... */
	    if (!linemode) goto hack;
	}
    }
    else {
	netbufpos += len;
      hack: /* corresponding ew. */
	process_net_data(netbuf, netbufpos);
	netbufpos = 0;
    }
}

static void mainloop() {
    bool done = false;
    while (!done) {
	assert(displayfd >= 0);
	display_update();

	int hifd=displayfd;
	fd_set xset;
	FD_ZERO(&xset);
	FD_SET(displayfd, &xset);
	if (serversock>=0) {
	    FD_SET(serversock, &xset);
	    if (hifd < serversock) hifd = serversock;
	}

	if (select(hifd+1, &xset, NULL, NULL, NULL)<0) {
	    continue;
	}

	if (FD_ISSET(displayfd, &xset)) {
	    if (trigger_display()) done = true;
	}
	if (serversock>=0 && FD_ISSET(serversock, &xset)) {
	    trigger_net();
	}
    }
}

////////////////////////////////////////////////////////////

static void opensock(const char *host, int port) {
    display_update();

    struct hostent *h = gethostbyname(host);
    if (!h) {
	say("Server %s: host not found", host);
	return;
    }

    int sock = socket(AF_INET, SOCK_STREAM, 0);
    if (sock<0) {
	say("socket: %s", strerror(errno));
	return;
    }

    struct sockaddr_in sn;
    memset(&sn, 0, sizeof(sn));
    sn.sin_family = AF_INET;
    sn.sin_port = htons(port);
    memcpy(&sn.sin_addr, h->h_addr, sizeof(sn.sin_addr));

    say("Connecting to server %s...", host);
    display_update();

    if (connect(sock, (struct sockaddr *)&sn, sizeof(sn))) {
	say("connect: %s", strerror(errno));
	close(sock);
	return;
    }

    say("Connected.");
    say("");
    display_update();

    if (serversock >= 0) {
	tell("QUIT\r\n");
	shutdown(serversock, 2);
	close(serversock);
    }

    serversock = sock;
    netbufpos = 0;
}

static void usage() {
    printf("Usage: pge [ -hc ] [ -s server ] [ -p port ] [server [port]]\n");
    printf("   -h: this help message\n");
    printf("   -c: enable color (your terminal must support color\n");
    printf("   -s: specify PGE server (default is %s)\n", DEFAULTSERVER);
    printf("   -p: specify port to contact (standard is %d)\n", PGE_PORT);
    printf("\n");
    printf("PGE also recognizes the following environment variables:\n");
    printf("    PGESERVER: override default pge server\n");
    printf("    PGEPORT: override default port number\n");
    printf("\n");
}

int main(int argc, char *argv[]) {
    int port = PGE_PORT;
    const char *server = DEFAULTSERVER;
    int color = 0;

    const char *tmp;

    tmp = getenv("PGESERVER");
    if (tmp) server = tmp;

    tmp = getenv("PGEPORT");
    if (tmp) port = atoi(tmp);

    int ch;
    while ((ch=getopt(argc, argv, "hcs:p:"))!=EOF) {
	switch (ch) {
	  case 'c': color = 1; break;
	  case 's': server = optarg; break;
	  case 'p': port = atoi(optarg); break;
	  case 'h': 
	  default:
	      usage(); return 0;
	}
    }
    argc -= optind;
    argv += optind;

    if (argc>0) {
	server = argv[0];
	argc--;
	argv++;
    }
    if (argc>0) {
	port = atoi(argv[0]);
	argc--;
	argv++;
    }

    opendisplay(color);

    opensock(server, port);

    mainloop();

    shutdown(serversock, 2);
    close(serversock);

    closedisplay();

    return 0;
}
