(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 * Written by David A. Holland.
 *
 * Copyright (c) 2020 David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

(*
 * Direct synthesis
 *
 * Here we pass the solver single queries of the form
 *    exists p, forall s, pre(s) -> post(p(s))
 * in spite of their undecidability/intractability, to see if it works
 * and if so how well.
 *)

open Symbolic
module MS = Mipsstate
module M = Mips
module MT = Mipstools
module S = Smt

type 't control = SUCCESS of 't | CONTINUE | ABORT

(*
 * Turn a list of assertions and variable bindings into a a single
 * expression with the bindings made into foralls.
 *)
let forallify decls fex =
   let collect decls =
      let once (bindings, assertions) d =
         match d with
         | S.BIND (name, ty) -> ((name, ty) :: bindings, assertions)
         | S.ASSERT e1 -> (bindings, e1 :: assertions)
         | S.COMMENT _ -> (bindings, assertions)
      in
      let (bindings, assertions) = List.fold_left once ([], []) decls in
      (List.rev bindings, List.rev assertions)
   in
   let quantify (name, ty) e =
      S.FORALL (name, ty, e)
   in
   let (bindings, assertions) = collect decls in
   let assertions' = S.and_list assertions in
   List.fold_right quantify bindings (fex assertions')

let forallify_id decls =
   forallify decls (fun assertions -> assertions)

let forallify_impl decls ex =
   forallify decls (fun assertions -> S.IMPLIES (assertions, ex))

let synthesize'' pre numinsns post controlenv controlfacts allpps symcode =
   Log.dcsay ("*** Synthesizing with " ^ string_of_int numinsns ^ " insns ***");

   let timings = ref [] in

   (*
    * Update the pre/postconditions to refer to the program points in
    * the sequence.
    *)

   (* this should always be a nop, but whatever, maintain the abstraction *)
   let prepoint = Mipstools.prepoint symcode in
   let pre = Mipstools.changepp pre (Ppoint.pre) prepoint in

   let postpoint = Mipstools.postpoint symcode in
   let post = Mipstools.changepp post (Ppoint.post) postpoint in

   (*
    * Symbolically execute the sequence to produce constraint expressions.
    *)

   let (symenv, symfacts) =
      Symexec.go symcode
   in
   let prefact = ("precondition", pre) in
   let postfact = ("postcondition", post) in
   let negpostfact = ("Negated postcondition", M.LOGNOT post) in
   let allfacts =
      prefact :: postfact :: negpostfact :: controlfacts @ symfacts
   in

   let allenv =
      let doadd z (v, t) = M.VarMap.add v t z in
      List.fold_left doadd controlenv symenv
   in

   let tc'fact when_ (what, e) =
      Typecheck.go allenv e BOOL (what ^ when_)
   in
   List.iter (tc'fact "") allfacts;

   (*
    * Simplify what we got. XXX notyet; doesn't do anything anyway
    *)
(*
   let (allenv, facts, symcode) =
      Simplify.go allenv facts symcode
   in
*)

(*  turn this off until Simplify does something as there's no point
   let allfacts =
      prefact :: postfact :: negpostfact :: controlfacts @ symfacts
   in
   List.iter (tc'fact " (after Simplify)") allfacts;
*)

   (*
    * Build the SMT for the control variables.
    *)
   let control_smt =
      let elabel = "Control variable environment:" in
      let clabel = "Control variable constraints:" in
      let env = Build.build'environment elabel allenv in
      let constraints = Build.build'envconstraints clabel allenv allpps in
      let facts = Build.build'facts None allpps controlfacts in
      env @ constraints @ facts
   in

   (*
    * Just do it!
    *)

   let guess_smt =
      let statestr = Smt.STATE 0 in (* formerly just "S" *)
      let env = Build.build'execenv "State variables:" statestr allpps in
      let pre = Build.build'fact (Some statestr) allpps prefact in
      let exec = Build.build'facts (Some statestr) allpps symfacts in
      let post = Build.build'fact (Some statestr) allpps postfact in
      let post' = forallify_id post in
      [S.ASSERT (forallify_impl (env @ exec) (forallify_impl pre post'))]
   in

   (* Guess a concrete code sequence. *)
   Log.vsay ("*** Guessing ***");
   Log.csay ("Guessing  " ^ string_of_int numinsns ^ " instructions");
   let (gduration, gsat) =
      Solver.go (control_smt @ guess_smt)
   in
   timings := (gduration, "guess") :: !timings;
   let result = match gsat with
      | Solver.UNSAT _ ->
           (* no viable candidate code blocks of this length *)
           None
      | Solver.SAT result ->
           (* got a result *)

           let result = Build.filter_candidate result in
           let code = Build.unguess result symcode in

           let codetxt = Printasm.print "   " code in
           Log.vsay ("Result:");
           Log.vcsay (codetxt);

           Some code
   in
   let resultstr =
      match result with
      | Some _ -> "Done"
      | None -> "Failed"
   in
   let runstr = string_of_int (List.length !timings) ^ " solver runs" in
   Log.vcsay ("*** " ^ resultstr ^ " (" ^ runstr ^ ") ***");
   let timingstr =
      let once (dur, str) = "   " ^ string_of_float dur ^ " (" ^ str ^ ")" in
      String.concat "\n" (List.map once (List.rev !timings))
   in
   Log.timingsay (resultstr ^ " in " ^ runstr ^ "\n" ^ timingstr);
   result

let synthesize'' pre numinsns sketch post =
   (*
    * Make the symbolic instruction sequence.
    *
    * This produces:
    *    control variables that govern what instructions the sequence is
    *    assertions about the control variables to keep things well formed
    *    a list of all program points in the instruction sequence
    *    the symbolic instruction sequence itself
    *
    * The assertions are Mips.expr expressions (paired with a name for
    * use when printing, like the various "facts" below).
    *
    * The four return values are:
    *    controlenv - control variable environment (map from names to types)
    *    controlfacts - assertions about the control variables
    *    allpps - the program points
    *    symcode - the symbolic instruction list
    *)

   match Syminsns.make_symbolic_insns numinsns sketch with
   | Some (controlenv, controlfacts, allpps, symcode) ->
        synthesize'' pre numinsns post controlenv controlfacts allpps symcode
   | None ->
        Log.dcsay "*** Unsat ***";
        exit 1

let synthesize (pre, post) numinsns sketch =
   let rec dosynth numinsns =
      match synthesize'' pre numinsns sketch post with
      | Some code -> code
      | None -> dosynth (1 + numinsns)
   in
   dosynth numinsns
