(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 * Written by David A. Holland.
 *
 * Copyright (c) 2020 David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

module I = Mintops
module S = Smt

(* begin mandatory cutpaste *)
type result = SAT of Smt.model | UNSAT of Smt.counterexample
(* end mandatory cutpaste *)

(*
let _ = Z3.set_global_param "verbose" "100"
*)

(*
let _ = Z3.enable_all_trace ()
*)

let z3ctx = Z3.mk_context [("model", "true"); ("proof", "false")]
let z3bool = Z3.Boolean.mk_sort z3ctx
let z3int = Z3.Arithmetic.Integer.mk_sort z3ctx
let z3bit w = Z3.BitVector.mk_sort z3ctx w

let mkintfunc name nargs ret =
   Z3.FuncDecl.mk_func_decl_s z3ctx name (Util.repeat nargs z3int) ret

(*
 * XXX: is it bad for these to exist if we aren't using them?
 *)

(* bitops *)
let z3bitnot = mkintfunc "Zbitnot" 1 z3int
let z3bitand = mkintfunc "Zbitand" 2 z3int
let z3bitor = mkintfunc "Zbitor" 2 z3int
let z3bitxor = mkintfunc "Zbitxor" 2 z3int
let z3bitshl = mkintfunc "Zbitshl" 2 z3int
let z3bitushr = mkintfunc "Zbitushr" 2 z3int
let z3bitsshr = mkintfunc "Zbitsshr" 2 z3int

(* non-bitops *)
let z3add = mkintfunc "Zadd" 2 z3int
let z3sub = mkintfunc "Zsub" 2 z3int
(* XXX: try both mulhi, mullo and mul/2^32, mul%2^32 *)
let z3mulhi = mkintfunc "Zmulhi" 2 z3int
let z3mullo = mkintfunc "Zmullo" 2 z3int
let z3div = mkintfunc "Zdiv" 2 z3int
let z3mod = mkintfunc "Zmod" 2 z3int
let z3lt = mkintfunc "Zlt" 2 z3bool

let nextrunid = ref 0


(**************************************************************)
(* mode tools *)

let mints_are_bitvectors = Modes.mints_are_bitvectors
let xints_are_bitvectors = Modes.xints_are_bitvectors


(**************************************************************)
(* expr tools *)

let rec collect_foralls x ty e =
   let (vars, ex) =
      match e with
      | S.FORALL (x', ty', e') -> collect_foralls x' ty' e'
      | _ -> ([], e)
   in
   ((x, ty) :: vars, ex)

let rec collect_existses x ty e =
   let (vars, ex) =
      match e with
      | S.EXISTS (x', ty', e') -> collect_existses x' ty' e'
      | _ -> ([], e)
   in
   ((x, ty) :: vars, ex)


(**************************************************************)
(* insert stuff into z3 *)

let rec conv'typename ty =
   match ty with
   | S.BOOL -> z3bool
   | S.XINT w -> if xints_are_bitvectors then z3bit w else z3int
   | S.MINT ->
        if mints_are_bitvectors then z3bit 32
        else z3int
   | S.ARRAY (tk, tv) ->
        Z3.Z3Array.mk_sort z3ctx (conv'typename tk) (conv'typename tv)

let conv'var name ty =
   let name' = S.string_of_varname name in
   match ty with
   | S.BOOL -> Z3.Boolean.mk_const_s z3ctx name'
   | S.XINT w ->
        if xints_are_bitvectors then
           Z3.BitVector.mk_const_s z3ctx name' w
        else
           Z3.Arithmetic.Integer.mk_const_s z3ctx name'
   | S.MINT ->
        if mints_are_bitvectors then
           Z3.BitVector.mk_const_s z3ctx name' 32
        else
           Z3.Arithmetic.Integer.mk_const_s z3ctx name'
   | S.ARRAY (tk, tv) ->
        Z3.Z3Array.mk_const_s z3ctx name' (conv'typename tk) (conv'typename tv)


let int2bv e = Z3.Arithmetic.Integer.mk_int2bv z3ctx 32 e
(*let bv2sint e = Z3.BitVector.mk_bv2int z3ctx e true*)
let bv2uint e = Z3.BitVector.mk_bv2int z3ctx e false

let conv'mintuop op e1' =
   match op with
   | I.BITNOT -> begin
        let bitop e = Z3.BitVector.mk_not z3ctx e in
        let uninterp e = Z3.FuncDecl.apply (*z3ctx*) z3bitnot [e] in

	match Modes.smt_mint_mode with
	| Modes.MINT_AS_BV -> bitop e1'
	| Modes.MINT_AS_INT_CONVERT -> bv2uint (bitop (int2bv e1'))
	| Modes.MINT_AS_INT_BITFUNC -> uninterp e1'
	| Modes.MINT_AS_INT_ALLFUNC -> uninterp e1'
     end
   | I.SBWIDEN
   | I.SHWIDEN
   | I.UBWIDEN
   | I.UHWIDEN
   | I.U5WIDEN -> e1'
   | I.TRUNCU5 ->
        if mints_are_bitvectors then
           let thirty_one = Z3.BitVector.mk_numeral z3ctx "31" 32 in
           Z3.BitVector.mk_and z3ctx e1' thirty_one
        else        
           let thirty_two = Z3.Arithmetic.Integer.mk_numeral_i z3ctx 32 in
           Z3.Arithmetic.Integer.mk_mod z3ctx e1' thirty_two
   | I.TOSIGNED ->
        if mints_are_bitvectors then e1'
        else
	   let lim = 2147483648 in
           let lim' = Z3.Arithmetic.Integer.mk_numeral_i z3ctx lim in
           let z = Z3.Arithmetic.Integer.mk_numeral_i z3ctx 0 in
           let checklt = Z3.Arithmetic.mk_lt z3ctx e1' z in
           let doadd = Z3.Arithmetic.mk_add z3ctx [e1'; lim'] in
           Z3.Boolean.mk_ite z3ctx checklt doadd e1'
   | I.TOUNSIGNED ->
        if mints_are_bitvectors then e1'
        else
	   let lim = 2147483648 in
           let lim' = Z3.Arithmetic.Integer.mk_numeral_i z3ctx lim in
           let checkge = Z3.Arithmetic.mk_ge z3ctx e1' lim' in
           let dosub = Z3.Arithmetic.mk_sub z3ctx [e1'; lim'] in
           Z3.Boolean.mk_ite z3ctx checkge dosub e1'
   
let conv'mintbop op e1' e2' =
   (*
    * Form for operations that are not bit ops (where z3 has an
    * integer op)
    *)
   let nonbit asbit asint uninterp =
      match Modes.smt_mint_mode with
      | Modes.MINT_AS_BV -> asbit e1' e2'
      | Modes.MINT_AS_INT_CONVERT -> asint e1' e2'
      | Modes.MINT_AS_INT_BITFUNC -> asint e1' e2'
      | Modes.MINT_AS_INT_ALLFUNC -> uninterp e1' e2'
   in
   (*
    * Form for operations that are bit ops (where z3 has no integer op)
    *)
   let bit asbit uninterp =
      match Modes.smt_mint_mode with
      | Modes.MINT_AS_BV -> asbit e1' e2'
      | Modes.MINT_AS_INT_CONVERT -> bv2uint (asbit (int2bv e1') (int2bv e2'))
      | Modes.MINT_AS_INT_BITFUNC -> uninterp e1' e2'
      | Modes.MINT_AS_INT_ALLFUNC -> uninterp e1' e2'
   in

   (* shared logic for integer version of signed and unsigned mulhi *)
   let asint_mulhi e1 e2 =
      let lim = 4294967296 in
      let lim' = Z3.Arithmetic.Integer.mk_numeral_i z3ctx lim in
      let full = Z3.Arithmetic.mk_mul z3ctx [e1; e2] in
      Z3.Arithmetic.mk_div z3ctx full lim'
   in
 
   match op with
   | I.EQ -> Z3.Boolean.mk_eq z3ctx e1' e2'
   | I.ULT ->
        let asbit e1 e2 = Z3.BitVector.mk_ult z3ctx e1 e2 in
        let asint e1 e2 = Z3.Arithmetic.mk_lt z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3lt [e1; e2] in
        nonbit asbit asint uninterp
   | I.SLT ->
        let asbit e1 e2 = Z3.BitVector.mk_slt z3ctx e1 e2 in
        let asint e1 e2 = Z3.Arithmetic.mk_lt z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3lt [e1; e2] in
        nonbit asbit asint uninterp
   | I.ADD ->
        let asbit e1 e2 = Z3.BitVector.mk_add z3ctx e1 e2 in
        let asint e1 e2 = Z3.Arithmetic.mk_add z3ctx [e1; e2] in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3add [e1; e2] in
        nonbit asbit asint uninterp
   | I.SUB ->
        let asbit e1 e2 = Z3.BitVector.mk_sub z3ctx e1 e2 in
        let asint e1 e2 = Z3.Arithmetic.mk_sub z3ctx [e1; e2] in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3sub [e1; e2] in
        nonbit asbit asint uninterp
   | I.SMUL_HI ->
        let asbit e1 e2 =
           let ee1 = Z3.BitVector.mk_sign_ext z3ctx 64 e1 in
           let ee2 = Z3.BitVector.mk_sign_ext z3ctx 64 e2 in
           let res = Z3.BitVector.mk_mul z3ctx ee1 ee2 in
           Z3.BitVector.mk_extract z3ctx 32 63 res
        in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3mulhi [e1; e2] in
        nonbit asbit asint_mulhi uninterp
   | I.UMUL_HI ->
        let asbit e1 e2 =
           let ee1 = Z3.BitVector.mk_zero_ext z3ctx 64 e1 in
           let ee2 = Z3.BitVector.mk_zero_ext z3ctx 64 e2 in
           let res = Z3.BitVector.mk_mul z3ctx ee1 ee2 in
           Z3.BitVector.mk_extract z3ctx 32 63 res
        in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3mulhi [e1; e2] in
        nonbit asbit asint_mulhi uninterp
   | I.MUL_LO ->
        let asbit e1 e2 =
           (* wonder if it's better to use zero_ext or sign_ext here *)
           let ee1 = Z3.BitVector.mk_zero_ext z3ctx 64 e1 in
           let ee2 = Z3.BitVector.mk_zero_ext z3ctx 64 e2 in
           let res = Z3.BitVector.mk_mul z3ctx ee1 ee2 in
           Z3.BitVector.mk_extract z3ctx 0 31 res
        in
        let asint e1 e2 =
           let lim = 4294967296 in
           let lim' = Z3.Arithmetic.Integer.mk_numeral_i z3ctx lim in
           let full = Z3.Arithmetic.mk_mul z3ctx [e1; e2] in
           Z3.Arithmetic.Integer.mk_mod z3ctx full lim'
        in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3mullo [e1; e2] in
        nonbit asbit asint uninterp
   | I.SDIV ->
        let asbit e1 e2 = Z3.BitVector.mk_sdiv z3ctx e1 e2 in
        let asint e1 e2 = Z3.Arithmetic.mk_div z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3div [e1; e2] in
        nonbit asbit asint uninterp
   | I.SMOD ->
        let asbit e1 e2 = Z3.BitVector.mk_smod z3ctx e1 e2 in
        let asint e1 e2 = Z3.Arithmetic.Integer.mk_mod z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3mod [e1; e2] in
        nonbit asbit asint uninterp
   | I.UDIV ->
        let asbit e1 e2 = Z3.BitVector.mk_udiv z3ctx e1 e2 in
        let asint e1 e2 = Z3.Arithmetic.mk_div z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3div [e1; e2] in
        nonbit asbit asint uninterp
   | I.UMOD ->
        let asbit e1 e2 = Z3.BitVector.mk_urem z3ctx e1 e2 in
        let asint e1 e2 = Z3.Arithmetic.Integer.mk_mod z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3mod [e1; e2] in
        nonbit asbit asint uninterp
   | I.BITAND ->
        let asbit e1 e2 = Z3.BitVector.mk_and z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3bitand [e1; e2] in
        bit asbit uninterp
   | I.BITOR ->
        let asbit e1 e2 = Z3.BitVector.mk_or z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3bitor [e1; e2] in
        bit asbit uninterp
   | I.BITXOR ->
        let asbit e1 e2 = Z3.BitVector.mk_xor z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3bitxor [e1; e2] in
        bit asbit uninterp
   | I.SHL ->
        let asbit e1 e2 = Z3.BitVector.mk_shl z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3bitshl [e1; e2] in
        bit asbit uninterp
   | I.USHR ->
        let asbit e1 e2 = Z3.BitVector.mk_lshr z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3bitushr [e1; e2] in
        bit asbit uninterp
   | I.SSHR ->
        let asbit e1 e2 = Z3.BitVector.mk_ashr z3ctx e1 e2 in
        let uninterp e1 e2 = Z3.FuncDecl.apply (*z3ctx*) z3bitsshr [e1; e2] in
        bit asbit uninterp

let rec conv'expr vars e =
   let make_int n =
      if Big_int.is_int_big_int n then
         Z3.Arithmetic.Integer.mk_numeral_i z3ctx (Big_int.int_of_big_int n)
      else
         Z3.Arithmetic.Integer.mk_numeral_s z3ctx (Big_int.string_of_big_int n)
   in
   let make_xint w n =
      if xints_are_bitvectors then
         Z3.BitVector.mk_numeral z3ctx (Big_int.string_of_big_int n) w
      else make_int n
   in
   let make_mint n =
     if mints_are_bitvectors then
        Z3.BitVector.mk_numeral z3ctx (Big_int.string_of_big_int n) 32
     else make_int n
   in

   (*
    * Build an array value. Do it by inserts, not by constraining the
    * elements.
    *)
   let make_array make_elt kw m d =
      let t' = conv'typename (S.XINT kw) in
      let d' = make_elt d in
      let d'' = Z3.Z3Array.mk_const_array z3ctx t' d' in
      let once z (k, v) =
         Z3.Z3Array.mk_store z3ctx z (make_xint kw k) (make_elt v)
      in
      List.fold_left once d'' (Types.BigIntMap.bindings m)
   in
      

   (*
    * Shared logic for forall/exists. varlist and ex come from collect_*;
    * varlist is the list of variables to bind (list of name/type pairs)
    * and ex is the underlying subexpression to continue with.
    *
    * Returns the z3 variable list and the compiled form of ex.
    *)
   let prep_quantifier varlist ex =
      (* make z3 vars *)
      let varlist =
         let once (x1, ty1) = (x1, ty1, conv'var x1 ty1) in
         List.map once varlist
      in
      (* make the variable environment for inside *)
      let vars' =
         let once vars' (x, ty, x') =
            S.VarMap.add x (ty, x') vars'
         in
         List.fold_left once vars varlist
      in
      (* extract just the z3 vars *)
      let xs' =
         let once (_, _, x') = x' in
         List.map once varlist
      in
      (* build the subexpression *)
      let ex' = conv'expr vars' ex in
      (xs', ex')
   in

   match e with
   | S.TRUE -> Z3.Boolean.mk_true z3ctx
   | S.FALSE -> Z3.Boolean.mk_false z3ctx
   | S.XINTVAL (n, w) -> make_xint w n
   | S.MINTVAL n -> make_mint n
   | S.BOOLARRAYVAL (m, d, kw) ->
        let make b =
           if b then Z3.Boolean.mk_true z3ctx else Z3.Boolean.mk_false z3ctx
        in
        make_array make kw m d
   | S.XINTARRAYVAL (m, d, kw, vw) -> make_array (make_xint vw) kw m d
   | S.MINTARRAYVAL (m, d, kw) -> make_array make_mint kw m d
   | S.READVAR x -> begin
        try
           let (_ty, e) = S.VarMap.find x vars in e
        with Not_found ->
           Util.crash ("solver: unbound smt var " ^ S.string_of_varname x)
     end
   | S.READARRAY (a, e1) ->
        let a' =
           try
              let (_ty, e) = S.VarMap.find a vars in e
           with Not_found ->
              Util.crash ("solver: unbound smt array " ^ S.string_of_varname a)
        in
        let e1' = conv'expr vars e1 in
        Z3.Z3Array.mk_select z3ctx a' e1'
   | S.UPDATEARRAY (a, e1, e2) ->
        let a' =
           try
              let (_ty, e) = S.VarMap.find a vars in e
           with Not_found ->
              Util.crash ("solver: unbound smt array " ^ S.string_of_varname a)
        in
        let e1' = conv'expr vars e1 in
        let e2' = conv'expr vars e2 in
        Z3.Z3Array.mk_store z3ctx a' e1' e2'
   | S.LET (x, ty, e1, e2) ->
        let x' = conv'var x ty in
        let _e1' = conv'expr vars e1 in
        let _e2' = conv'expr (S.VarMap.add x (ty, x') vars) e2 in
        Util.crash "solver: let-binding" (* XXX *)
   | S.IF (c, t, f) ->
        let c' = conv'expr vars c in
        let t' = conv'expr vars t in
        let f' = conv'expr vars f in
        Z3.Boolean.mk_ite z3ctx c' t' f'
   | S.NOT e1 ->
        let e1' = conv'expr vars e1 in
        Z3.Boolean.mk_not z3ctx e1'
   | S.AND (e1, e2) ->
        let e1' = conv'expr vars e1 in
        let e2' = conv'expr vars e2 in
        Z3.Boolean.mk_and z3ctx [e1'; e2']
   | S.OR (e1, e2) ->
        let e1' = conv'expr vars e1 in
        let e2' = conv'expr vars e2 in
        Z3.Boolean.mk_or z3ctx [e1'; e2']
   | S.XOR (e1, e2) ->
        let e1' = conv'expr vars e1 in
        let e2' = conv'expr vars e2 in
        Z3.Boolean.mk_xor z3ctx e1' e2'
   | S.EQ (e1, e2) ->
        let e1' = conv'expr vars e1 in
        let e2' = conv'expr vars e2 in
        Z3.Boolean.mk_eq z3ctx e1' e2'
   | S.IMPLIES (e1, e2) ->
        let e1' = conv'expr vars e1 in
        let e2' = conv'expr vars e2 in
        Z3.Boolean.mk_implies z3ctx e1' e2'
   | S.FORALL (x, ty, e1) ->
        let (varlist, ex) = collect_foralls x ty e1 in
        let (xs', ex') = prep_quantifier varlist ex in
        let q =
           Z3.Quantifier.mk_forall_const z3ctx xs' ex' None [] [] None None
        in
        Z3.Quantifier.expr_of_quantifier q
   | S.EXISTS (x, ty, e1) ->
        let (varlist, ex) = collect_existses x ty e1 in
        let (xs', ex') = prep_quantifier varlist ex in
        let q =
           Z3.Quantifier.mk_exists_const z3ctx xs' ex' None [] [] None None
        in
        Z3.Quantifier.expr_of_quantifier q
   | S.MINTUOP (op, e1) ->
        let e1' = conv'expr vars e1 in
        conv'mintuop op e1'
   | S.MINTBOP (op, e1, e2) ->
        let e1' = conv'expr vars e1 in
        let e2' = conv'expr vars e2 in
        conv'mintbop op e1' e2'

let insert'decl vars solver d =
   match d with
   | S.BIND (name, ty) -> begin
        let e' = conv'var name ty in
        vars := S.VarMap.add name (ty, e') !vars
        (* XXX is this how you're supposed to do this? *)
        (* guess not *)
        (*Z3.Solver.add solver [e']*)
     end
   | S.ASSERT e ->
        let e' = conv'expr !vars e in
        Z3.Solver.add solver [e']
   | S.COMMENT _txt -> ()

(**************************************************************)
(* extract results *)

let decode_bool ze where =
   if Z3.Boolean.is_bool ze then
      match Z3.Boolean.get_bool_value ze with
      | Z3enums.L_FALSE -> S.FALSE
      | Z3enums.L_TRUE -> S.TRUE
      | Z3enums.L_UNDEF ->
           Util.crash ("solver: decode: " ^ where ^ ": L_UNDEF")
   else
       Util.crash ("solver: decode: " ^ where ^ ": not boolean")

let decode_number ze where =
   if Z3.Arithmetic.is_int_numeral ze then
      Z3.Arithmetic.Integer.get_big_int ze
   else if Z3.BitVector.is_bv ze then
      (* this actually produces a string *)
      let s = Z3.BitVector.numeral_to_string ze in
      Big_int.big_int_of_string s
   else
      Util.crash ("solver: decode_number: " ^ where ^ ": not a number")

let decode_xint w ze where =
   let k = decode_number ze (where ^ " (mint)") in
   S.XINTVAL (k, w)

let decode_mint ze where =
   let k = decode_number ze (where ^ " (xint)") in
   S.MINTVAL k

(*
 * With z3 4.8.3, array values come back as "constants", but if you
 * use get_const_interp it throws an exception telling you to use a
 * function interpretation. Unfortunately, if you do that it also
 * throws, saying that it's not an array value. This would seem to be
 * a showstopper, but it turns out you can extract the array model by
 * hand: asking the model to evaluate the array variable gives you the
 * expression for the interpretation, which is a lambda; then you need
 * to know that lambdas are quantifiers (which is not obvious from the
 * api docs) and then you can unwind the body because it's got a
 * stylized structure.
 *)
let decode_array model name arrexpr ty decode =
   let crash msg =
      Util.crash ("solver: decode_model_array: " ^ msg)
   in
   let rec unpack body =
      if Z3.Boolean.is_ite body then begin
         let bodyparts = Z3.Expr.get_args body in
         let cond = List.nth bodyparts 0 in
         let iftrue = List.nth bodyparts 1 in
         let iffalse = List.nth bodyparts 2 in
         let iftrue' = decode iftrue "true branch of if" in
         if not (Z3.Boolean.is_eq cond) then
            crash "condition of if not an equality test"
         else ();
         let condparts = Z3.Expr.get_args cond in
         let _condvar = List.nth condparts 0 in
         let condval = List.nth condparts 1 in
         if not (Z3.Expr.is_numeral condval) then
            crash "equality test in if not constant on RHS"
         else ();
         let k = decode_number condval
                    "condition constant in array function"
         in
         let (more, default) = unpack iffalse in
         ((k, iftrue') :: more, default)
      end else if Z3.Z3Array.is_store body then begin
         let bodyparts = Z3.Expr.get_args body in
         let iffalse = List.nth bodyparts 0 in
         let condval = List.nth bodyparts 1 in
         let iftrue = List.nth bodyparts 2 in
         let iftrue' = decode iftrue "value part of array store op" in
         let k = decode_number condval "condition constant in array store" in
         let (more, default) = unpack iffalse in
         ((k, iftrue') :: more, default)
      end else if Z3.Boolean.is_eq body then begin
         (* it simplifies if x = n then true else false to just x = n *)
         let condparts = Z3.Expr.get_args body in
         let _condvar = List.nth condparts 0 in
         let condval = List.nth condparts 1 in
         if not (Z3.Expr.is_numeral condval) then
            crash "equality test in final if not constant on RHS"
         else ();
         let k = decode_number condval
                    "condition constant in array function final step"
         in
         ((k, S.TRUE) :: [], S.FALSE)
      end else if Z3.Z3Array.is_constant_array body then begin
         let bodyparts = Z3.Expr.get_args body in
         let condval = List.nth bodyparts 0 in
         let default = decode condval "array constant expression" in
         ([], default)
      end else if Z3.Boolean.is_bool body || Z3.Expr.is_numeral body then begin
         let body' = decode body "final branch of if tree" in
         ([], body')
      end else
         Util.crash "solver: Unidentified flying array model"
   in
   let arrval =
      match Z3.Model.eval model arrexpr true with
      | None -> crash "eval yielded None"
      | Some e -> e
   in

   let is_quantifier e =
      (* there doesn't seem to be an is_quantifier for expressions, wtf *)
      Z3.AST.is_quantifier (Z3.Expr.ast_of_expr e)
   in
   let body =
      if is_quantifier arrval then
         (* lambda expression *)
         let q = Z3.Quantifier.quantifier_of_expr arrval in
         Z3.Quantifier.get_body q
      else
         arrval
   in
   let (cases, default) = unpack body in
   let map =
      let doadd z (k, v) = Types.BigIntMap.add k v z in
      List.fold_left doadd Types.BigIntMap.empty cases
   in
   S.ASSIGNARRAY (name, ty, map, default)


let decode_model_expr varinfo model zfd =
   let sym' = Z3.Symbol.get_string (Z3.FuncDecl.get_name zfd) in
   let sym = S.varname_of_string sym' in

   let (ty, z3expr) =
      try
         S.VarMap.find sym varinfo
      with Not_found ->
         Util.crash ("solver: decode: no varinfo for " ^ sym' ^
                     "(parsed name: " ^ S.string_of_varname sym ^ ")")
   in

   (*
    * Check for an array. Do this first because the get_const_interp
    * call on an array value will throw an exception.
    *)
   match ty with
   | S.ARRAY _ ->
        let decode =
           match ty with
           | S.ARRAY (_, S.BOOL) -> decode_bool
           | S.ARRAY (_, S.XINT w) -> decode_xint w
           | S.ARRAY (_, S.MINT) -> decode_mint
           | _ -> Util.crash "solver: decode_model_expr: bad array"
        in
        decode_array model sym z3expr ty decode
   | _ -> (* continue *)

   match Z3.Model.get_const_interp model zfd with
   | None -> Util.crash ("solver: decode: no result for " ^ sym')
   | Some ze ->
        match ty with
        | S.BOOL ->
             let e = decode_bool ze ("boolean value for " ^ sym') in
             S.ASSIGN (sym, S.BOOL, e)
        | S.XINT w ->
             let e = decode_xint w ze ("integer value for " ^ sym') in
             S.ASSIGN (sym, S.XINT w, e)
        | S.MINT ->
             let e = decode_mint ze ("machine integer value for " ^ sym') in
             S.ASSIGN (sym, S.MINT, e)
        | S.ARRAY _ -> Util.crash ("solver: decode: stray array??")
      

(**************************************************************)
(* external interface *)

let verbose = ref false

let setup vrb =
   verbose := vrb;
(*
   if Z3.Log.open_ (nameprefix ^ ".z3.log") then ()
   else Util.crash ("Failed to open z3 log");
*)
   Log.dcsay ("z3 version: " ^ Z3.Version.full_version)

let shutdown () = ()
(*
   Z3.Log.close ()
*)

let vsay msg =
   if !verbose then Log.dsay msg else ()

let vcsay msg =
   Log.csay msg;
   if !verbose then Log.dsay msg else ()

let checkit say (smt: Smt.smt) =
(*
   say ("Input smt:");
   say (Printsmt.print "   " smt);
   Smtcheck.go smt;
*)

   let smt' = Smtopt.go smt in
   say ("Improved smt:");
   say (Printsmt.print "   " smt');
   Log.flush ();

   Smtcheck.go smt';

(*
   let csmt = Smtcanon.go smt' in
   say ("Canonicalized smt:");
   say (Printsmt.print "   " csmt);
   Log.flush ();
*)
   smt'

let go (smt: Smt.smt) =
   let runid = !nextrunid in
   nextrunid := 1 + runid;

   vcsay ("--- Solver run ---");
   vcsay ("Run " ^ string_of_int runid);

   let smt' = checkit vsay smt in

   let solver = Z3.Solver.mk_solver z3ctx None in
   let varinfo = ref S.VarMap.empty in
   List.iter (insert'decl varinfo solver) smt';

   Log.flush ();

   let starttime = Unix.gettimeofday () in
   let result = Z3.Solver.check solver [] in
   let endtime = Unix.gettimeofday () in
   let duration = endtime -. starttime in

   match result with
   | Z3.Solver.UNSATISFIABLE ->
        vcsay ("Result: unsat");
        (duration, UNSAT []) (* XXX *)
   | Z3.Solver.UNKNOWN ->
        vcsay ("Result: unknown");
        Util.crash "solver: UNKNOWN"
   | Z3.Solver.SATISFIABLE ->
        match Z3.Solver.get_model solver with
        | None -> Util.crash "solver: no model"
        | Some m ->
             vcsay ("result: sat");

             vsay ("Raw model:");
             vsay (Z3.Model.to_string m);

             let decls = Z3.Model.get_const_decls m in
             vsay (string_of_int (List.length decls) ^ " values");
             let assignments = List.map (decode_model_expr !varinfo m) decls in
             vsay ("model:");
             let printone a =
                match a with
                | S.ASSIGN (x, ty, e) ->
                     let x' = S.string_of_varname x in
                     let ty' = Printsmt.print_typename ty in
                     let e' = Printsmt.print_expr "" e in
                     let txt = x' ^ " :: " ^ ty' ^ " = " ^ e' in
                     vsay txt
                | S.ASSIGNARRAY (x, ty, map, default) ->
                     let x' = S.string_of_varname x in
                     let ty' = Printsmt.print_typename ty in
                     let once (k, v) =
                        let v' = Printsmt.print_expr "" v in
                        "   " ^ Big_int.string_of_big_int k ^ " -> " ^ v'
                     in
                     let default' =
                        "   _ -> " ^ Printsmt.print_expr "" default
                     in
                     let strings =
                        List.map once (Types.BigIntMap.bindings map)
                        @ [default']
                     in
                     let txt = x' ^ " :: " ^ ty' ^ " =" in
                     vsay txt; List.iter vsay strings
             in
             List.iter printone assignments;
             (duration, SAT assignments)

let dryrun isguess (smt: Smt.smt) =
   let _smt' = checkit Log.dsay smt in
   if isguess then (1.0, SAT [])
   else (1.0, UNSAT [])
