(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 * Written by David A. Holland.
 *
 * Copyright (c) 2020 David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

(* 
 * build SMT
 *)

open Symbolic
module I = Mintops
module MS = Mipsstate
module M = Mips
module MT = Mipstools
module S = Smt

(* XXX? but 256 program points is a *lot*... *)
let programpoint_bitwidth = 8

(**************************************************************)
(* grr %#$!$# *)

let rec jackass_compare_expr e1 e2 =
   let chain a b () =
      match a () with
      | 0 -> b ()
      | c -> c
   in
   let r a b () = jackass_compare_expr a b in
   let rr a b c d () = chain (r a b) (r c d) () in
(*
   let rec rs a b () =
      match a, b with
      | [], [] -> 0
      | _, [] -> 1
      | [], _ -> -1
      | a' :: amore, b' :: bmore -> chain (r a' b') (rs amore bmore) ()
   in
*)
   let p a b () = compare a b in

   match e1, e2 with
   | S.XINTVAL (a, aw), S.XINTVAL (b, bw) ->
        chain (fun () -> Big_int.compare_big_int a b) (p aw bw) ()
   | S.MINTVAL a, S.MINTVAL b ->
        Big_int.compare_big_int a b
   | S.LET (x1, ty1, e1a', e1b'), S.LET (x2, ty2, e2a', e2b') ->
        chain (p x1 x2) (chain (p ty1 ty2) (rr e1a' e1b' e2a' e2b')) ()
   | S.NOT e1', S.NOT e2' -> r e1' e2' ()
   | S.AND (e1a', e1b'), S.AND (e2a', e2b') -> rr e1a' e1b' e2a' e2b' ()
   | S.OR (e1a', e1b'), S.OR (e2a', e2b') -> rr e1a' e1b' e2a' e2b' ()
   | S.XOR (e1a', e1b'), S.XOR (e2a', e2b') -> rr e1a' e1b' e2a' e2b' ()
   | S.EQ (e1a', e1b'), S.EQ (e2a', e2b') -> rr e1a' e1b' e2a' e2b' ()
   | S.IMPLIES (e1a', e1b'), S.IMPLIES (e2a', e2b') ->
        rr e1a' e1b' e2a' e2b' ()
   | S.FORALL (x1, ty1, e1'), S.FORALL (x2, ty2, e2') ->
        chain (p x1 x2) (chain (p ty1 ty2) (r e1' e2')) ()
   | S.EXISTS (x1, ty1, e1'), S.EXISTS (x2, ty2, e2') ->
        chain (p x1 x2) (chain (p ty1 ty2) (r e1' e2')) ()
   | S.MINTUOP (f1, e1'), S.MINTUOP (f2, e2') ->
        chain (p f1 f2) (r e1' e2') ()
   | S.MINTBOP (f1, e1a', e1b'), S.MINTBOP (f2, e2a', e2b') ->
        chain (p f1 f2) (rr e1a' e1b' e2a' e2b') ()
   | _, _ -> compare e1 e2

let jackass_compare_decl d1 d2 =
   match d1, d2 with
   | S.ASSERT e1, S.ASSERT e2 -> jackass_compare_expr e1 e2
   | _, _ -> compare d1 d2

(**************************************************************)
(* stringery *)

(**************************************************************)
(* smt-level projections of things *)

(*
let proj_reg r =
   match r with
   | SYM _ -> Util.crash "proj_reg: symbolic"
   | CONC r' -> MS.int_of_reg r'

let proj_creg r =
   match r with
   | SYM _ -> Util.crash "proj_creg: symbolic"
   | CONC cr -> MS.int_of_creg cr

let proj_freg r =
   match r with
   | SYM _ -> Util.crash "proj_freg: symbolic"
   | CONC fr -> MS.int_of_freg fr

let proj_fcreg r =
   match r with
   | SYM _ -> Util.crash "proj_fcreg: symbolic"
   | CONC fcr -> MS.int_of_fcreg fcr
*)

let proj_cachestate c =
   MS.int_of_cachestate c

let proj_cache c =
   MS.int_of_cache c

let proj_cacheentryid c =
   MS.int_of_cacheentryid c

let proj_cacheentry (c, ce) =
   MS.int_of_cacheentry (c, ce)

let proj_golem g =
   MS.int_of_golem g

let proj_pp pp = Ppoint.to_smt pp


(* smt-level type projection *)
let rec mk'typename ty =
   match ty with
   | M.BOOL -> S.BOOL
   | M.CACHESTATE -> S.XINT MS.cachestate_bitwidth
   | M.UINT5
   | M.SINT8
   | M.UINT8
   | M.SINT16
   | M.UINT16
   | M.SINT32
   | M.UINT32 -> S.MINT
   | M.REGISTER
   | M.CREGISTER
   | M.FREGISTER
   | M.FCREGISTER -> S.XINT MS.register_bitwidth
   | M.HIREGISTER -> Util.crash "build: mk'typename: HIREGISTER singleton"
   | M.LOREGISTER -> Util.crash "build: mk'typename: LOREGISTER singleton"
   | M.GOLEM -> S.XINT MS.golem_bitwidth
   | M.PROGRAMPOINT -> S.XINT programpoint_bitwidth
   | M.WHICHCACHE -> S.XINT MS.whichcache_bitwidth
   | M.WHICHCACHEENTRYID -> S.XINT MS.whichcacheentryid_bitwidth
   | M.WHICHCACHEENTRY -> S.XINT MS.whichcacheentry_bitwidth
   | M.INSN -> S.XINT Insns.insn_bitwidth
   | M.MAP (t1, t2) -> S.ARRAY (mk'typename t1, mk'typename t2)

(**************************************************************)
(* variable names *)

(* DISABLED

let conv_name (k, corename) = 
   "insn" ^ string_of_int k ^ "_" ^ corename

(*
 * Absolute name: variable independent of startstate choice or program
 * point. Cannot be a golem (golems are per-startstate) and cannot be
 * the variable holding the value of something as those variables
 * exist when those values are startstate or program point dependent.
 *)
let gen_absname corename =
   let golem = S.NOTGOLEM in
   let isvalue = S.NOTVALUE in
   let qual = None in
   { S.state = None; S.pp = None; S.golem; S.isvalue; S.corename; S.qual; }

(*
 * Miscellaneous name: variable dependent on both startstate choice
 * and program point, but not a golem 
 *)
let gen_name state pp corename =
   let state = Some state in
   let pp = Some pp in
   let golem = S.NOTGOLEM in
   let isvalue = S.NOTVALUE in
   let qual = None in
   { S.state; S.pp; S.golem; S.isvalue; S.corename; S.qual; }

(*
 * Name dependent on startstate choice and program point. Not a golem
 * (golems are independent of program point) -- might be either a
 * variable for an object (NOTVALUE) or the value found in an object
 * named by a symbol (ISVALUE).
 *)
let gen_statename state pp isvalue corename =
   let state = Some state in
   let pp = Some pp in
   let golem = S.NOTGOLEM in
   let qual = None in
   { S.state; S.pp; S.golem; S.isvalue; S.corename; S.qual; }

(*
 * Name dependent on startstate choice but not program point. Instead
 * it has the instruction number in it from upstream. XXX: these should
 * not exist or be systematized better here.
 *
 * Might be either a variable for an object (NOTVALUE) or the value
 * found in an object named by a symbol (ISVALUE).
 *)
let gen_statename' state isvalue corename =
   let state = Some state in
   let pp = None in
   let golem = S.NOTGOLEM in
   let qual = None in
   { S.state; S.pp; S.golem; S.isvalue; S.corename; S.qual; }

(*
 * Name of a golem variable.
 *)
let gen_golemname state corename =
   let state = Some state in
   let golem = S.ISGOLEM in
   let isvalue = S.NOTVALUE in
   let qual = None in
   { S.state; S.pp = None; S.golem; S.isvalue;  S.corename; S.qual; }

DISABLED *)

let change_state olde neue x =
   let wrong () =
      let olde' =
         match olde with
         | S.INIT k -> "INIT " ^ string_of_int k
         | S.STATE k -> "STATE " ^ string_of_int k
         | S.VERIFY -> "VERIFY"
      in
      Util.crash ("build: change_state: " ^ olde' ^
                  " does not match " ^ S.string_of_varname x)
   in
   let change s =
      if s <> olde then wrong ()
      else neue
   in
   match x with
   | S.PLAINVAR _ -> wrong ()
   | S.CONTROLVAR _ -> wrong ()
   | S.STATEVAR (s, k, x') -> S.STATEVAR (change s, k, x')
   | S.MACHINESTATE (s, pp, x') -> S.MACHINESTATE (change s, pp, x')
   | S.GOLEMVAR (s, x') -> S.GOLEMVAR (change s, x')


(**************************************************************)
(* state names *)

(*
let regarrayname = "regarray"
let cregarrayname = "cregarray"
let fregarrayname = "fregarray"
let fcregarrayname = "fcregarray"
let golemarrayname = "golemarray"
*)

let gen_state_varnames state pp =
   let regs = List.map MS.string_of_reg (Mipstools.nonz0_regs_raw ()) in
   let cregs = List.map MS.string_of_creg (Mipstools.all_cregs_raw ()) in
   let fregs = List.map MS.string_of_freg (Mipstools.all_fregs_raw ()) in
   let fcregs = List.map MS.string_of_fcreg (Mipstools.all_fcregs_raw ()) in
   let allregs = regs @ cregs @ fregs @ fcregs in
   let allregs_types = List.map (fun r -> (r, S.MINT)) allregs in

(*
   let regarrays = [
      regarrayname;
      cregarrayname;
      fregarrayname;
      fcregarrayname;
   ] in
   let regarrays_types =
      let mk a = (a, S.ARRAY (S.XINT MS.register_bitwidth, S.MINT)) in
      List.map mk regarrays
   in
*)

   let reg_types =
(*
      match Modes.smt_register_mode with
      | Modes.REGS_VIA_VARS ->
*)
           allregs_types
(*
      | Modes.REGS_VIA_ARRAYEQS
      | Modes.REGS_VIA_ARRAYSTORES -> regarrays_types
*)
   in

   let cacheentries_types =
      let mk ce =
         (MS.string_of_cacheentry ce, S.XINT MS.cachestate_bitwidth)
      in
      List.map mk (Mipstools.all_cacheentries_raw ())
   in

   let mksym (name, ty) = (S.MACHINESTATE (state, pp, name), ty) in
   List.map mksym (reg_types @ cacheentries_types)

let gen_golem_varnames state =
(*
   match Modes.smt_register_mode with
   | Modes.REGS_VIA_VARS ->
*)
        let make g = (S.GOLEMVAR (state, MS.string_of_golem g), S.BOOL) in
        List.map make (Mipstools.all_golems_raw ())
(*
   | Modes.REGS_VIA_ARRAYEQS
   | Modes.REGS_VIA_ARRAYSTORES ->
        let w = MS.golem_bitwidth in
        [(S.GOLEMVAR (state, golemarrayname), S.ARRAY (S.XINT w, S.BOOL))]
*)

let gen_state_binding statestr pp =
   let bind (sym, ty) =
      S.BIND (sym, ty)
   in
   List.map bind (gen_state_varnames statestr pp)

let gen_golem_binding statestr =
   let bind (sym, ty) = S.BIND (sym, ty) in
   List.map bind (gen_golem_varnames statestr)

let gen_state_bindings statestr pps =
   List.concat (List.map (gen_state_binding statestr) pps)

let gen_state_constraint state pp =
   let regarrayconstraints =
(*
      match Modes.smt_register_mode with
      | Modes.REGS_VIA_VARS ->
*)
           []
(*
      | Modes.REGS_VIA_ARRAYEQS ->
           let n = S.MACHINESTATE (state, pp, regarrayname) in
           let z0 = S.xintval (MS.int_of_reg MS.Z0) MS.register_bitwidth in
           [ S.ASSERT (S.minteq (S.READARRAY (n, z0)) (S.mintval 0)); ]
      | Modes.REGS_VIA_ARRAYSTORES ->
           (* XXX should do this in store style... not sure how though *)
           let n = S.MACHINESTATE (state, pp, regarrayname) in
           let z0 = S.xintval (MS.int_of_reg MS.Z0) MS.register_bitwidth in
           [ S.ASSERT (S.minteq (S.READARRAY (n, z0)) (S.mintval 0)); ]
*)
   in
   let cacheconstraints =
      let states = Mipstools.all_cachestates_raw () in
      let statevals = List.map MS.int_of_cachestate states in
      let w = MS.cachestate_bitwidth in
      let constrain_entry ce =
         let name = S.MACHINESTATE (state, pp, MS.string_of_cacheentry ce) in
         let readvar = S.READVAR name in
         let possible_value v = S.xinteq readvar (S.xintval v w) in
         (*
          * XXX: apparently it's important for the solver for these
	  * lists to be in reverse order, i.e., zero is last, not
	  * first. With both this and the other case below in forward
	  * order, tests/assign.pen takes ~30 iterations; with either
	  * one and not the other, it takes 4 iterations; with neither
	  * it takes 1. So list them in reverse order. Review this at
	  * some future point.
	  *)
         S.ASSERT (S.or_list (List.map possible_value (List.rev statevals)))
      in
      List.map constrain_entry (Mipstools.all_cacheentries_raw ()) 
   in
   regarrayconstraints @ cacheconstraints

let gen_state_constraints statestr pp =
   List.concat (List.map (gen_state_constraint statestr) pp)

let gen_state_different statestr1 otherstatestrs pp require_all =
   let vars1 = gen_state_varnames statestr1 pp in
   let onebatch statestr2 =
      let vars2 = gen_state_varnames statestr2 pp in
      let pairs = Util.zip vars1 vars2 in
      let once ((sym1, ty1), (sym2, ty2)) =
         assert (ty1 = ty2);
         match ty1 with
         | S.BOOL -> S.xinteq (S.READVAR sym1) (S.NOT (S.READVAR sym2))
         | S.XINT _ -> S.NOT (S.xinteq (S.READVAR sym1) (S.READVAR sym2))
         | S.MINT -> S.NOT (S.minteq (S.READVAR sym1) (S.READVAR sym2))
         | S.ARRAY _ -> S.NOT (S.xinteq (S.READVAR sym1) (S.READVAR sym2))
      in
      (* do we want OR (at least one different) or AND (all different)? *)
      let verb = if require_all then S.and_list else S.or_list in
      verb (List.map once pairs)
   in
   S.and_list (List.map onebatch otherstatestrs)

let gen_golem_different state1 state2 =
   let vars1 = gen_golem_varnames state1 in
   let vars2 = gen_golem_varnames state2 in
   let pairs = Util.zip vars1 vars2 in
   let once ((sym1, ty1), (sym2, ty2)) =
      assert (ty1 = S.BOOL && ty2 = S.BOOL);
      S.xinteq (S.READVAR sym1) (S.NOT (S.READVAR sym2))
   in
   S.and_list (List.map once pairs)

(**************************************************************)
(* machine state access *)

(*
 * Figure out which array to look in for an expression 
 * (XXX: gross)
 *)
(*
let rec getarray mkname e =
   match e with
   | M.REGVALUE _ -> mkname regarrayname
   | M.CREGVALUE _ -> mkname cregarrayname
   | M.FREGVALUE _ -> mkname fregarrayname
   | M.FCREGVALUE _ -> mkname fcregarrayname
   | M.READVAR (_, M.REGISTER) -> mkname regarrayname
   | M.READVAR (_, M.CREGISTER) -> mkname cregarrayname
   | M.READVAR (_, M.FREGISTER) -> mkname fregarrayname
   | M.READVAR (_, M.FCREGISTER) -> mkname fcregarrayname
   | M.LET (_, _, _, e2) -> getarray mkname e2
   | M.IF (_, _, e2) -> getarray mkname e2
   | _ -> Util.crash "build: getarray: not state element"
*)

(*
 * Access to concrete state 
 *)

let readreg mkname r =
(*
   match Modes.smt_register_mode with
   | Modes.REGS_VIA_VARS ->
*)
        if r = MS.Z0 then S.mintval 0
        else S.READVAR (mkname (MS.string_of_reg r))
(*
   | Modes.REGS_VIA_ARRAYEQS
   | Modes.REGS_VIA_ARRAYSTORES ->
        let w = MS.register_bitwidth in
        S.READARRAY (mkname regarrayname, S.xintval (MS.int_of_reg r) w)
*)

let readcreg mkname cr =
(*
   match Modes.smt_register_mode with
   | Modes.REGS_VIA_VARS ->
*)
        S.READVAR (mkname (MS.string_of_creg cr))
(*
   | Modes.REGS_VIA_ARRAYEQS
   | Modes.REGS_VIA_ARRAYSTORES ->
        let w = MS.register_bitwidth in
        S.READARRAY (mkname cregarrayname, S.xintval (MS.int_of_creg cr) w)
*)

let readfreg mkname fr =
(*
   match Modes.smt_register_mode with
   | Modes.REGS_VIA_VARS ->
*)
        S.READVAR (mkname (MS.string_of_freg fr))
(*
   | Modes.REGS_VIA_ARRAYEQS
   | Modes.REGS_VIA_ARRAYSTORES ->
        let w = MS.register_bitwidth in
        S.READARRAY (mkname fregarrayname, S.xintval (MS.int_of_freg fr) w)
*)

let readfcreg mkname fcr =
(*
   match Modes.smt_register_mode with
   | Modes.REGS_VIA_VARS ->
*)
        S.READVAR (mkname (MS.string_of_fcreg fcr))
(*
   | Modes.REGS_VIA_ARRAYEQS
   | Modes.REGS_VIA_ARRAYSTORES ->
        let w = MS.register_bitwidth in
        S.READARRAY (mkname fcregarrayname, S.xintval (MS.int_of_fcreg fcr) w)
*)

let readhi mkname =
   S.READVAR (mkname "hi")

let readlo mkname =
   S.READVAR (mkname "lo")

let readcacheentry mkname (c, ce) =
   S.READVAR (mkname (MS.string_of_cacheentry (c, ce)))

let readgolem state g =
(*
   match Modes.smt_register_mode with
   | Modes.REGS_VIA_VARS ->
*)
        S.READVAR (S.GOLEMVAR (state, MS.string_of_golem g))
(*
   | Modes.REGS_VIA_ARRAYEQS
   | Modes.REGS_VIA_ARRAYSTORES ->
        S.READARRAY (S.GOLEMVAR (state, golemarrayname),
                     S.xintval (MS.int_of_golem g) MS.golem_bitwidth)
*)


(**************************************************************)
(* compilation *)

type ctx = {
   state: S.stateid option;
   allpps: Ppoint.t list;
}

let ctx_state ctx =
   match ctx.state with
   | Some s -> s
   | None -> Util.crash "build: ctx_state: no state provided"

let ctx_mkname ctx pp x =
   match ctx.state with
   | Some s -> S.MACHINESTATE (s, pp, x)
   | None -> Util.crash "build: ctx_mkname: no state provided"

(*
 * For a type (our type) that's enumerated, enumerate all the cases.
 *)
let tycases ctx ty =
   match ty with
   | M.BOOL -> None (* bools happen as SMT bools *)
   | M.CACHESTATE ->
        let cases =
           List.map proj_cachestate (Mipstools.all_cachestates_raw ())
        in
        Some (cases, MS.cachestate_bitwidth)
   | M.UINT5
   | M.SINT8
   | M.UINT8
   | M.SINT16
   | M.UINT16
   | M.SINT32
   | M.UINT32 -> None (* these are SMT bitvector types *)
(* XXX these should use all_*_raw *)
   | M.REGISTER ->
        let cases = List.map MS.int_of_reg (Mipstools.all_regs_raw ()) in
        Some (cases, MS.register_bitwidth)
   | M.CREGISTER ->
        let cases = List.map MS.int_of_creg (Mipstools.all_cregs_raw ()) in
        Some (cases, MS.register_bitwidth)
   | M.FREGISTER ->
        let cases = List.map MS.int_of_freg (Mipstools.all_fregs_raw ()) in
        Some (cases, MS.register_bitwidth)
   | M.FCREGISTER ->
        let cases = List.map MS.int_of_fcreg (Mipstools.all_fcregs_raw ()) in
        Some (cases, MS.register_bitwidth)
   | M.HIREGISTER -> Util.crash "Enumerating $hi register"
   | M.LOREGISTER -> Util.crash "Enumerating $lo register"
   | M.GOLEM ->
        let cases = List.map proj_golem (Mipstools.all_golems_raw ()) in
        Some (cases, MS.golem_bitwidth)
   | M.WHICHCACHE ->
        let cases = List.map proj_cache (Mipstools.all_caches_raw ()) in
        Some (cases, MS.whichcache_bitwidth)
   | M.WHICHCACHEENTRYID ->
        let cases =
           List.map proj_cacheentryid (Mipstools.all_cacheentryids_raw ())
        in
        Some (cases, MS.whichcacheentryid_bitwidth)
   | M.WHICHCACHEENTRY ->
        let cases =
           List.map proj_cacheentry (Mipstools.all_cacheentries_raw ())
        in
        Some (cases, MS.whichcacheentry_bitwidth)
   | M.PROGRAMPOINT ->
        Some (List.map proj_pp ctx.allpps, programpoint_bitwidth)
   | M.INSN ->
        (* this could be made to work, but painfully and it's not worth it *)
        Util.crash "Explicit forall/exists of instruction type"
   | M.MAP _ ->
        (* uh no *)
        Util.crash "Explicit forall/exists of map/array type"

let constraints'type allpps name ty =
   let x = S.READVAR name in
   let bounds lb ub = [
      S.MINTBOP (SLT, lb, x);
      S.MINTBOP (SLT, x, ub);
   ] in
   let enum elems w =
      let mk elem =
         S.EQ (x, S.xintval elem w)
      in
      (* XXX this seems to need to be backwards (see other case above) *)
      [S.or_list (List.map mk (List.rev elems))]
   in
   match ty with
   | M.BOOL -> []
   | M.CACHESTATE ->
       let cases = List.map MS.int_of_cachestate (MT.all_cachestates_raw ()) in
       enum cases MS.cachestate_bitwidth
   | M.UINT5 -> bounds (S.mintval (-1)) (S.mintval 32)
   | M.SINT8 -> bounds (S.mintval (-129)) (S.mintval 128)
   | M.UINT8 -> bounds (S.mintval (-1)) (S.mintval 256)
   | M.SINT16 -> bounds (S.mintval (-32769)) (S.mintval 32768)
   | M.UINT16 -> bounds (S.mintval (-1)) (S.mintval 65536)
   | M.SINT32 ->
        if Modes.mints_are_bitvectors then []
        else bounds (S.mintval (-2147483649)) (S.mintval 2147483648)
   | M.UINT32 ->
        if Modes.mints_are_bitvectors then []
        else bounds (S.mintval (-1)) (S.mintval 4294967296)
   | M.REGISTER -> enum (List.map MS.int_of_reg (MT.all_regs_raw ())) 5
   | M.CREGISTER -> enum (List.map MS.int_of_creg (MT.all_cregs_raw ())) 5
   | M.FREGISTER -> enum (List.map MS.int_of_freg (MT.all_fregs_raw ())) 5
   | M.FCREGISTER -> enum (List.map MS.int_of_fcreg (MT.all_fcregs_raw ())) 5
   | M.HIREGISTER -> Util.crash "build: constraints'type: HIREGISTER"
   | M.LOREGISTER -> Util.crash "build: constraints'type: LOREGISTER"
   | M.GOLEM ->
        let cases = List.map MS.int_of_golem (MT.all_golems_raw ()) in
        enum cases MS.golem_bitwidth
   | M.WHICHCACHE ->
        let cases = List.map MS.int_of_cache (MT.all_caches_raw ()) in
        enum cases MS.whichcache_bitwidth
   | M.WHICHCACHEENTRYID ->
        let cases =
           List.map MS.int_of_cacheentryid (MT.all_cacheentryids_raw ())
        in
        enum cases MS.whichcacheentryid_bitwidth
   | M.WHICHCACHEENTRY ->
        let cases =
           List.map MS.int_of_cacheentry (MT.all_cacheentries_raw ())
        in
        enum cases MS.whichcacheentry_bitwidth
   | M.PROGRAMPOINT ->
        enum (List.map Ppoint.to_smt allpps) programpoint_bitwidth
   | M.INSN -> [] (* already generated *)
   | M.MAP _ -> [] (* not needed *)

let tycases_implies name cases w e' =
   let once n =
      S.IMPLIES (S.EQ (S.READVAR name, S.xintval n w), e')
   in
   List.map once cases

let mk'var ctx name =
   match name with
   | M.PLAINVAR x -> S.PLAINVAR x
   | M.CONTROLVAR (k, x) -> S.CONTROLVAR (k, x)
   | M.STATEVAR (k, x) -> S.STATEVAR (ctx_state ctx, k, x)

let rec mk'expr ctx e =
   let plain e' = ([], [], e') in
   match e with
   | M.TRUE -> plain S.TRUE
   | M.FALSE -> plain S.FALSE
   | M.CACHESTATEVALUE cs ->
        plain (S.xintval (MS.int_of_cachestate cs) (MS.cachestate_bitwidth))
   | M.INTVALUE (x, _ty) -> plain (S.mintval x)
   | M.REGVALUE r -> plain (S.xintval (MS.int_of_reg r) 5)
   | M.CREGVALUE cr -> plain (S.xintval (MS.int_of_creg cr) 5)
   | M.FREGVALUE fr -> plain (S.xintval (MS.int_of_freg fr) 5)
   | M.FCREGVALUE fcr -> plain (S.xintval (MS.int_of_fcreg fcr) 5)
   | M.HIREGVALUE -> Util.crash "build: mk'expr: materializing HIREGVALUE"
   | M.LOREGVALUE -> Util.crash "build: mk'expr: materializing LOREGVALUE"
   | M.GOLEMVALUE g -> plain (S.xintval (MS.int_of_golem g) MS.golem_bitwidth)
   | M.WHICHCACHEVALUE c ->
        plain (S.xintval (MS.int_of_cache c) MS.whichcache_bitwidth)
   | M.WHICHCACHEENTRYIDVALUE c ->
        let w = MS.whichcacheentryid_bitwidth in
        plain (S.xintval (MS.int_of_cacheentryid c) w)
   | M.WHICHCACHEENTRYVALUE (c, ce) ->
        let w = MS.whichcacheentry_bitwidth in
        plain (S.xintval (MS.int_of_cacheentry (c, ce)) w)
   | M.PPVALUE pp ->
        plain (S.xintval (Ppoint.to_smt pp) programpoint_bitwidth)
   | M.INSNVALUE insn ->
        plain (S.xintval (Insns.insncode insn) Insns.insn_bitwidth)
   | M.READSTATE (M.REGVALUE MS.Z0, _pp, M.Z_ZERO) ->
        (* Is this ever actually reached? *)
	plain (S.mintval 0)
   | M.READSTATE (M.REGVALUE MS.Z0, _pp, M.Z_ANY) ->
	(*
	 * Upstream is responsible for removing logic that tries
	 * to explicitly constrain the value of z0 ($0) after an
	 * instruction; values written to z0 by instructions are
	 * discarded.
	 *)
	Util.crash "build: mk'expr: concrete $0 has arbitrary value"
   | M.READSTATE (M.REGVALUE r, pp, _zs) ->
        plain (readreg (ctx_mkname ctx pp) r)
   | M.READSTATE (M.CREGVALUE cr, pp, _zs) ->
        plain (readcreg (ctx_mkname ctx pp) cr)
   | M.READSTATE (M.FREGVALUE fr, pp, _zs) ->
        plain (readfreg (ctx_mkname ctx pp) fr)
   | M.READSTATE (M.FCREGVALUE fcr, pp, _zs) ->
        plain (readfcreg (ctx_mkname ctx pp) fcr)
   | M.READSTATE (M.HIREGVALUE, pp, _zs) ->
        plain (readhi (ctx_mkname ctx pp))
   | M.READSTATE (M.LOREGVALUE, pp, _zs) ->
        plain (readlo (ctx_mkname ctx pp))
   | M.READSTATE (M.WHICHCACHEENTRYVALUE (c, ce), pp, _zs) ->
        plain (readcacheentry (ctx_mkname ctx pp) (c, ce))
   | M.READSTATE (_se, _pp, _zs) -> begin
(*
        match Modes.smt_register_mode with
        | Modes.REGS_VIA_VARS ->
*)
             Util.crash "build: mk'expr: leftover symbolic readstate"
(*
        | Modes.REGS_VIA_ARRAYEQS
        | Modes.REGS_VIA_ARRAYSTORES ->
             let (bind, props, se') = mk'expr ctx se in
             let arr = getarray (ctx_mkname ctx pp) se in
             (bind, props, S.READARRAY (arr, se'))
*)
     end
   | M.READGOLEM (M.GOLEMVALUE g) ->
        plain (readgolem (ctx_state ctx) g)
   | M.READGOLEM _ge -> begin
(*
        match Modes.smt_register_mode with
        | Modes.REGS_VIA_VARS ->
*)
             Util.crash "build: mk'expr: leftover symbolic readgolem"        
(*
        | Modes.REGS_VIA_ARRAYEQS
        | Modes.REGS_VIA_ARRAYSTORES ->
             let (bind, props, ge') = mk'expr ctx ge in
             let arr = S.GOLEMVAR (ctx_state ctx, golemarrayname) in
             (bind, props, S.READARRAY (arr, ge'))
*)
     end
(*
   | M.READALLSTATE (M.REGISTER, pp) ->
        plain (S.READVAR (S.MACHINESTATE (ctx_state ctx, pp, regarrayname)))
   | M.READALLSTATE (M.CREGISTER, pp) ->
        plain (S.READVAR (S.MACHINESTATE (ctx_state ctx, pp, cregarrayname)))
   | M.READALLSTATE (M.FREGISTER, pp) ->
        plain (S.READVAR (S.MACHINESTATE (ctx_state ctx, pp, fregarrayname)))
   | M.READALLSTATE (M.FCREGISTER, pp) ->
        plain (S.READVAR (S.MACHINESTATE (ctx_state ctx, pp, fcregarrayname)))
   | M.READALLSTATE (M.GOLEM, _pp) ->
        plain (S.READVAR (S.GOLEMVAR (ctx_state ctx, golemarrayname)))
   | M.READALLSTATE (_, _) ->
        Util.crash "build: mk'expr: illegal READALLSTATE"
   | M.UPDATEMAP (m, k, v) ->
        let (bind'm, props'm, m') = mk'expr ctx m in
        let (bind'k, props'k, k') = mk'expr ctx k in
        let (bind'v, props'v, v') = mk'expr ctx v in
        let m'' = match m' with
           | S.READVAR mx -> mx
           | _ -> Util.crash "build: mk'expr: updatemap: bad map"
        in
        let e' = S.UPDATEARRAY (m'', k', v') in
        (bind'm @ bind'k @ bind'v, props'm @ props'k @ props'v, e')
*)
   | M.READVAR (name, _ty) -> plain (S.READVAR (mk'var ctx name))
   | M.GOLEMIZE (g, e1) ->
        let (gbind, gprops, g') = mk'expr ctx g in
        let (ebind, eprops, e1') = mk'expr ctx e1 in
        let eq1 = S.minteq e1' (S.mintval 1234) in
        (gbind @ ebind, gprops @ eprops, S.AND (eq1, g'))
   | M.FORALL (name, ty, e1) -> begin
        let (ebind, eprops, e1') = mk'expr ctx e1 in
        match tycases ctx ty with
        | None ->
	     (* XXX, should figure out how to handle the scoping *)
	     (ebind, eprops, S.FORALL (mk'var ctx name, mk'typename ty, e1'))
        | Some (cases, w) ->
             let name' = mk'var ctx name in
             let binding = (name', mk'typename ty, ty) in
	     let tprops = constraints'type ctx.allpps name' ty in
	     let e' = S.and_list (tycases_implies name' cases w e1') in
	     (binding :: ebind, tprops @ eprops, e')
     end
   | M.EXISTS (name, ty, e1) -> begin
        let (ebind, eprops, e1') = mk'expr ctx e1 in
        match tycases ctx ty with
        | None ->
	     (* XXX, should figure out how to handle the scoping *)
	     (ebind, eprops, S.EXISTS (mk'var ctx name, mk'typename ty, e1'))
        | Some (cases, w) ->
             let name' = mk'var ctx name in
             let binding = (name', mk'typename ty, ty) in
	     let tprops = constraints'type ctx.allpps name' ty in
	     let e' = S.or_list (tycases_implies name' cases w e1') in
	     (binding :: ebind, tprops @ eprops, e')
     end
   | M.IMPLIES (e1, e2) ->
        let (bind'e1, props'e1, e1') = mk'expr ctx e1 in
        let (bind'e2, props'e2, e2') = mk'expr ctx e2 in
        (* the props for e2 depend on e1 *)
        let props'e2 =
	   (* we should maybe make a variable for e1' instead of duplicating *)
           List.map (fun p -> S.IMPLIES (e1', p)) props'e2
        in
        (bind'e1 @ bind'e2, props'e1 @ props'e2, S.IMPLIES (e1', e2'))
   | M.LET (name, ty, e1, e2) ->
	(* XXX, should figure out how to handle the scoping *)
        let name' = mk'var ctx name in
        let ty' = mk'typename ty in
        let (bind'e1, props'e1, e1') = mk'expr ctx e1 in
        let (bind'e2, props'e2, e2') = mk'expr ctx e2 in
        (* the props for e2 might reference the var *)
        let props'e2 =
	   (* we ... have a variable for e1 ... but it isn't global *)
           List.map (fun p -> S.LET (name', ty', e1', p)) props'e2
        in
        (bind'e1 @ bind'e2, props'e1 @ props'e2, S.LET (mk'var ctx name, ty', e1', e2'))
   | M.IF (c, t, f) ->
        let (bind'c, props'c, c') = mk'expr ctx c in
        let (bind't, props't, t') = mk'expr ctx t in
        let (bind'f, props'f, f') = mk'expr ctx f in
        (* the props need to depend on c *)
        let props't =
	   (* we should maybe make a variable for c' instead of duplicating *)
           List.map (fun p -> S.IMPLIES (c', p)) props't
        in
        let props'f =
           List.map (fun p -> S.IMPLIES (S.NOT c', p)) props'f
        in
        let bind = bind'c @ bind't @ bind'f in
        let props = props'c @ props't @ props'f in
        (bind, props, S.IF (c', t', f'))
   | M.LOGNOT e1 ->
        let (bind'e1, props'e1, e1') = mk'expr ctx e1 in
        (bind'e1, props'e1, S.NOT e1')
   | M.LOGAND (e1, e2) ->
        let (bind'e1, props'e1, e1') = mk'expr ctx e1 in
        let (bind'e2, props'e2, e2') = mk'expr ctx e2 in
        (bind'e1 @ bind'e2, props'e1 @ props'e2, S.AND (e1', e2'))
   | M.LOGOR (e1, e2) ->
        let (bind'e1, props'e1, e1') = mk'expr ctx e1 in
        let (bind'e2, props'e2, e2') = mk'expr ctx e2 in
        (bind'e1 @ bind'e2, props'e1 @ props'e2, S.OR (e1', e2'))
   | M.LOGXOR (e1, e2) ->
        let (bind'e1, props'e1, e1') = mk'expr ctx e1 in
        let (bind'e2, props'e2, e2') = mk'expr ctx e2 in
        (bind'e1 @ bind'e2, props'e1 @ props'e2, S.XOR (e1', e2'))
   | M.LOGEQ (e1, e2) ->
        let (bind'e1, props'e1, e1') = mk'expr ctx e1 in
        let (bind'e2, props'e2, e2') = mk'expr ctx e2 in
        (bind'e1 @ bind'e2, props'e1 @ props'e2, S.EQ (e1', e2'))
   | M.MINTUOP (op, e1) ->
        let (bind'e1, props'e1, e1') = mk'expr ctx e1 in
        (bind'e1, props'e1, S.MINTUOP (op, e1'))
   | M.MINTBOP (op, e1, e2) ->
        let (bind'e1, props'e1, e1') = mk'expr ctx e1 in
        let (bind'e2, props'e2, e2') = mk'expr ctx e2 in
        (bind'e1 @ bind'e2, props'e1 @ props'e2, S.MINTBOP (op, e1', e2'))

let mk'logic ctx (factnum, (factdesc, fact)) =
   let (bindings, constraints, main) = mk'expr ctx fact in

(*
   let bindings = bindings'expr ctx fact in
   let constraints = constraints'expr ctx fact in
   let main = mk'expr ctx fact in
*)

   (*
    * bindings[] returns the logic-level type in case we need to emit
    * value constraints based on that. We haven't been and the values
    * produced should always be constrained by other things that are
    * constrained, but in case that changes we have what we need.
    * (Except for the list of program points...)
    *)
   let bindings =
      let once (name, ty, _mty) = S.BIND (name, ty) in
      List.map once bindings
   in

   (* XXX kinda gross that we need to do this *)
   let bindings = List.sort_uniq jackass_compare_decl bindings in
   (* XXX even more so this *)
   let constraints = List.sort_uniq jackass_compare_expr constraints in

   (*let facttxt = Printlogic.print "#" fact in*)
   let factstr =
      if factnum < 0 then "Fact"
      else "Fact " ^ string_of_int factnum
   in
   [S.COMMENT (factstr ^ ": " ^ factdesc)] @
   (*[S.COMMENT ("Upstream formula:\n" ^ facttxt)] @*)
   [S.COMMENT "Bindings:"] @
   bindings @
   [S.COMMENT "Constraints:"] @
   [S.ASSERT (S.and_list constraints)] @
   [S.COMMENT "Main assertion:"] @
   [S.ASSERT main] @
   [S.COMMENT "End fact."]

let mk'binding (name, mty) =
   S.BIND (name, mk'typename mty)

(*
let mk'states ctx states =
   let onestate (_i, state) =
      let onebindings (_name, e) = bindings'expr ctx e in
      let oneconstraints (_name, e) = constraints'expr ctx e in
      let oneval (name, e) = S.EQ (S.READVAR name, mk'expr ctx e) in
      let bindings = List.concat (List.map onebindings state) in
      let constraints = List.concat (List.map oneconstraints state) in
      let vals = List.map oneval state in
      (bindings, S.and_list (constraints @ vals))
   in
   let (bindings, props) = Util.unzip (List.map onestate states) in
   S.COMMENT "...bindings:" ::
   List.concat bindings @ [
      S.COMMENT "...assertion:";
      S.ASSERT (S.or_list props);
   ]
*)


(**************************************************************)
(* concretize *)

let concretize'any intvals mk elem =
   match elem with
   | CONC _ -> elem
   | SYM (k, x) ->
        let x' = S.CONTROLVAR (k, x) in
        try
           let bk = S.VarMap.find x' intvals in
           CONC (mk (Big_int.int_of_big_int bk))
        with Not_found ->
           let kx' = string_of_symi (k, x) in
           Util.crash ("build: concretize: no value for " ^ kx')

let concretize'reg intvals r = concretize'any intvals MS.reg_of_int r
let concretize'creg intvals cr = concretize'any intvals MS.creg_of_int cr
let concretize'freg intvals fr = concretize'any intvals MS.freg_of_int fr
let concretize'fcreg intvals fcr = concretize'any intvals MS.fcreg_of_int fcr
let concretize'whichcache intvals c = concretize'any intvals MS.cache_of_int c

let concretize'imm intvals ty imm =
   (*
    * The machine int values that come back from z3 are always
    * converted to big_int as unsigned 32-bit values, because that's
    * all any of the plumbing knows about. However, the value should
    * be within the proper bounds for the requested type because a
    * constraint to that effect was included. So for unsigned values
    * we can just crosscheck the bounds but for unsigned values we need
    * to offset by 2^32.
    *)
   let fixup k =
      let testbounds lb k ub what =
         if Big_int.gt_big_int (Big_int.big_int_of_int lb) k then
            Util.crash ("build: concretize'imm: " ^ what ^ " out of range")
         ;
         if Big_int.gt_big_int k (Big_int.big_int_of_int ub) then
            Util.crash ("build: concretize'imm: " ^ what ^ " out of range")
         ;
      in
      match ty with
      | M.UINT5 ->
           testbounds 0 k 31 "shift count immediate"; k
      | M.UINT16 ->
           testbounds 0 k 65535 "uint16 immediate"; k
      | M.SINT16 ->
           let k =
              (* XXX this will blow up on 32-bit ocaml builds *)
              if Big_int.gt_big_int k (Big_int.big_int_of_int 0x7fffffff) then
                 Big_int.sub_big_int k (Big_int.big_int_of_int 0x100000000)
              else k
           in
           testbounds (-32768) k 32767 "sint16 immediate"; k
      | _ -> Util.crash "build: concretize'imm: unexpected type"
   in
   match imm with
   | CONC k -> CONC k
   | SYM (k, x) ->
        let x' = S.CONTROLVAR (k, x) in
        try
           let bk = S.VarMap.find x' intvals in
           CONC (Big_int.int_of_big_int (fixup bk))
        with Not_found ->
           let kx' = string_of_symi (k, x) in
           Util.crash ("build: concretize: no value for " ^ kx')

let concretize'extcodelabel intvals l =
   let mk _ =
      (* XXX notyet *)
      "somecodelabel"
   in
   concretize'any intvals mk l

let concretize'extdatalabel intvals l =
   let mk _ =
      (* XXX notyet *)
      "somedatalabel"
   in
   concretize'any intvals mk l

let concretize'localcodelabel intvals l =
   concretize'any intvals Ppoint.from_smt l

let concretize intvals insn =
   let c'reg r = concretize'reg intvals r in
   let c'creg cr = concretize'creg intvals cr in
   let c'freg fr = concretize'freg intvals fr in
   let c'fcreg fcr = concretize'fcreg intvals fcr in
   let c'whichcache c = concretize'whichcache intvals c in
   let c'imm ty imm = concretize'imm intvals ty imm in
   let c'xc xc = concretize'extcodelabel intvals xc in
   let c'xd xd = concretize'extdatalabel intvals xd in
   let c'lc lc = concretize'localcodelabel intvals lc in
   let c'mem (M.MEMORY (r, imm)) = M.MEMORY (c'reg r, c'imm M.SINT16 imm) in

   match insn with
   (* arithmetic *)
   | M.ADDU (rd, rs, rt) -> M.ADDU (c'reg rd, c'reg rs, c'reg rt)
   | M.ADDIU (rd, rs, imm) -> M.ADDIU (c'reg rd, c'reg rs, c'imm M.SINT16 imm)
   | M.DIV (rs, rt) -> M.DIV (c'reg rs, c'reg rt)
   | M.DIVU (rs, rt) -> M.DIVU (c'reg rs, c'reg rt)
   | M.LUI (rd, imm) -> M.LUI (c'reg rd, c'imm M.UINT16 imm)
   | M.MULT (rs, rt) -> M.MULT (c'reg rs, c'reg rt)
   | M.MULTU (rs, rt) -> M.MULTU (c'reg rs, c'reg rt)
   | M.SUBU (rd, rs, rt) -> M.SUBU (c'reg rd, c'reg rs, c'reg rt)
   | M.LUI_extcode_hi (rd, l) ->
        M.LUI_extcode_hi (c'reg rd, c'xc l)
   | M.ADDIU_extcode_lo (rd, rs, l) ->
        M.ADDIU_extcode_lo (c'reg rd, c'reg rs, c'xc l)
   | M.LUI_extdata_hi (rd, l) ->
        M.LUI_extdata_hi (c'reg rd, c'xd l)
   | M.ADDIU_extdata_lo (rd, rs, l) ->
        M.ADDIU_extdata_lo (c'reg rd, c'reg rs, c'xd l)
   | M.LUI_localcode_hi (rd, l) ->
        M.LUI_localcode_hi (c'reg rd, c'lc l)
   | M.ADDIU_localcode_lo (rd, rs, l) ->
        M.ADDIU_localcode_lo (c'reg rd, c'reg rs, c'lc l)
   (* comparisons *)
   | M.SLT (rd, rs, rt) -> M.SLT (c'reg rd, c'reg rs, c'reg rt)
   | M.SLTI (rd, rs, imm) -> M.SLTI (c'reg rd, c'reg rs, c'imm M.SINT16 imm)
   | M.SLTU (rd, rs, rt) -> M.SLTU (c'reg rd, c'reg rs, c'reg rt)
   | M.SLTIU (rd, rs, imm) -> M.SLTIU (c'reg rd, c'reg rs, c'imm M.SINT16 imm)
   (* bitwise logic *)
   | M.AND (rd, rs, rt) -> M.AND (c'reg rd, c'reg rs, c'reg rt)
   | M.ANDI (rd, rs, imm) -> M.ANDI (c'reg rd, c'reg rs, c'imm M.UINT16 imm)
   | M.NOR (rd, rs, rt) -> M.NOR (c'reg rd, c'reg rs, c'reg rt)
   | M.OR (rd, rs, rt) -> M.OR (c'reg rd, c'reg rs, c'reg rt)
   | M.ORI (rd, rs, imm) -> M.ORI (c'reg rd, c'reg rs, c'imm M.UINT16 imm)
   | M.XOR (rd, rs, rt) -> M.XOR (c'reg rd, c'reg rs, c'reg rt)
   | M.XORI (rd, rs, imm) -> M.XORI (c'reg rd, c'reg rs, c'imm M.UINT16 imm)
   (* shifts *)
   | M.SLL (rd, rs, imm) -> M.SLL (c'reg rd, c'reg rs, c'imm M.UINT5 imm)
   | M.SLLV (rd, rs, rt) -> M.SLLV (c'reg rd, c'reg rs, c'reg rt)
   | M.SRA (rd, rs, imm) -> M.SRA (c'reg rd, c'reg rs, c'imm M.UINT5 imm)
   | M.SRAV (rd, rs, rt) -> M.SRAV (c'reg rd, c'reg rs, c'reg rt)
   | M.SRL (rd, rs, imm) -> M.SRL (c'reg rd, c'reg rs, c'imm M.UINT5 imm)
   | M.SRLV (rd, rs, rt) -> M.SRLV (c'reg rd, c'reg rs, c'reg rt)
   (* transfers *)
   | M.B l -> M.B (c'lc l)
   | M.BEQ (rd, rs, l) -> M.BEQ (c'reg rd, c'reg rs, c'lc l)
   | M.BNE (rd, rs, l) -> M.BNE (c'reg rd, c'reg rs, c'lc l)
   | M.J l -> M.J (c'xc l)
   | M.JAL l -> M.JAL (c'xc l)
   | M.JALR rs -> M.JALR (c'reg rs)
   | M.JR rs -> M.JR (c'reg rs)
   (* moves *)
   | M.CFC1 (rd, fcs) -> M.CFC1 (c'reg rd, c'fcreg fcs)
   | M.CTC1 (rd, fcs) -> M.CTC1 (c'reg rd, c'fcreg fcs)
   | M.MFC0 (rd, cs) -> M.MFC0 (c'reg rd, c'creg cs)
   | M.MFC1 (rd, fs) -> M.MFC1 (c'reg rd, c'freg fs)
   | M.MFHI rd -> M.MFHI (c'reg rd)
   | M.MFLO rd -> M.MFLO (c'reg rd)
   | M.MTC0 (rd, cs) -> M.MTC0 (c'reg rd, c'creg cs)
   | M.MTC1 (rd, fs) -> M.MTC1 (c'reg rd, c'freg fs)
   | M.MTHI rs -> M.MTHI (c'reg rs)
   | M.MTLO rs -> M.MTLO (c'reg rs)
   | M.MOVN (rd, rs, rt) -> M.MOVN (c'reg rd, c'reg rs, c'reg rt)
   | M.MOVZ (rd, rs, rt) -> M.MOVZ (c'reg rd, c'reg rs, c'reg rt)
   (* loads *)
   | M.LB (rd, m) -> M.LB (c'reg rd, c'mem m)
   | M.LBU (rd, m) -> M.LBU (c'reg rd, c'mem m)
   | M.LDC1 (fd, m) -> M.LDC1 (c'freg fd, c'mem m)
   | M.LH (rd, m) -> M.LH (c'reg rd, c'mem m)
   | M.LHU (rd, m) -> M.LHU (c'reg rd, c'mem m)
   | M.LW (rd, m) -> M.LW (c'reg rd, c'mem m)
   | M.LWC1 (fd, m) -> M.LWC1 (c'freg fd, c'mem m)
   (* stores *)
   | M.SB (rd, m) -> M.SB (c'reg rd, c'mem m)
   | M.SDC1 (fd, m) -> M.SDC1 (c'freg fd, c'mem m)
   | M.SH (rd, m) -> M.SH (c'reg rd, c'mem m)
   | M.SW (rd, m) -> M.SW (c'reg rd, c'mem m)
   | M.SWC1 (fd, m) -> M.SWC1 (c'freg fd, c'mem m)
   (* system control *)
   | M.CACHE_INDEX_WBINV (c, m, g) ->
	M.CACHE_INDEX_WBINV (c'whichcache c, c'mem m, g)
   | M.CACHE_VADDR_WB (c, m, g) ->
	M.CACHE_VADDR_WB (c'whichcache c, c'mem m, g)
   | M.CACHE_VADDR_WBINV (c, m, g) ->
	M.CACHE_VADDR_WBINV (c'whichcache c, c'mem m, g)
   | M.CACHE_VADDR_INV (c, m, g) ->
        M.CACHE_VADDR_INV (c'whichcache c, c'mem m, g)
   | M.TLBP -> M.TLBP
   | M.TLBR -> M.TLBR
   | M.TLBWI -> M.TLBWI
   | M.TLBWR -> M.TLBWR


(**************************************************************)
(* model extraction *)

let extract_boolarray map default =
   let conv e =
      match e with
      | S.TRUE -> true
      | S.FALSE -> false
      | _ -> Util.crash "build: extract_boolarray: invalid value"
   in
   (Types.BigIntMap.map conv map, conv default)

let extract_mintarray map default =
   let conv e =
      match e with
      | S.MINTVAL k -> k
      | _ -> Util.crash "build: extract_mintarray: invalid value"
   in
   (Types.BigIntMap.map conv map, conv default)

(*
 * Produce (boolvals, intvals), two maps from variable names to
 * values, for the boolean and integer variables respectively.
 * Also boolarrayvals and intarrayvals...
 *
 * The model is a list of Smt.assignment; the Z3 stuff should
 * already have been unwrapped by Solver.
 *)
let extract_from_model model = 
   let add = S.VarMap.add in
   let empty = S.VarMap.empty in
   let addone (bvs, ivs, bavs, iavs) a =
      match a with
      | S.ASSIGN (x, S.BOOL, S.TRUE) ->
           (add x true bvs, ivs, bavs, iavs)
      | S.ASSIGN (x, S.BOOL, S.FALSE) ->
           (add x false bvs, ivs, bavs, iavs)
        (* XXX should really distinguish xints and mints *)
      | S.ASSIGN (x, S.XINT _, S.XINTVAL (k, _)) ->
           (bvs, add x k ivs, bavs, iavs)
      | S.ASSIGN (x, S.MINT, S.MINTVAL k) ->
           (bvs, add x k ivs, bavs, iavs)
      | S.ASSIGN (x, _, _) ->
           let x' = S.string_of_varname x in
           Util.crash ("build: unguess: invalid scalar result for " ^ x')
      | S.ASSIGNARRAY (x, S.ARRAY (_, S.BOOL), map, default) ->
           let (map', default') = extract_boolarray map default in
           (bvs, ivs, add x (map', default') bavs, iavs)
      | S.ASSIGNARRAY (x, S.ARRAY (_, S.MINT), map, default) ->
           let (map', default') = extract_mintarray map default in
           (bvs, ivs, bavs, add x (map', default') iavs)
      | S.ASSIGNARRAY (x, _, _, _) ->
           let x' = S.string_of_varname x in
           Util.crash ("build: unguess: invalid array result for " ^ x')
   in
   List.fold_left addone (empty, empty, empty, empty) model


(**************************************************************)
(* external interface *)

(*
 * Given a variable environment (map from identifiers to types)
 * produce the corresponding SMT-level variable bindings as a
 * list of declarations.
 *
 * Skips the state-specific ones.
 *)
let build'environment label env =
   let bindings =
      let once (name, ty) =
         match name with
         | M.PLAINVAR x -> [(S.PLAINVAR x, ty)]
         | M.CONTROLVAR (k, x) -> [(S.CONTROLVAR (k, x), ty)]
         | M.STATEVAR _name -> []
      in
      List.concat (List.map once (M.VarMap.bindings env))
   in
   S.COMMENT label ::
   List.map mk'binding bindings

(*
 * Does the state-specific ones only.
 *)
let build'more'environment label env allstates =
   let bindings =
      let once (name, ty) =
         match name with
         | M.PLAINVAR _ -> []
         | M.CONTROLVAR _ -> []
         | M.STATEVAR (k, x) ->
              let onceonce s =
                 (S.STATEVAR (s, k, x), ty)
              in
              List.map onceonce allstates
      in
      List.concat (List.map once (M.VarMap.bindings env))
   in
   S.COMMENT label ::
   List.map mk'binding bindings


(*
 * Given a variable environment and list of program points, produce
 * the SMT-level assertions needed to enforce the declared types of
 * the variables.
 *
 * Note: we need the list of program points in case one of the
 * variables is a program point variable.
 *
 * Skips the state-specific ones.
 *)
let build'envconstraints label env allpps =
   let bindings =
      let once (name, ty) =
         match name with
         | M.PLAINVAR x -> [(S.PLAINVAR x, ty)]
         | M.CONTROLVAR (k, x) -> [(S.CONTROLVAR (k, x), ty)]
         | M.STATEVAR _ -> []
      in
      List.concat (List.map once (M.VarMap.bindings env))
   in
   let logicconstraints' =
      let once (name, ty) =
         let e = Smt.and_list (constraints'type allpps name ty) in
         Smt.ASSERT e
      in
      S.COMMENT label ::
      List.map once bindings
   in
   logicconstraints'

(*
 * Does the state-specific ones only.
 * XXX share more of the logic
 *)
let build'more'envconstraints label env allstates allpps =
   let bindings =
      let once (name, ty) =
         match name with
         | M.PLAINVAR _ -> []
         | M.CONTROLVAR _ -> []
         | M.STATEVAR (k, x) ->
              let onceonce s =
                 (S.STATEVAR (s, k, x), ty)
              in
              List.map onceonce allstates 
      in
      List.concat (List.map once (M.VarMap.bindings env))
   in
   let logicconstraints' =
      let once (name, ty) =
         let e = Smt.and_list (constraints'type allpps name ty) in
         Smt.ASSERT e
      in
      S.COMMENT label ::
      List.map once bindings
   in
   logicconstraints'

(*
 * Build the SMT-level variable declarations for an execution,
 * where the start state of the execution is tagged by the
 * string STATESTR. Generates a set of machine state variables
 * for each program point in ALLPPS.
 *)
let build'execenv label state allpps =
   S.COMMENT label ::
   gen_state_bindings state allpps @ gen_golem_binding state @
   [S.COMMENT (label ^ " constraints")] @
   gen_state_constraints state allpps

let assert_different_execenv label state otherstates pp require_all =
   let a1 =
      S.ASSERT (gen_state_different state otherstates pp require_all)
   in
   let a2 =
      let n = List.length otherstates in
      if n > 0 then
         let state2 = List.nth otherstates (n-1) in
         [S.ASSERT (gen_golem_different state state2)]
      else []
   in
   S.COMMENT label :: a1 :: a2

(*
 * A "fact" is a pair (labelstring, expr)
 *
 * This is the form both the pre/postconditions and the symbolic
 * execution logic come in.
 *)

let build'fact state allpps fact =
   let ctx = { allpps; state; } in
   mk'logic ctx (-1, fact)

let build'facts state allpps facts =
   let ctx = { allpps; state; } in
   S.COMMENT (string_of_int (List.length facts) ^ " facts:") ::
   List.concat (List.map (mk'logic ctx) (Util.number facts))

(*
 * Build an smt model for a candidate instruction sequence, which is a
 * list of assignments to the logic variables that define it. This is
 * simpler than one instance of "states" because the assignments never
 * get mapped back to logic from smt.
 *)
let assert_assignment model : Smt.smt =
   let set a =
      match a with
      | S.ASSIGN (x, ty, e) ->
           let x' = S.READVAR x in
           let eq =
              match ty with
              | BOOL
              | XINT _ -> S.EQ (x', e)
              | MINT -> S.MINTBOP (I.EQ, x', e)
              | ARRAY _ -> Util.crash "build: assert_assignment: unexpected array"
           in
           S.ASSERT eq
      | S.ASSIGNARRAY (x, ty, map, default) ->
           let arr =
              match ty with
              | ARRAY (XINT kw, BOOL) ->
                   let (map', default') = extract_boolarray map default in
                   S.BOOLARRAYVAL (map', default', kw)
              | ARRAY (XINT kw, MINT) ->
                   let (map', default') = extract_mintarray map default in
                   S.MINTARRAYVAL (map', default', kw)
              | _ -> Util.crash "build: assert_assignment: bad array"
           in
           S.ASSERT (S.EQ (S.READVAR x, arr))
   in
   S.COMMENT "Model from guess:" ::
   List.map set model

(*
 * Discard any assertions about processor state found in a model for
 * guessing an instruction sequence.
 *)
let filter_candidate model =
   let keep a =
      let x =
         match a with
         | S.ASSIGN (x, _ty, _e) -> x
         | S.ASSIGNARRAY (x, _ty, _m, _d) -> x
      in
      (* String.sub x 0 4 = "insn" *)
(*
      match x.state, x.pp, x.golem, x.isvalue with
      | None, None, NOTGOLEM, NOTVALUE
             when String.sub x.corename 0 4 = "insn" -> true
      | _, _, _, _ -> false
*)
      match x with
      | CONTROLVAR _ -> true
      | _ -> false
   in
   List.filter keep model

(*
 * Fold an instruction sequence model back into an instruction sequence.
 *)
let unguess model symcode : Mips.code =
   let (_boolvals, intvals, _boolarrays, _intarrays) =
      extract_from_model model
   in

   let decode (pp, insn) =
      let cinsn = match insn with
         | CONC cinsn -> (concretize intvals cinsn)
         | SYM (i, sym) ->
              let sym' = S.CONTROLVAR (i, sym) in
              let (_, opset) = Insns.mkoperandset i in
              let code =
                 try S.VarMap.find sym' intvals
                 with Not_found ->
                    Util.crash ("build: unguess: no value for insn " ^ sym)
              in
              Insns.decode opset (Big_int.int_of_big_int code)
      in
      (pp, CONC (concretize intvals cinsn))
   in
   let (insns, postpp) = symcode in
   (List.map decode insns, postpp)

(*
 * Fold a counterexample model back into a start state.
 *)
let arbitrary_number = ref 2

let get_arbitrary_number () =
   let ret = !arbitrary_number in
   arbitrary_number := 1 + ret;
   Big_int.big_int_of_int ret

let unverify vstate nstate prepoint counterexample : S.expr =
   let (boolvals, intvals, boolarrayvals, intarrayvals) =
      extract_from_model counterexample
   in

   let prenames =
      gen_state_varnames vstate prepoint
      @ gen_golem_varnames vstate
   in

   let extract_bool name =
      try
         S.VarMap.find name boolvals
      with Not_found -> false
   in
   let extract_int name =
      try
         S.VarMap.find name intvals
      with Not_found ->
         (*
          * Completely unconstrained values apparently don't get
          * assignments in the model, so we need to allow this case
          * and make something up.
          *
          * 20181112: This used to just use 0. But instead I think
          * we'll use a distinct integer every time, starting with
          * 2, to avoid the various degenerate cases associated with
          * 0 and 1.
          *)
         get_arbitrary_number ()
   in
   let extract_boolarray name =
      try
         S.VarMap.find name boolarrayvals
      with Not_found ->
         (Types.BigIntMap.empty, false)
   in
   let extract_intarray name =
      try
         S.VarMap.find name intarrayvals
      with Not_found ->
         (*
          * XXX we should populate it using arbitrary_number... but we
          * don't have the info we need to do that here
          *)
         (Types.BigIntMap.empty, get_arbitrary_number ())
   in

   let once (name, ty) =
      let k = match ty with
         | S.BOOL -> S.boolval (extract_bool name)
         | S.XINT w -> S.XINTVAL (extract_int name, w)
         | S.MINT -> S.MINTVAL (extract_int name)
         | S.ARRAY (XINT kw, S.BOOL) ->
              let (m, d) = extract_boolarray name in S.BOOLARRAYVAL (m, d, kw)
         | S.ARRAY (XINT kw, S.MINT) ->
              let (m, d) = extract_intarray name in S.MINTARRAYVAL (m, d, kw)
         | _ -> Util.crash "build: unverify: unexpected array type"
      in
      let name' = change_state vstate nstate name in
      let readvar = S.READVAR name' in
      match ty with
      | S.BOOL
      | S.XINT _ -> S.xinteq readvar k
      | S.MINT -> S.minteq readvar k
      | S.ARRAY _ -> S.xinteq readvar k
   in
   let preprops = List.map once prenames in
   S.and_list preprops
