(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

open Symbolic
open Mipsstate
open Mips
module MT = Mipstools
module I = Insns
module E = Enabled

(**************************************************************)
(* register values *)

(*
 * Since we have taken care to ensure that the meaning of the various
 * register fields is uniform across all the instructions they appear
 * in (and irrelevant otherwise) we can bind all the values up front
 * and not per-instruction. This significantly reduces the amount of
 * logic emitted.
 *
 * Note: if for some reason (currently it's impossible, but it might
 * become possible in the future after some preprocessing step) an
 * instruction has a concrete register in some field in the symbolic
 * execution, it needs to use that register instead of the shared
 * value -- note that because that might _not_ be uniform across
 * instructions it's wrong to relate it to the instruction-independent
 * variables, which should be ignored instead. This should be done
 * by using get_one_regstate.
 *
 * Note that because this uses symbols directly from the opset, it
 * needs to take account of which registers are enabled -- it should
 * avoid e.g. producing FPU logic unless the FPU is enabled. This
 * requires duplicating the logic from insns.ml and isn't robust. (XXX)
 *
 * Extract the cache golem value here too even though it's not really
 * a register (e.g. its type is BOOL) and it complicates the logic
 * some; otherwise it would require a fair amount of cutpaste.
 *)

let read_regstate ppre _i opset ppost =
   let mkstatevar (k, symname) =
      STATEVAR (k, "val" ^ symname)
   in

   let binding sym t = (mkstatevar sym, t) in

   let readstatevar sym t = READVAR (mkstatevar sym, t) in
   let readfieldvar sym t = READVAR (CONTROLVAR sym, t) in
   let readstate pp zs e = READSTATE (e, pp, zs) in
   let readprestate = readstate ppre Z_ZERO in
   let readpoststate = readstate ppost Z_ANY in

   let props' eq t2 doreadstate sym t cases =
      let once (_, e) =
         let prem = eq_logic (readfieldvar sym t) e in
         let conc = eq (readstatevar sym t2) (doreadstate e) in
         IMPLIES (prem, conc)
      in
      and_list (List.map once cases)
   in
   let props = props' eq_mint UINT32 in
   let g_props = props' eq_logic BOOL in

   let extract sr =
      match sr with
      | SYM sym -> sym
      | CONC _ -> Util.crash "symexec: read_regstate: concrete register"
   in

   let r_cases_pre =
      List.map (fun r -> (REGISTER, REGVALUE r)) (MT.all_regs_raw ())
   in
   let r_cases_post =
      List.map (fun r -> (REGISTER, REGVALUE r)) (MT.nonz0_regs_raw ())
   in
   let c_cases =
      List.map (fun r -> (CREGISTER, CREGVALUE r)) (MT.all_cregs_raw ())
   in
   let f_cases =
      List.map (fun r -> (FREGISTER, FREGVALUE r)) (MT.all_fregs_raw ())
   in
   let fc_cases =
      List.map (fun r -> (FCREGISTER, FCREGVALUE r)) (MT.all_fcregs_raw ())
   in
   let g_cases =
      List.map (fun g -> (GOLEM, GOLEMVALUE g)) (MT.all_golems_raw ())
   in

   let generate ispost sym t cases =
      let doreadstate = if ispost then readpoststate else readprestate in
      let sym' = extract sym in
      let mapping = (CONTROLVAR sym', READVAR (mkstatevar sym', UINT32)) in
      ((binding sym' UINT32, props doreadstate sym' t cases), mapping)
   in
   let g_generate sym cases =
      let doread g = READGOLEM g in
      let mapping = (CONTROLVAR sym, READVAR (mkstatevar sym, BOOL)) in
      ((binding sym BOOL, g_props doread sym GOLEM cases), mapping)
   in

   (* Don't need a case for opset.mem since it uses opset.rt. *)
   let results = [
      (true,            generate true opset.I.rd REGISTER r_cases_post);
      (!E.use_insn_c0,  generate true opset.I.cd CREGISTER c_cases);
      (!E.use_insn_fpu, generate true opset.I.fd FREGISTER f_cases);
      (!E.use_insn_fpu, generate true opset.I.fcd FCREGISTER fc_cases);
      (true,            generate false opset.I.rs REGISTER r_cases_pre);
      (!E.use_insn_c0,  generate false opset.I.cs CREGISTER c_cases);
      (!E.use_insn_fpu, generate false opset.I.fs FREGISTER f_cases);
      (!E.use_insn_fpu, generate false opset.I.fcs FCREGISTER fc_cases);
      (true,            generate false opset.I.rt REGISTER r_cases_pre);
      (!E.use_cache,    g_generate opset.I.cachegolemsym g_cases);
   ] in
   let results =
      List.map (fun (_, r) -> r) (List.filter (fun (p, _) -> p) results)
   in
   let (results, mappings) = Util.unzip results in
   let (bindings, props) = Util.unzip results in
   let map =
      let doadd z (k, v) = VarMap.add k v z in
      List.fold_left doadd VarMap.empty mappings
   in
   (bindings, props, map)

(*
 * Get a register value using the variables produced by read_regstate.
 * (map should be the map returned from same.)
 *
 * Note that the expressions in the map are already associated with
 * the proper program point.
 *)
let get_one_regstate map ctor _t sr pp zs =
   match sr with
   | CONC r -> READSTATE (ctor r, pp, zs)
   | SYM sym ->
(*
        if Modes.registers_use_vars then
*)
           try
              VarMap.find (CONTROLVAR sym) map
           with Not_found ->
              Util.crash "symexec: get_one_regstate: symbol missing from map"
(*
        else
           match t with
           | GOLEM -> READGOLEM (READVAR (CONTROLVAR sym, t))
           | _ -> READSTATE (READVAR (CONTROLVAR sym, t), pp, zs)
*)


(**************************************************************)
(* cache values *)

(*
 * Same but for the cache state.
 *
 * The cache golem value can be read out just like a register and is
 * handled in the register logic above.
 *
 * The choice of cache *entry* depends on machine state, so what we do
 * is bind a variable to hold which cacheentryid is picked (blue or
 * green) and bind the rest up front in terms of that.  Then the
 * per-instruction logic needs to bind it. We know we only need one of
 * these per instruction so it's ok to give it a fixed name.
 *
 * The variables involved are:
 *    fieldvar:   CONTROLVAR whichcache
 *    cestatevar: STATEVAR ("val" ^ "cacheentryid")
 *    estatevar:  STATEVAR ("preval" ^ "cacheentry")    (or "postval")
 * where
 *    - fieldvar is from the opset and already bound
 *      type is WHICHCACHE
 *      it holds the (symbolic) choice of which cache to look at
 *      its value is not fixed
 *      we use its value to set cstatevar
 *    - we bind cestatevar
 *      type is BOOL (would be WHICHCACHEENTRYID, but the golem produces BOOL)
 *      it holds the concrete choice of which cacheentryid to look at
 *      the per-instruction code sets its value (using the golem etc.)
 *      we use its value to set estatevar
 *    - we bind estatevar
 *      type is CACHESTATE
 *      it holds the concrete value read from the cache entry
 *      we set its value using readstate and cases of fieldvar/cestatevar
 *      its value is used by get_one_cachestate
 *      there are two of it, one for the prestate and one for the poststate
 *)

let read_cachestate ppre i opset ppost =
   let cestatevar =
      STATEVAR (i, "val" ^ "cacheentryid")
   in
   let mk_estatevar ispost =
      let p = if ispost then "post" else "pre" in
      STATEVAR (i, p ^ "val" ^ "cacheentry")
   in

   let cebinding = (cestatevar, BOOL) in
   let ebinding ispost = (mk_estatevar ispost, CACHESTATE) in

   let read_fieldvar sym = READVAR (CONTROLVAR sym, WHICHCACHE) in
   let read_cestatevar = READVAR (cestatevar, BOOL) in
   let read_estatevar ispost = READVAR (mk_estatevar ispost, CACHESTATE) in

   let readstate ispost e =
      let pp = if ispost then ppost else ppre in
      let zs = if ispost then Z_ZERO else Z_ANY in
      READSTATE (e, pp, zs)
   in

   let mkprops sym cases =
      let once (c, ce) =
         let c' = WHICHCACHEVALUE c in
         let prem1 = eq_logic (read_fieldvar sym) c' in
         let ce'', prem2 =
            let ce'', ce''' =
               match ce with
               | CACHE_BLUE -> CACHE_GREEN, TRUE
               | CACHE_GREEN -> CACHE_BLUE, FALSE
            in
            ce'', eq_logic read_cestatevar ce'''
         in
         let prem = LOGAND (prem1, prem2) in
         let mkconc ispost =
            let e = WHICHCACHEENTRYVALUE (c, ce) in
            eq_logic (read_estatevar ispost) (readstate ispost e)
         in
         let conc1 = mkconc false in
         let conc2 = mkconc true in
         let e3 = WHICHCACHEENTRYVALUE (c, ce'') in
         let conc3 = eq_logic (readstate false e3) (readstate true e3) in
         IMPLIES (prem, LOGAND (conc1, LOGAND (conc2, conc3)))
      in
      and_list (List.map once cases)
   in

   let extract sr =
      match sr with
      | SYM sym -> sym
      | CONC _ -> Util.crash "symexec: read_cachestate: concrete cache"
   in

   let cases =
      let once (c, ce) = (c, ce) in
      List.map once (MT.all_cacheentries_raw ())
   in

   let generate () =
      let csym = opset.I.whichcache in
      let csym' = extract csym in
      mkprops csym' cases
   in

   let vars = (cestatevar, mk_estatevar false, mk_estatevar true) in

   if !E.use_cache then
      let bindings = [ cebinding; ebinding false; ebinding true; ] in
      let props = [ generate (); ] in
      (bindings, props, vars)
   else
      ([], [], vars)


(**************************************************************)
(* main symbolic execution *)

(*
let type_of_stateelement elt =
   match elt with
   | REG _ -> REGISTER
   | CREG _ -> CREGISTER
   | FREG _ -> FREGISTER
   | FCREG _ -> FCREGISTER
   | HI -> HIREGISTER
   | LO -> HIREGISTER
   | CACHEENTRY _ -> WHICHCACHEENTRY
*)

let expr_of_stateelement elt =
   match elt with
   | REG (SYM sym) -> READVAR (CONTROLVAR sym, REGISTER)
   | CREG (SYM sym) -> READVAR (CONTROLVAR sym, CREGISTER)
   | FREG (SYM sym) -> READVAR (CONTROLVAR sym, FREGISTER)
   | FCREG (SYM sym) -> READVAR (CONTROLVAR sym, FCREGISTER)
   | REG (CONC r) -> REGVALUE r
   | CREG (CONC r) -> CREGVALUE r
   | FREG (CONC r) -> FREGVALUE r
   | FCREG (CONC r) -> FCREGVALUE r
   | HI -> HIREGVALUE
   | LO -> LOREGVALUE
   | CACHEENTRY (c, ce) ->
        let c' =
           match c with
           | CONC c' -> c'
           | SYM _ -> Util.crash "symexec: expr_of_statelement: symbolic cacheentry"
        in
        let ce' =
           match ce with
           | CONC ce' -> ce'
           | SYM _ -> Util.crash "symexec: expr_of_statelement: symbolic cacheentry"
        in
        WHICHCACHEENTRYVALUE (c', ce')

let unaffected_except_n ppre ppost elts =
   let mkeq elt e1 e2 =
      match elt with
      | REG _
      | CREG _
      | FREG _
      | FCREG _
      | HI
      | LO -> eq_mint e1 e2
      | CACHEENTRY _ -> eq_logic e1 e2
   in

(*
   let usesarray elt =
      match elt with
      | REG _
      | CREG _
      | FREG _
      | FCREG _ -> true
      | _ -> false
   in
*)

   let matches elt1 elt2 =
(*
      match Modes.smt_register_mode with
      | Modes.REGS_VIA_VARS
      | Modes.REGS_VIA_ARRAYEQS ->
*)
           elt1 = elt2
(*
      | Modes.REGS_VIA_ARRAYSTORES ->
           match elt1, elt2 with
           | REG _, REG _ -> true
           | CREG _, CREG _ -> true
           | FREG _, FREG _ -> true
           | FCREG _, FCREG _ -> true
           | _, _ -> elt1 = elt2
*)
   in

   let once checkelt =
      (*
       * all_stateelements() returns concrete register names, so we
       * know checkelt must be concrete. Asserting this is a pain, so
       * we shan't.
       *)
      let onceonce excludeelt =
         if matches excludeelt checkelt then FALSE (* simple cases *)
         else
            let ex sym ty'sym k =
               neq_logic (READVAR (CONTROLVAR sym, ty'sym)) k
            in
            let ex2 sym1 ty'sym1 k1 sym2 ty'sym2 k2 =
               LOGOR (ex sym1 ty'sym1 k1, ex sym2 ty'sym2 k2)
            in
            match excludeelt, checkelt with
            | REG (SYM sym), REG (CONC r) -> ex sym REGISTER (REGVALUE r)
            | CREG (SYM sym), CREG (CONC r) -> ex sym CREGISTER (CREGVALUE r)
            | FREG (SYM sym), FREG (CONC r) -> ex sym FREGISTER (FREGVALUE r)
            | FCREG (SYM sym), FCREG (CONC r) -> ex sym FCREGISTER (FCREGVALUE r)
            | CACHEENTRY (SYM sym1, SYM sym2), CACHEENTRY (CONC c, CONC ce) ->
                 ex2 sym1 WHICHCACHE (WHICHCACHEVALUE c)
                     sym2 WHICHCACHEENTRY (WHICHCACHEENTRYIDVALUE ce)
            | CACHEENTRY (SYM sym, CONC ce1), CACHEENTRY (CONC c, CONC ce2) ->
                 if ce1 = ce2 then
                    ex sym WHICHCACHE (WHICHCACHEVALUE c)
                 else TRUE
            | CACHEENTRY (CONC c1, SYM sym), CACHEENTRY (CONC c2, CONC ce) ->
                 if c1 = c2 then
                    ex sym WHICHCACHEENTRY (WHICHCACHEENTRYIDVALUE ce)
                 else TRUE
            | _ -> TRUE
      in
      let prestate, poststate, mkeq' =
(*
         match Modes.smt_register_mode with
         | Modes.REGS_VIA_ARRAYSTORES when usesarray checkelt ->
              let t = type_of_stateelement checkelt in
              READALLSTATE (t, ppre),
              READALLSTATE (t, ppost),
              eq_logic
         | _ ->
*)
              let e = expr_of_stateelement checkelt in
              READSTATE (e, ppre, Z_ZERO),
              READSTATE (e, ppost, Z_ANY),
              mkeq checkelt
      in              
      IMPLIES (and_list (List.map onceonce elts),
               mkeq' prestate poststate)
   in
   let stateelements =
(*
      match Modes.smt_register_mode with
      | Modes.REGS_VIA_VARS
      | Modes.REGS_VIA_ARRAYEQS ->
*)
           Mipstools.all_stateelements ()
(*
      | Modes.REGS_VIA_ARRAYSTORES ->
           (* XXX this should probably go in Mipstools *)
           let reg = [ REG (CONC V0); ] in
           let creg = match Mipstools.all_cregs () with
              | [] -> []
              | r :: _ -> [CREG r]
           in
           let freg = match Mipstools.all_fregs () with
              | [] -> []
              | r :: _ -> [FREG r]
           in
           let fcreg = match Mipstools.all_fcregs () with
              | [] -> []
              | r :: _ -> [FCREG r]
           in
           let hilo =
              if !Enabled.use_hilo then [HI; LO] else []
           in
           let cache =
              List.map (fun c -> CACHEENTRY c) (Mipstools.all_cacheentries ())
           in
           reg @ creg @ freg @ fcreg @ hilo @ cache
*)
   in
   List.map once stateelements

let unaffected ppre ppost = unaffected_except_n ppre ppost []
let unaffected_except ppre ppost elt = unaffected_except_n ppre ppost [elt]

let unaffected_except_cache ppre ppost c =
   let elts = [
      CACHEENTRY (c, CONC CACHE_BLUE);
      CACHEENTRY (c, CONC CACHE_GREEN);
   ] in
   unaffected_except_n ppre ppost elts

(*
 * symbolically execute one concrete instruction
 *)
let symexec'cinsn regmap cachevars ppre cinsn ppost =

   let getzs pp = if pp = ppost then Z_ANY else Z_ZERO in
   let readreg ctor t r pp =
      get_one_regstate regmap ctor t r pp (getzs pp)
   in
   let readreg_r = readreg (fun r -> REGVALUE r) REGISTER in
   let readreg_c = readreg (fun r -> CREGVALUE r) CREGISTER in
   let readreg_f = readreg (fun r -> FREGVALUE r) FREGISTER in
   let readreg_fc = readreg (fun r -> FCREGVALUE r) FCREGISTER in

   let readhi pp = READSTATE (HIREGVALUE, pp, getzs pp) in
   let readlo pp = READSTATE (LOREGVALUE, pp, getzs pp) in

   let readimm imm ty =
      match imm with
      | CONC n -> INTVALUE (n, ty)
      | SYM sym -> READVAR (CONTROLVAR sym, ty)
   in
   let readsimm16 imm = MINTUOP (SHWIDEN, readimm imm SINT16) in
   let readuimm16 imm = MINTUOP (UHWIDEN, readimm imm UINT16) in
   let readuimm5 imm = readimm imm UINT5 in

   let readmem (MEMORY (rt, simm16)) =
      let pre'rt = readreg_r rt ppre in
      let imm = MINTUOP (TOUNSIGNED, readsimm16 simm16) in
      MINTBOP (ADD, pre'rt, imm)
   in

(*
   let doset post'rd cond =
      let t = IMPLIES (cond, MINTBOP (EQ, post'rd, INTVALUE (1, UINT32))) in
      let f = IMPLIES (LOGNOT cond, MINTBOP (EQ, post'rd, INTVALUE (0, UINT32))) in
      LOGAND (t, f)
   in
*)
   let doset cond =
      IF (cond, INTVALUE (1, UINT32), INTVALUE (0, UINT32))
   in

   let store_rd rd result =
(*
      match Modes.smt_register_mode with
      | Modes.REGS_VIA_VARS
      | Modes.REGS_VIA_ARRAYEQS ->
*)
           let post'rd = readreg_r rd ppost in
           let prop'rd = MINTBOP (EQ, post'rd, result) in
           prop'rd :: unaffected_except ppre ppost (REG rd)
(*
      | Modes.REGS_VIA_ARRAYSTORES ->
           let pre'regs = READALLSTATE (REGISTER, ppre) in
           let post'regs = READALLSTATE (REGISTER, ppost) in
           let rd' = match rd with
              | CONC r -> REGVALUE r
              | SYM sym -> READVAR (CONTROLVAR sym, REGISTER)
           in
           let update = UPDATEMAP (pre'regs, rd', result) in
           let prop'eq = eq_logic post'regs update in
           prop'eq :: unaffected_except ppre ppost (REG rd)
*)
   in
   let store_cd cd result =
(*
      match Modes.smt_register_mode with
      | Modes.REGS_VIA_VARS
      | Modes.REGS_VIA_ARRAYEQS ->
*)
           let post'cd = readreg_c cd ppost in
           let prop'cd = MINTBOP (EQ, post'cd, result) in
           prop'cd :: unaffected_except ppre ppost (CREG cd)
(*
      | Modes.REGS_VIA_ARRAYSTORES ->
           let pre'regs = READALLSTATE (CREGISTER, ppre) in
           let post'regs = READALLSTATE (CREGISTER, ppost) in
           let cd' = match cd with
              | CONC r -> CREGVALUE r
              | SYM sym -> READVAR (CONTROLVAR sym, CREGISTER)
           in
           let update = UPDATEMAP (pre'regs, cd', result) in
           let prop'eq = eq_logic post'regs update in
           prop'eq :: unaffected_except ppre ppost (CREG cd)
*)
   in
   let store_fd fd result =
(*
      match Modes.smt_register_mode with
      | Modes.REGS_VIA_VARS
      | Modes.REGS_VIA_ARRAYEQS ->
*)
           let post'fd = readreg_f fd ppost in
           let prop'fd = MINTBOP (EQ, post'fd, result) in
           prop'fd :: unaffected_except ppre ppost (FREG fd)
(*
      | Modes.REGS_VIA_ARRAYSTORES ->
           let pre'regs = READALLSTATE (REGISTER, ppre) in
           let post'regs = READALLSTATE (REGISTER, ppost) in
           let fd' = match fd with
              | CONC r -> FREGVALUE r
              | SYM sym -> READVAR (CONTROLVAR sym, FREGISTER)
           in
           let update = UPDATEMAP (pre'regs, fd', result) in
           let prop'eq = eq_logic post'regs update in
           prop'eq :: unaffected_except ppre ppost (FREG fd)
*)
   in
   let store_fcd fcd result =
(*
      match Modes.smt_register_mode with
      | Modes.REGS_VIA_VARS
      | Modes.REGS_VIA_ARRAYEQS ->
*)
           let post'fcd = readreg_fc fcd ppost in
           let prop'fcd = MINTBOP (EQ, post'fcd, result) in
           prop'fcd :: unaffected_except ppre ppost (FCREG fcd)
(*
      | Modes.REGS_VIA_ARRAYSTORES ->
           let pre'regs = READALLSTATE (REGISTER, ppre) in
           let post'regs = READALLSTATE (REGISTER, ppost) in
           let fcd' = match fcd with
              | CONC r -> FCREGVALUE r
              | SYM sym -> READVAR (CONTROLVAR sym, FCREGISTER)
           in
           let update = UPDATEMAP (pre'regs, fcd', result) in
           let prop'eq = eq_logic post'regs update in
           prop'eq :: unaffected_except ppre ppost (FCREG fcd)
*)
   in

   let cache_golem ctor sc golemsym =
      match sc with
      | CONC c -> READGOLEM (GOLEMVALUE (ctor c))
      | SYM _sym ->
           let x = SYM golemsym in
           get_one_regstate regmap (fun g -> GOLEMVALUE g) GOLEM x ppre Z_ZERO
   in
   let cache_byindex c fakeaddr golemsym =
      let g = cache_golem (fun c' -> G_INDEX c') c golemsym in
      GOLEMIZE (g, fakeaddr)
   in
   let cache_byvaddr c vaddr golemsym =
      let g = cache_golem (fun c' -> G_VADDR c') c golemsym in
      GOLEMIZE (g, vaddr)
   in
   let cache_transition c entryp transition =
      begin match c with
      | SYM _ -> ()
      | CONC _ -> Util.crash "symexec: cache_transition: concrete cache"
      end;
      let (idvar, prevar, postvar) = cachevars in
      let idpred =
         eq_logic (READVAR (idvar, BOOL)) entryp
      in
      let prestate = READVAR (prevar, CACHESTATE) in
      let poststate = READVAR (postvar, CACHESTATE) in
(*
      let once color =
         let handle = CACHEENTRY (c, CONC color) in
         let e'handle = expr_of_stateelement handle in
         let prestate = READSTATE (e'handle, ppre, Z_ZERO) in
         let poststate = READSTATE (e'handle, ppost, Z_ZERO) in
*)
      let entrypred =
	 let impl (a, b) =
	    let ax = CACHESTATEVALUE a in
	    let bx = CACHESTATEVALUE b in
	    IMPLIES (eq_logic prestate ax, eq_logic poststate bx)
	 in
         let unaff =
            unaffected_except_cache ppre ppost c
         in
         and_list (List.map impl transition @ unaff)
      in
(*
      IF (entryp, once CACHE_BLUE, once CACHE_GREEN)
*)
      LOGAND (idpred, entrypred)
   in
   let cache_wb_transition = [
      (CACHE_PRESENT_DIRTY, CACHE_PRESENT_CLEAN);
      (CACHE_PRESENT_CLEAN, CACHE_PRESENT_CLEAN);
      (CACHE_ABSENT_VALID, CACHE_ABSENT_VALID);
      (CACHE_ABSENT_INVALID, CACHE_ABSENT_INVALID);
   ] in
   let cache_wbinv_transition = [
      (CACHE_PRESENT_DIRTY, CACHE_ABSENT_VALID);
      (CACHE_PRESENT_CLEAN, CACHE_ABSENT_VALID);
      (CACHE_ABSENT_VALID, CACHE_ABSENT_VALID);
      (CACHE_ABSENT_INVALID, CACHE_ABSENT_INVALID);
   ] in
   let cache_inv_transition = [
      (CACHE_PRESENT_DIRTY, CACHE_ABSENT_INVALID);
      (CACHE_PRESENT_CLEAN, CACHE_ABSENT_VALID);
      (CACHE_ABSENT_VALID, CACHE_ABSENT_VALID);
      (CACHE_ABSENT_INVALID, CACHE_ABSENT_INVALID);
   ] in


   match cinsn with
   (* arithmetic *)
   | ADDU (rd, rs, rt) ->
(*
       let e'rs = expr_of_stateelement (REG rs) in
       let e'rt = expr_of_stateelement (REG rt) in
       let e'rd = expr_of_stateelement (REG rd) in
       let pre'rs = READSTATE (e'rs, ppre, Z_ZERO) in
       let pre'rt = READSTATE (e'rt, ppre, Z_ZERO) in
       let post'rd = READSTATE (e'rd, ppost, Z_ANY) in
*)
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       store_rd rd (MINTBOP (ADD, pre'rs, pre'rt))
   | ADDIU (rd, rs, simm16) ->
       let pre'rs = readreg_r rs ppre in
       let imm = MINTUOP (TOUNSIGNED, readsimm16 simm16) in
       store_rd rd (MINTBOP (ADD, pre'rs, imm))
   | DIV (rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       let post'lo = readlo ppost in
       let post'hi = readhi ppost in
       let prop'lo = MINTBOP (EQ, post'lo, MINTBOP (SDIV, pre'rs, pre'rt)) in
       let prop'hi = MINTBOP (EQ, post'hi, MINTBOP (SMOD, pre'rs, pre'rt)) in
       prop'lo :: prop'hi :: unaffected_except_n ppre ppost [LO; HI]
   | DIVU (rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       let post'lo = readlo ppost in
       let post'hi = readhi ppost in
       let prop'lo = MINTBOP (EQ, post'lo, MINTBOP (UDIV, pre'rs, pre'rt)) in
       let prop'hi = MINTBOP (EQ, post'hi, MINTBOP (UMOD, pre'rs, pre'rt)) in
       prop'lo :: prop'hi :: unaffected_except_n ppre ppost [LO; HI]
   | LUI (rd, uimm16) ->
       let imm = readuimm16 uimm16 in
       store_rd rd (MINTBOP (SHL, imm, INTVALUE (16, UINT5)))
   | MULT (rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       let post'lo = readlo ppost in
       let post'hi = readhi ppost in
       let prop'lo = MINTBOP (EQ, post'lo, MINTBOP (MUL_LO, pre'rs, pre'rt)) in
       let prop'hi = MINTBOP (EQ, post'hi, MINTBOP (SMUL_HI, pre'rs, pre'rt)) in
       prop'lo :: prop'hi :: unaffected_except_n ppre ppost [LO; HI]
   | MULTU (rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       let post'lo = readlo ppost in
       let post'hi = readhi ppost in
       let prop'lo = MINTBOP (EQ, post'lo, MINTBOP (MUL_LO, pre'rs, pre'rt)) in
       let prop'hi = MINTBOP (EQ, post'hi, MINTBOP (UMUL_HI, pre'rs, pre'rt)) in
       prop'lo :: prop'hi :: unaffected_except_n ppre ppost [LO; HI]
   | SUBU (rd, rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       store_rd rd (MINTBOP (SUB, pre'rs, pre'rt))
(* XXX notyet
   | LUI_extcode_hi (rd, xcl) ->
       store_rd rd ...
   | ADDIU_extcode_lo (rd, rs, xcl) ->
       store_rd rd ...
   | LUI_extdata_hi (rd, xdl) ->
       store_rd rd ...
   | ADDIU_extdata_lo (rd, rs, xdl) ->
       store_rd rd ...
   | LUI_localcode_hi (rd, lcl) ->
       store_rd rd ...
   | ADDIU_localcode_lo (rd, rs, lcl) ->
       store_rd rd ...
*)

     (* comparisons *)
   | SLT (rd, rs, rt) ->
       let pre'rs = MINTUOP (TOSIGNED, readreg_r rs ppre) in
       let pre'rt = MINTUOP (TOSIGNED, readreg_r rt ppre) in
       store_rd rd (doset (MINTBOP (SLT, pre'rs, pre'rt)))
   | SLTI (rd, rs, simm16) ->
       let pre'rs = MINTUOP (TOSIGNED, readreg_r rs ppre) in
       let imm = readsimm16 simm16 in
       store_rd rd (doset (MINTBOP (SLT, pre'rs, imm)))
   | SLTU (rd, rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       store_rd rd (doset (MINTBOP (ULT, pre'rs, pre'rt)))
   | SLTIU (rd, rs, simm16) ->
       let pre'rs = readreg_r rs ppre in
       let imm = MINTUOP (TOUNSIGNED, readsimm16 simm16) in
       store_rd rd (doset (MINTBOP (ULT, pre'rs, imm)))

     (* bitwise logic *)
   | AND (rd, rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       store_rd rd (MINTBOP (BITAND, pre'rs, pre'rt))
   | ANDI (rd, rs, uimm16) ->
       let pre'rs = readreg_r rs ppre in
       let imm = readuimm16 uimm16 in
       store_rd rd (MINTBOP (BITAND, pre'rs, imm))
   | NOR (rd, rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       store_rd rd (MINTUOP (BITNOT, MINTBOP (BITOR, pre'rs, pre'rt)))
   | OR (rd, rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       store_rd rd (MINTBOP (BITOR, pre'rs, pre'rt))
   | ORI (rd, rs, uimm16) ->
       let pre'rs = readreg_r rs ppre in
       let imm = readuimm16 uimm16 in
       store_rd rd (MINTBOP (BITOR, pre'rs, imm))
   | XOR (rd, rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       store_rd rd (MINTBOP (BITXOR, pre'rs, pre'rt))
   | XORI (rd, rs, uimm16) ->
       let pre'rs = readreg_r rs ppre in
       let imm = readuimm16 uimm16 in
       store_rd rd (MINTBOP (BITXOR, pre'rs, imm))

     (* shifts *)
   | SLL (rd, rs, uimm5) ->
       let pre'rs = readreg_r rs ppre in
       let imm = readuimm5 uimm5 in
       store_rd rd (MINTBOP (SHL, pre'rs, imm))
   | SLLV (rd, rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = MINTUOP (TRUNCU5, readreg_r rt ppre) in
       store_rd rd (MINTBOP (SHL, pre'rs, pre'rt))
   | SRA (rd, rs, uimm5) ->
       let pre'rs = MINTUOP (TOSIGNED, readreg_r rs ppre) in
       let imm = readuimm5 uimm5 in
       store_rd rd (MINTUOP (TOUNSIGNED, MINTBOP (SSHR, pre'rs, imm)))
   | SRAV (rd, rs, rt) ->
       let pre'rs = MINTUOP (TOSIGNED, readreg_r rs ppre) in
       let pre'rt = MINTUOP (TRUNCU5, readreg_r rt ppre) in
       store_rd rd (MINTUOP (TOUNSIGNED, MINTBOP (SSHR, pre'rs, pre'rt)))
   | SRL (rd, rs, uimm5) ->
       let pre'rs = readreg_r rs ppre in
       let imm = readuimm5 uimm5 in
       store_rd rd (MINTBOP (USHR, pre'rs, imm))
   | SRLV (rd, rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = MINTUOP (TRUNCU5, readreg_r rt ppre) in
       store_rd rd (MINTBOP (USHR, pre'rs, pre'rt))

     (* transfers *)
(* XXX notyet
   | B lcl ->
   | BEQ (rs, rt, lcl) ->
   | BNE (rs, rt, lcl) ->
   | J xcl ->
   | JAL xcl ->
   | JALR rs ->
       ... unaffected_except ppre ppost (REG RA)
   | JR rs ->
*)

     (* moves *)
   | CFC1 (rd, fcs) ->
       let pre'fcs = readreg_fc fcs ppre in
       store_rd rd pre'fcs
   | CTC1 (rs, fcd) ->
       (* yes, destination is on the right *)
       let pre'rs = readreg_r rs ppre in
       store_fcd fcd pre'rs
   | MFC0 (rd, cs) ->
       let pre'cs = readreg_c cs ppre in
       store_rd rd pre'cs
   | MFC1 (rd, fs) ->
       let pre'fs = readreg_f fs ppre in
       store_rd rd pre'fs
   | MFHI rd ->
       let pre'hi = readhi ppre in
       store_rd rd pre'hi
   | MFLO rd ->
       let pre'lo = readlo ppre in
       store_rd rd pre'lo
   | MTC0 (rs, cd) ->
       (* yes, destination is on the right *)
       let pre'rs = readreg_r rs ppre in
       store_cd cd pre'rs
   | MTC1 (rs, fd) ->
       (* yes, destination is on the right *)
       let pre'rs = readreg_r rs ppre in
       store_fd fd pre'rs
   | MTHI rs ->
       let pre'rs = readreg_r rs ppre in
       let post'hi = readhi ppost in
       let prop'rd = MINTBOP (EQ, pre'rs, post'hi) in
       prop'rd :: unaffected_except ppre ppost HI
   | MTLO rs ->
       let pre'rs = readreg_r rs ppre in
       let post'lo = readlo ppost in
       let prop'rd = MINTBOP (EQ, pre'rs, post'lo) in
       prop'rd :: unaffected_except ppre ppost LO
   | MOVN (rd, rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       let cond = MINTBOP (EQ, pre'rt, INTVALUE (0, UINT32)) in
       let yes = IMPLIES (cond, and_list (unaffected ppre ppost)) in
       let no = IMPLIES (LOGNOT cond, and_list (store_rd rd pre'rs)) in
       [yes; no]
   | MOVZ (rd, rs, rt) ->
       let pre'rs = readreg_r rs ppre in
       let pre'rt = readreg_r rt ppre in
       let cond = MINTBOP (EQ, pre'rt, INTVALUE (0, UINT32)) in
       let yes = IMPLIES (cond, and_list (store_rd rd pre'rs)) in
       let no = IMPLIES (LOGNOT cond, and_list (unaffected ppre ppost)) in
       [yes; no]

     (* loads *)
(* XXX notyet
   | LB (rd, MEMORY (rt, simm16)) ->
   | LBU (rd, MEMORY (rt, simm16)) ->
   | LDC1 (fd, MEMORY (rt, simm16)) ->
   | LH (rd, MEMORY (rt, simm16)) ->
   | LHU (rd, MEMORY (rt, simm16)) ->
   | LW (rd, MEMORY (rt, simm16)) ->
   | LWC1 (fd, MEMORY (rt, simm16)) ->
*)
     (* stores *)
(* XXX notyet
   | SB (rs, MEMORY (rt, simm16)) ->
   | SDC1 (fs, MEMORY (rt, simm16)) ->
   | SH (rs, MEMORY (rt, simm16)) ->
   | SW (rs, MEMORY (rt, simm16)) ->
   | SWC1 (rs, MEMORY (rt, simm16)) ->
     (* system control *)
*)
   | CACHE_INDEX_WBINV (c, m, golemsym) ->
       let entryp = cache_byindex c (readmem m) golemsym in
       let wbinvpred = cache_transition c entryp cache_wbinv_transition in
       [ wbinvpred; ]
   | CACHE_VADDR_WB (c, m, golemsym) ->
       let entryp = cache_byvaddr c (readmem m) golemsym in
       let wbpred = cache_transition c entryp cache_wb_transition in
       [ wbpred; ]
   | CACHE_VADDR_WBINV (c, m, golemsym) ->
       let entryp = cache_byvaddr c (readmem m) golemsym in
       let wbinvpred = cache_transition c entryp cache_wbinv_transition in
       [ wbinvpred; ]
   | CACHE_VADDR_INV (c, m, golemsym) ->
       let entryp = cache_byvaddr c (readmem m) golemsym in
       let invpred = cache_transition c entryp cache_inv_transition in
       [ invpred; ]

(* XXX notyet
   | TLBP ->
   | TLBR ->
   | TLBWI ->
   | TLBWR ->
*)
(* XXX temporary *)
   | _ -> [TRUE]

(*
 * symbolically execute one instruction
 *)
let symexec'insn (i, (ppre, insn, ppost)) =
   let (_opsetbindings, opset) = Insns.mkoperandset i in
   let (regbindings, regstateprops, regmap) =
(*
      if Modes.registers_use_vars then
*)
         read_regstate ppre i opset ppost
(*
      else
         ([], [], VarMap.empty)
*)
   in
   let (cachebindings, cachestateprops, cachevars) =
      read_cachestate ppre i opset ppost
   in
   let bindings = regbindings @ cachebindings in
   let stateprops = regstateprops @ cachestateprops in

   let execprops =
      match insn with
      | CONC cinsn -> symexec'cinsn regmap cachevars ppre cinsn ppost
      | SYM sym ->
           let insns = Insns.insns opset in
           let once cinsn =
              let prop =
                 and_list (symexec'cinsn regmap cachevars ppre cinsn ppost)
              in
              let read = READVAR (CONTROLVAR sym, INSN) in
              IMPLIES (LOGEQ (read, INSNVALUE cinsn), prop)
           in
           List.map once insns
   in
   let label = "symbolic execution for insn" ^ string_of_int i in
   (bindings, (label, and_list (stateprops @ execprops)))

(*
 * symbolically execute symcode: generate a list of assertions about
 * all the program points.
 *
 * The passed-in pre and post arguments are the specification assertions
 * about the first and last program point. We can inspect these or pass
 * them through as we see fit.
 *)
let go symcode =
   let (ppinsns, ppost) = symcode in

   (* give each insn its own ref to its pre- and post- program points *)
   let (ppinsns', _) =
      let once (ppre, insn) (z, ppost) = ((ppre, insn, ppost) :: z, ppre) in
      List.fold_right once ppinsns ([], ppost)
   in

   let results = (List.map symexec'insn (Util.number ppinsns')) in

   let (bindingses, props) = Util.unzip results in
   let bindings = List.concat bindingses in
   (bindings, props)
