(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

(*
 * Program points appear between instructions. We number them for
 * lookup purposes.
 *)

type ppoint = PPOINT of int
type t = ppoint

(*
 * Program points to use when writing code-independent preconditions
 * and postconditions.
 *)
let pre = PPOINT 0
let post = PPOINT 1

(*
 * Printing
 *)
let to_string (PPOINT n) =
   "pp." ^ string_of_int n

let prefix (PPOINT n) =
   "pp" ^ string_of_int n ^ "_"

let to_smt (PPOINT n) = n
let from_smt n = PPOINT n

(*
 * Point generation
 *)
type generate_env = int ref
let generate_prep () = ref 0
let generate (env: generate_env) =
   let ret = !env in
   env := 1 + ret;
   PPOINT ret
let generated (env: generate_env) =
   (* XXX: should this be !env, or !env - 1? *)
   List.init !env (fun i -> PPOINT i)

(*
 * Map type
 *)

module Key = struct
   type t = ppoint
   let compare (PPOINT x) (PPOINT y) = compare x y
end

module SymKey = struct
   type t = (ppoint * string)
   let compare = compare
end

module OSymKey = struct
   type t = (ppoint option * string)
   let compare = compare
end

module OSymMap = Map.Make(OSymKey)
module SymMap = Map.Make(SymKey)
module Map = Map.Make(Key)
