(*
 * Copyright (c) 2018
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

type uop =
   | BITNOT	(* bitwise not *)
   | SBWIDEN	(* widen signed 8 bits to 32 *)
   | SHWIDEN	(* widen signed 16 bits to 32 *)
   | UBWIDEN	(* widen unsigned 8 bits to 32 *)
   | UHWIDEN	(* widen unsigned 16 bits to 32 *)
   | U5WIDEN	(* widen unsigned 5 bits to 32 *)
   | TRUNCU5	(* truncate unsigned 32 bits to 5 *)
   | TOSIGNED	(* convert 32 bit unsigned to signed *)
   | TOUNSIGNED	(* convert 32 bit signed to unsigned *)

type bop =
   | EQ		(* equality *)
   | ULT	(* unsigned less than *)
   | SLT	(* signed less than *)
   | ADD	(* add *)
   | SUB	(* subtract *)
   | SMUL_HI	(* signed multiply, high word of result *)
   | UMUL_HI	(* unsigned multiply, high word of result *)
   | MUL_LO	(* multiply, low word of result *)
   | SDIV	(* signed divide *)
   | SMOD	(* signed modulus *)
   | UDIV	(* unsigned divide *)
   | UMOD	(* unsigned modulus *)
   | BITAND	(* bitwise and *)
   | BITOR	(* bitwise or *)
   | BITXOR	(* bitwise xor *)
   | SHL	(* bitwise shift left *)
   | USHR	(* bitwise unsigned shift right *)
   | SSHR	(* bitwise signed shift right *)

let string_of_uop op =
   match op with
   | BITNOT -> "!"
   | SBWIDEN -> "cast (int32 <- int8)"
   | SHWIDEN -> "cast (int32 <- int16)" 
   | UBWIDEN -> "cast (uint32 <- uint8)"
   | UHWIDEN -> "cast (uint32 <- uint16)"
   | U5WIDEN -> "cast (uint32 <- uint5)"
   | TRUNCU5 -> "cast (uint5 <- uint32)"
   | TOSIGNED -> "cast (int32 <- uint32)"
   | TOUNSIGNED -> "cast (uint32 <- int32)"

let string_of_bop op =
   match op with
   | BITOR -> "|"
   | BITXOR -> "^"
   | BITAND -> "&"
   | EQ -> "==m"
   | ULT -> "<u"
   | SLT -> "<s"
   | SHL -> "<<"
   | USHR -> ">>u"
   | SSHR -> ">>s"
   | ADD -> "+"
   | SUB -> "-"
   | SMUL_HI -> "*sh"
   | UMUL_HI -> "*uh"
   | MUL_LO -> "*l"
   | SDIV -> "/s"
   | SMOD -> "%s"
   | UDIV -> "/u"
   | UMOD -> "%u"

let prec_of_bop op =
   match op with
   | BITOR -> 30
   | BITXOR -> 31
   | BITAND -> 32
   | EQ -> 40
   | ULT -> 41
   | SLT -> 41
   | SHL -> 45
   | USHR -> 45
   | SSHR -> 45
   | ADD -> 50
   | SUB -> 50
   | SMUL_HI -> 60
   | UMUL_HI -> 60
   | MUL_LO -> 60
   | SDIV -> 60
   | SMOD -> 60
   | UDIV -> 60
   | UMOD -> 60

(* used to print SMT expressions *)
let name_of_uop op =
   match op with
   | BITNOT -> "not"
   | SBWIDEN -> "s8widen"
   | SHWIDEN -> "s16widen"
   | UBWIDEN -> "u8widen"
   | UHWIDEN -> "u16widen"
   | U5WIDEN -> "u5widen"
   | TRUNCU5 -> "u5trunc"
   | TOSIGNED -> "tosigned"
   | TOUNSIGNED -> "tounsigned"

(* used to print SMT expressions *)
let name_of_bop op =
   match op with
   | BITOR -> "or"
   | BITXOR -> "xor"
   | BITAND -> "and"
   | EQ -> "eq"
   | ULT -> "ult"
   | SLT -> "slt"
   | SHL -> "shl"
   | USHR -> "ushr"
   | SSHR -> "sshr"
   | ADD -> "add"
   | SUB -> "sub"
   | SMUL_HI -> "smulhi"
   | UMUL_HI -> "umulhi"
   | MUL_LO -> "mullo"
   | SDIV -> "sdiv"
   | SMOD -> "smod"
   | UDIV -> "udiv"
   | UMOD -> "umod"
