(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 * Written by David A. Holland.
 *
 * Copyright (c) 2020 David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

(*
 * Penguin's current log model is as follows:
 *
 * - Some ("console") stuff is printed to stdout. This is supposed to be
 * stuff of interest to humans while it's running, e.g. progress updates.
 * This is suppressed if quiet is enabled with -q.
 *
 * - If dumps are turned on with -d, the internal representations are
 * dumped to a log file, whose name is not specified but derived from
 * the input spec name: foo.pen -> foo.log.
 *
 * - If verbose is turned on with -v, additional stuff is dumped to the
 * log file.
 *
 * - Timing information is, if enabled with -t, printed to a separate
 * file.
 *
 * The following families of print functions are provided:
 *   csay       print to console
 *   dsay       print to dump file
 *   vsay       print to dump file if verbose
 *   dcsay      print to both console and dump file
 *   vcsay      print to both console and dump-file-if-verbose
 *   timingsay  print to the timing file
 *
 * By default the say functions take an entire line; to print a prefix
 * use *rawsay.
 *
 * It is not clear currently if this model is adequate. We've found on
 * this project that managing the outputs of synthesis tools is a
 * large pain, because on the one hand it's important to be able to
 * see what's going on but on the other hand solver nondeterminism
 * makes a lot of what one can print be subject to arbitrary
 * variations and thus not be good for testing.
 *
 * Some conclusions so far:
 *
 * 1. Even for bulk runs and tests it's still useful to print
 * interactive progress to stdout, because these are often run by hand
 * with someone checking in on them periodically. What gets printed
 * should be enough to be able to see if things are progressing (vs.
 * stuck somehow or about to time out in the solver) but not so much
 * that it prints too much to follow or to review in a largish window.
 *
 * 2. For this reason the stdout material should not be logged to a
 * file while testing, but should be displayed normally. Instead
 * anything with test value should be repeated into a a test log.
 *
 * 3. But because tests will also be run as batch jobs whose output
 * is ignored (or which shouldn't produce output because it will be
 * sent back in some inconvenient way) there should be an option to
 * suppress the console output, and this should be controllable from
 * the test infrastructure rather than baked in.
 *
 * 4. Logging the details of the solver interactions (e.g. the exact
 * formulae sent off and models sent back) is not worthwhile because
 * in most cases a small fluctuation caused by arbitrary solver
 * nondeterminism will produce gigantic log diffs that cannot be
 * inspected. Also the resulting logs are large and while debugging
 * sometimes requires manual analysis mostly they're useless.
 *
 * 5. It is nonetheless desirable to log SMT-level output for testing
 * purposes so it doesn't arbitrarily break, but this is directly at
 * odds with point (4).
 *
 * 6. Timings vary so much that they should in no way appear in any
 * test log, unless maybe rounded on a log scale or something.
 *
 *
 * While the log model above matches these considerations reasonably
 * well, I'm pretty sure the things that actually get logged need
 * improvement. XXX.
 *)

let logfile = ref stdout
let timingfile = ref None

let dodumps = ref false
let quiet = ref false
let verbose = ref false

let setup logname timingname dodumps_val quiet_val verbose_val =
   dodumps := dodumps_val;
   quiet := quiet_val;
   verbose := verbose_val;
   if dodumps_val then
      logfile := open_out logname
   else ();
   match timingname with
   | None -> ()
   | Some fn -> timingfile := Some (open_out fn)

let shutdown () =
   close_out (!logfile);
   match !timingfile with
   | None -> ()
   | Some f -> close_out f

let csay txt =
   if !quiet then ()
   else begin
      print_string txt;
      print_string "\n"
   end

let crawsay txt =
   if !quiet then ()
   else begin
      print_string txt
   end

let dsay txt =
   if !dodumps then begin
      output_string !logfile txt;
      output_string !logfile "\n"
   end else ()

let vsay txt =
   if !dodumps && !verbose then begin
      output_string !logfile txt;
      output_string !logfile "\n"
   end else ()

let drawsay txt =
   if !dodumps then begin
      output_string !logfile txt
   end else ()

let dcsay txt = csay txt; dsay txt
let vcsay txt = csay txt; vsay txt
let dcrawsay txt = crawsay txt; drawsay txt
   

let timingsay txt =
   match !timingfile with
   | None -> ()
   | Some f ->
        output_string f txt;
        output_string f "\n"


let flush () =
   flush stdout;
   flush !logfile;
   match !timingfile with
   | None -> ()
   | Some f -> flush f
