(*
 * Copyright (c) 2020 David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

(*
 * I keep writing different versions of this in the hopes of finding
 * an approarch that works out to not be gross...
 *)

(* begin mandatory cutpaste *)
type 't result =
| FAILURE of string
| SUCCESS of 't

type 't optinfo =
| NOARG of (string -> 't option)
| HASARG of (string -> string -> 't option)
(* end mandatory cutpaste *)

type 't state = {
   check: string -> 't optinfo option;

   curarg: char list;
   inargs: string list;
   outopts: 't list;
   outargs: string list;
}

let state_init check inargs =
   { check; curarg = []; inargs; outopts = []; outargs = []; }

let state_getcurarg (s: 't state) =
   match s.curarg with
   | [] -> None
   | ch :: curarg -> Some (ch, { s with curarg; })

let state_getnextarg (s: 't state) =
   match s.inargs with
   | [] -> None
   | arg :: inargs -> Some (arg, { s with inargs; })

let state_setcurarg (s: 't state) (curargstr: string) =
   assert (s.curarg = []);
   let curarg = Util.charlist_of_string curargstr in
   { s with curarg; }

let state_takecurarg (s: 't state) =
   match s.curarg with
   | [] -> None
   | chars -> Some (Util.string_of_charlist chars, { s with curarg = []; })

let state_addresult (s: 't state) (result: 't) =
   let outopts = result :: s.outopts in
   { s with outopts; }

let state_extract (s: 't state) =
   (List.rev s.outopts, List.rev s.outargs)

let state_shiftword (s: 't state) w =
   let outargs = w :: s.outargs in
   { s with outargs; }

let state_stop (s: 't state) =
   let outargs = List.rev s.inargs @ s.outargs in
   { s with outargs; }


let dispatch_noarg dispatch opt state =
   match dispatch opt with
   | None -> state
   | Some result -> state_addresult state result

let dispatch_arg dispatch opt arg state =
   match dispatch opt arg with
   | None -> state
   | Some result -> state_addresult state result

let apply_short optch state =
   let opt = String.make 1 optch in
   match state.check opt with
   | None -> FAILURE ("Unknown option -" ^ opt)
   | Some (NOARG dispatch) -> SUCCESS (dispatch_noarg dispatch opt state)
   | Some (HASARG dispatch) ->
        begin match state_takecurarg state with
        | Some (arg, state') -> SUCCESS (dispatch_arg dispatch opt arg state')
        | None ->
             begin match state_getnextarg state with
             | None -> FAILURE ("Option -" ^ opt ^ " expects an argument")
             | Some (arg, state') ->
                  SUCCESS (dispatch_arg dispatch opt arg state')
             end
        end

let apply_long opt state =
   match String.index_opt opt '=' with
   | Some pos ->
        let arg = String.sub opt 0 pos in
        let opt = String.sub opt (pos + 1) (String.length opt - pos - 1) in
        begin match state.check opt with
        | None -> FAILURE ("Unknown option --" ^ opt)
        | Some (NOARG _) ->
             FAILURE ("Option --" ^ opt ^ " expects no argument")
        | Some (HASARG dispatch) ->
             SUCCESS (dispatch_arg dispatch opt arg state)
        end
   | None ->
        begin match state.check opt with
        | None -> FAILURE ("Unknown option --" ^ opt)
        | Some (NOARG dispatch) -> SUCCESS (dispatch_noarg dispatch opt state)
        | Some (HASARG dispatch) ->
             begin match state_getnextarg state with
             | None -> FAILURE ("Option --" ^ opt ^ " expects an argument")
             | Some (arg, state') ->
                  SUCCESS (dispatch_arg dispatch opt arg state')
             end
        end

let apply_any arg state =
   (*
    * The arg may be:
    *    - a non-option word
    *    - a long option beginning with --
    *    - one or more short options beginning with -
    *    - the literal "--" (end of options)
    *    - the literal "-" (treat as a non-option word)
    *)
   if arg = "--" then SUCCESS (state_stop state)
   else if arg = "-" then SUCCESS (state_shiftword state arg)
   else 
     let len = String.length arg in
     if len > 2 && String.sub arg 0 2 = "--" then
        apply_long (String.sub arg 2 (len - 2)) state
     else if len > 1 && String.get arg 0 = '-' then
        SUCCESS (state_setcurarg state (String.sub arg 1 (len - 1)))
     else
        SUCCESS (state_shiftword state arg)
   
let rec examine state =
   match state_getcurarg state with
   | None ->
        begin match state_getnextarg state with
        | None -> SUCCESS state
        | Some (arg, state') ->
             begin match apply_any arg state' with
             | FAILURE msg -> FAILURE msg
             | SUCCESS state'' -> examine state''
             end
        end
   | Some (optch, state') ->
        begin match apply_short optch state' with
        | FAILURE msg -> FAILURE msg
        | SUCCESS state'' -> examine state''
        end

let getopt' check argv =
   let argv' = Array.to_list argv in
   match argv' with
   | [] -> Util.crash "no argv[0]?!?"
   | argv0 :: more ->
        let state = state_init check more in
        match examine state with
        | FAILURE msg -> FAILURE msg
        | SUCCESS state' ->
             let (opts, args) = state_extract state' in
             SUCCESS (opts, argv0 :: args)

let getopt opts argv =
   let map =
      let once m (name, info) = Types.StringMap.add name info m in
      List.fold_left once Types.StringMap.empty opts
   in
   let check opt =
      try
         Some (Types.StringMap.find opt map)
      with Not_found -> None
   in
   getopt' check argv
