%{
(*
 * Copyright (c) 2017, 2018, 2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

open Pos
module T = Spectree

let pos () = Pos.fromparser ()

let bop op e1 e2 =
   T.BOP ((*pos (), XXX*) op, e1, e2)

let uop op e1 =
   T.UOP ((*pos (), XXX*) op, e1)

            (*        opcode              insn field      value *)
type insn = Pos.pos * string * (Pos.pos * string option * T.expr list) list

type sketchentry =
   | ANYTHING
   | CHOICES of insn list

type fileitem =
   | ASSERT of T.expr
   | COUNT of int
   | SKETCH of sketchentry list

let rec ispost e =
   match e with
   | T.FORALL (_, e1) -> ispost e1
   | T.EXISTS (_, e1) -> ispost e1
   | T.LET (e1, _, e2) -> ispost e1 || ispost e2
   | T.BOP (_, e1, e2) -> ispost e1 || ispost e2
   | T.UOP (_, e1) -> ispost e1
   | T.READQUALFUNC (ppstr, _, _) -> ppstr = "post"
   | T.READQUAL (ppstr, _) -> ppstr = "post"
   | T.READVAR _ -> false
   | T.INTCONST _ -> false

let and_list es =
   match es with
   | [] -> T.READVAR "true"
   | e0 :: es' ->
        let build z e = T.BOP (T.LOGAND, z, e) in
        List.fold_left build e0 es'

let buildsketch entries =
   let one_insn (oppos, opcode, operands) =
      let (opcode', fields) = Insns.get_code_and_fields oppos opcode in
      let field_of_operandpos =
         let doadd m (i, f) = Types.IntMap.add i f m in
         List.fold_left doadd Types.IntMap.empty (Util.number fields)
      in
      let operandpos_of_field =
         let doadd m (i, f) = Types.StringMap.add f i m in
         List.fold_left doadd Types.StringMap.empty (Util.number fields)
      in
      let one_operand (results, i) (fpos, field, values) =
         let field', i' =
            match field with
            | Some f -> begin
                 try
                    let _ = Types.StringMap.find f operandpos_of_field in
                    f, i
                 with Not_found ->
                    Util.crash (Pos.string_of_pos fpos ^  ": " ^
                                "Instruction does not use field " ^ f)
              end
            | None -> begin
                 try
                    Types.IntMap.find i field_of_operandpos, i + 1
                 with Not_found ->
                    Util.crash (Pos.string_of_pos fpos ^  ": " ^
                                "Too many operands for instruction")
              end
         in
         ((field', values) :: results, i')
      in
      let (operands', _) = List.fold_left one_operand ([], 0) operands in
      let operands' = List.rev operands' in

      (* check for duplicate fields *)
      let _ =
         let once s (f, _) =
            if Types.StringSet.mem f s then
               Util.crash (Pos.string_of_pos oppos ^ ": Multiple value sets " ^
                           "given for instruction field " ^ f)
            else
               Types.StringSet.add f s
         in
         List.fold_left once Types.StringSet.empty operands'
      in

      (opcode', operands')
   in
   let one_entry entry =
      match entry with
      | ANYTHING -> T.ANYTHING
      | CHOICES insns -> T.CHOICES (List.map one_insn insns)
   in
   List.map one_entry entries

let buildspec items =
   let count = ref 1 in
   let sketch = ref None in
   let pre = ref [] in
   let post = ref [] in
   let handle item =
      match item with
      | ASSERT e -> if ispost e then post := e :: !post else pre := e :: !pre
      | COUNT n -> count := n
      | SKETCH s -> sketch := Some s
   in
   List.iter handle items;
   let (sketch : T.sketch option) =
      match !sketch with
      | None -> None
      | Some s -> Some (buildsketch s)
   in
   ((and_list !pre, and_list !post), !count, sketch)

%}

%token EOF
%token <int> NUMBER
%token <string> /*QSTRING*/ IDENT
/* reserved words */
%token COUNT EXISTS FORALL IN LET SIGN SKETCH UNSIGN WIDEN
/* grouping punctuation */
%token LBRACE RBRACE /*LBRACK RBRACK*/ LPAREN RPAREN
/* multicharacter punctuation */
%token AMPAMP BANGEQ BARBAR CARETCARET /*COLONCOLON*/ DOTDOTDOT EQEQ GTEQ GTGT
%token LTEQ LTLT MINUSGT
/* single-character punctuation */
%token AMP /*AT*/ BANG BAR CARET COLON COMMA DOT EQ GT LT
%token MINUS PCT PLUS SEMIC SLASH
%token STAR TILDE

%type <((Spectree.expr * Spectree.expr) * int * Spectree.sketch option)> file

%start file

%%

file:
   fileitems EOF		{ buildspec (List.rev $1) }
;

fileitems: /* built in reverse order */
     /*nil*/			{ [] }
   | fileitems fileitem		{ $2 :: $1 }
;

fileitem:
     assertion			{ ASSERT $1 }
   | COUNT NUMBER SEMIC		{ COUNT $2 }
   | SKETCH template		{ SKETCH $2 }
;

assertion:
   expr SEMIC			{ $1 }
;

template:
   LBRACE entries RBRACE	{ List.rev $2 }
;

entries: /* built in reverse order */
     entry			{ [$1] }
   | entries entry		{ $2 :: $1 }
;

entry:
     DOTDOTDOT			{ ANYTHING }
   | insn			{ CHOICES [$1] }
   | LBRACE insns RBRACE	{ CHOICES (List.rev $2) }
;

insns: /* build in reverse order */
     insn			{ [$1] }
   | insns insn			{ $2 :: $1 }
;

insn:
     IDENT SEMIC		{ (pos (), $1, []) }
   | IDENT operands SEMIC	{ (pos (), $1, List.rev $2) }
;

operands: /* built in reverse order */
     operand			{ [$1] }
   | operands COMMA operand	{ $3 :: $1 }
;

operand:
     expr			{ (pos (), None, [$1]) }
   | LBRACE exprs RBRACE	{ (pos (), None, List.rev $2) }
   | IDENT EQ expr		{ (pos (), Some $1, [$3]) }
   | IDENT EQ LBRACE exprs RBRACE { (pos (), Some $1, List.rev $4) }
;

exprs: /* built in reverse order */
     expr			{ [$1] }
   | exprs COMMA expr		{ $3 :: $1 }
;


/**************************************************************/

expr:
     implies_expr			{ $1 }
   | implies_tailexpr			{ $1 }
;

implies_expr:
     logor_expr				{ $1 }
   | logor_expr MINUSGT implies_expr	{ bop T.IMPLIES $1 $3 }
;
implies_tailexpr:
     logor_tailexpr			{ $1 }
   | logor_expr MINUSGT implies_tailexpr { bop T.IMPLIES $1 $3 }
;

logor_expr:
     logxor_expr			{ $1 }
   | logor_expr BARBAR logxor_expr	{ bop T.LOGOR $1 $3 }
;
logor_tailexpr:
     logxor_tailexpr			{ $1 }
   | logor_expr BARBAR logxor_tailexpr	{ bop T.LOGOR $1 $3 }
;

logxor_expr:
     logand_expr			{ $1 }
   | logxor_expr CARETCARET logand_expr { bop T.LOGXOR $1 $3 }
;
logxor_tailexpr:
     logand_tailexpr			{ $1 }
   | logxor_expr CARETCARET logand_tailexpr { bop T.LOGXOR $1 $3 }
;

logand_expr:
     bitor_expr				{ $1 }
   | logand_expr AMPAMP bitor_expr	{ bop T.LOGAND $1 $3 }
;
logand_tailexpr:
     bitor_tailexpr			{ $1 }
   | logand_expr AMPAMP bitor_tailexpr	{ bop T.LOGAND $1 $3 }
;

bitor_expr:
     bitxor_expr			{ $1 }
   | bitor_expr BAR bitxor_expr		{ bop T.BITOR $1 $3 }
;
bitor_tailexpr:
     bitxor_tailexpr			{ $1 }
   | bitor_expr BAR bitxor_tailexpr	{ bop T.BITOR $1 $3 }
;

bitxor_expr:
     bitand_expr			{ $1 }
   | bitxor_expr CARET bitand_expr	{ bop T.BITXOR $1 $3 }
;
bitxor_tailexpr:
     bitand_tailexpr			{ $1 }
   | bitxor_expr CARET bitand_tailexpr	{ bop T.BITXOR $1 $3 }
;

bitand_expr:
     equal_expr				{ $1 }
   | bitand_expr AMP equal_expr		{ bop T.BITAND $1 $3 }
;
bitand_tailexpr:
     equal_tailexpr			{ $1 }
   | bitand_expr AMP equal_tailexpr	{ bop T.BITAND $1 $3 }
;

equal_expr:
     rel_expr				{ $1 }
   | equal_expr EQEQ rel_expr		{ bop T.EQ $1 $3 }
   | equal_expr BANGEQ rel_expr		{ bop T.NEQ $1 $3 }
;
equal_tailexpr:
     rel_tailexpr			{ $1 }
   | equal_expr EQEQ rel_tailexpr	{ bop T.EQ $1 $3 }
   | equal_expr BANGEQ rel_tailexpr	{ bop T.NEQ $1 $3 }
;

rel_expr:
     shift_expr				{ $1 }
   | rel_expr LT shift_expr		{ bop T.LT $1 $3 }
   | rel_expr GT shift_expr		{ bop T.GT $1 $3 }
   | rel_expr LTEQ shift_expr		{ bop T.LTEQ $1 $3 }
   | rel_expr GTEQ shift_expr		{ bop T.GTEQ $1 $3 }
;
rel_tailexpr:
     shift_tailexpr			{ $1 }
   | rel_expr LT shift_tailexpr		{ bop T.LT $1 $3 }
   | rel_expr GT shift_tailexpr		{ bop T.GT $1 $3 }
   | rel_expr LTEQ shift_tailexpr	{ bop T.LTEQ $1 $3 }
   | rel_expr GTEQ shift_tailexpr	{ bop T.GTEQ $1 $3 }
;

shift_expr:
     add_expr			{ $1 }
   | shift_expr LTLT add_expr	{ bop T.SHL $1 $3 }
   | shift_expr GTGT add_expr	{ bop T.SHR $1 $3 }
;
shift_tailexpr:
     add_tailexpr		{ $1 }
   | shift_expr LTLT add_tailexpr { bop T.SHL $1 $3 }
   | shift_expr GTGT add_tailexpr { bop T.SHR $1 $3 }
;

add_expr:
     mul_expr			{ $1 }
   | add_expr PLUS mul_expr	{ bop T.ADD $1 $3 }
   | add_expr MINUS mul_expr	{ bop T.SUB $1 $3 }
;
add_tailexpr:
     mul_tailexpr		{ $1 }
   | add_expr PLUS mul_tailexpr	{ bop T.ADD $1 $3 }
   | add_expr MINUS mul_tailexpr { bop T.SUB $1 $3 }
;

mul_expr:
     prefix_expr		{ $1 }
   | mul_expr STAR prefix_expr	{ bop T.MUL $1 $3 }
   | mul_expr SLASH prefix_expr	{ bop T.DIV $1 $3 }
   | mul_expr PCT prefix_expr	{ bop T.MOD $1 $3 }
;
mul_tailexpr:
     prefix_tailexpr		{ $1 }
   | mul_expr STAR prefix_tailexpr { bop T.MUL $1 $3 }
   | mul_expr SLASH prefix_tailexpr { bop T.DIV $1 $3 }
   | mul_expr PCT prefix_tailexpr { bop T.MOD $1 $3 }
;

prefix_expr:
     primary_expr		{ $1 }
   | PLUS prefix_expr		{ $2 }
   | MINUS prefix_expr		{ uop T.NEG $2 }
   | BANG prefix_expr		{ uop T.LOGNOT $2 }
   | TILDE prefix_expr		{ uop T.BITNOT $2 }
   | WIDEN prefix_expr		{ uop T.WIDEN $2 }
   | SIGN prefix_expr		{ uop T.TOSIGNED $2 }
   | UNSIGN prefix_expr		{ uop T.TOUNSIGNED $2 }
;
prefix_tailexpr:
     primary_tailexpr		{ $1 }
   | PLUS prefix_tailexpr	{ $2 }
   | MINUS prefix_tailexpr	{ uop T.NEG $2 }
   | BANG prefix_tailexpr	{ uop T.LOGNOT $2 }
   | TILDE prefix_tailexpr	{ uop T.BITNOT $2 }
   | WIDEN prefix_tailexpr	{ uop T.WIDEN $2 }
   | SIGN prefix_tailexpr	{ uop T.TOSIGNED $2 }
   | UNSIGN prefix_tailexpr	{ uop T.TOUNSIGNED $2 }
;

primary_expr:
     IDENT			{ T.READVAR $1 }
   | IDENT DOT IDENT		{ T.READQUAL ($1, $3) }
   | IDENT DOT IDENT LPAREN expr RPAREN { T.READQUALFUNC ($1, $3, $5) }
   | NUMBER			{ T.INTCONST $1 }
   | LPAREN expr RPAREN		{ $2 }
;

primary_tailexpr:
     LET IDENT EQ expr IN expr	{ T.LET ($4, $2, $6) }
   | FORALL IDENT COMMA expr	{ T.FORALL ($2, $4) }
   | EXISTS IDENT COMMA expr	{ T.EXISTS ($2, $4) }
;

%%
