(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

module I = Mintops
module T = Spectree
module M = Mips
module MS = Mipsstate

(* XXX doesn't this exist somewhere already? *)
let isint ty =
   match ty with
   | M.UINT5
   | M.UINT8
   | M.UINT16
   | M.UINT32
   | M.SINT8
   | M.SINT16
   | M.SINT32 -> true
   | _ -> false

let issigned ty =
   match ty with
   | M.UINT5
   | M.UINT8
   | M.UINT16
   | M.UINT32 -> false
   | M.SINT8
   | M.SINT16
   | M.SINT32 -> true
   | _ -> false

let isdigit c =
   c >= '0' && c <= '9'

let strip_digit_suffix s =
   let rec getsize n =
      if n > 0 && isdigit s.[n - 1] then getsize (n - 1)
      else n
   in
   String.sub s 0 (getsize (String.length s))

(* quick self-tests *)
let _ =
   assert (strip_digit_suffix "x" = "x");
   assert (strip_digit_suffix "x0" = "x");
   assert (strip_digit_suffix "x0a" = "x0a");
   assert (strip_digit_suffix "x0a0" = "x0a");
   assert (strip_digit_suffix "abc123" = "abc");
   assert (strip_digit_suffix "abc123def" = "abc123def");
   assert (strip_digit_suffix "abc123def456" = "abc123def")

let string_of_typename ty = Printlogic.string_of_typename ty

(**************************************************************)
(* builtin symbol data *)

(*
 * XXX these are currently filtered by what processor state is enabled
 * by default, and it should really get all of them and then use what's
 * mentioned to decide what to disable.
 *)

(*
 * These are:
 *    object name, type read from it, constant expr for it, stateelement for it
 *)
let builtin_regs =
   let populate r =
      (MS.string_of_reg r, M.REGISTER, M.REGVALUE r, MS.REG (CONC r))
   in
   List.map populate (Mipstools.all_regs_raw ())

let builtin_cregs =
   let populate r =
      (MS.string_of_creg r, M.CREGISTER, M.CREGVALUE r, MS.CREG (CONC r))
   in
   List.map populate (Mipstools.all_cregs_raw ())

let builtin_fregs =
   let populate r =
      (MS.string_of_freg r, M.FREGISTER, M.FREGVALUE r, MS.FREG (CONC r))
   in
   List.map populate (Mipstools.all_fregs_raw ())

let builtin_fcregs =
   let populate r =
      (MS.string_of_fcreg r, M.FCREGISTER, M.FCREGVALUE r, MS.FCREG (CONC r))
   in
   List.map populate (Mipstools.all_fcregs_raw ())

let builtin_allregs =
   builtin_regs @ builtin_cregs @ builtin_fregs @ builtin_fcregs

(*
 * This is:
 *    object name, cache, and golem variable to use
 * (building the state element and expressions is complicated
 * and can't reasonably be done in advance)
 *)
let builtin_cacherefs =
   let populate c =
      (* don't use string_of_cache as it gives I D L2 L3 *)
      let cname = match c with
         | MS.ICACHE -> "icache"
         | MS.DCACHE -> "dcache"
         | MS.L2CACHE -> "l2cache"
         | MS.L3CACHE -> "l3cache"
      in
      let one iname g =
         let name = cname ^ "_" ^ iname in
         (name, c, g)
      in
      [
         one "vaddr" (MS.G_VADDR c);
         one "index" (MS.G_INDEX c);
      ]
   in
   List.concat (List.map populate (Mipstools.all_caches_raw ()))

let builtin_statesyms =
   let reduce (name, _ty, _e, elt) = (name, elt) in
   List.map reduce builtin_allregs @
   [
      ("hi", MS.HI);
      ("lo", MS.LO);
   ]

let builtin_cachestates =
   let populate cs =
      (MS.string_of_cachestate cs, M.CACHESTATE, M.CACHESTATEVALUE cs)
   in
   List.map populate (Mipstools.all_cachestates_raw ())

let builtin_exprsyms =
   let reduce (name, ty, e, _elt) = (name, ty, e) in
   List.map reduce builtin_allregs @
   [
      ("true", M.BOOL, M.TRUE);
      ("false", M.BOOL, M.FALSE);
      ("pre", M.PROGRAMPOINT, M.PPVALUE (Ppoint.pre));
      ("post", M.PROGRAMPOINT, M.PPVALUE (Ppoint.post));
   ] @
   builtin_cachestates

(**************************************************************)
(* alpha conversion *)

type alphactx = {
   existence: Types.StringSet.t ref;
   conversion: string Types.StringMap.t ref;
   hints: int Types.StringMap.t ref;
}

let newalphactx () =
   let ctx0 = {
      existence = ref Types.StringSet.empty;
      conversion = ref Types.StringMap.empty;
      hints = ref Types.StringMap.empty;
   } in
   let doadd x =
      ctx0.existence := Types.StringSet.add x !(ctx0.existence);
      ctx0.conversion := Types.StringMap.add x x !(ctx0.conversion)
   in
   List.iter (fun (x, _, _) -> doadd x) builtin_exprsyms;
   List.iter (fun (x, _, _) -> doadd x) builtin_cacherefs;
   ctx0

let alpha'gen ctx x =
   let x0 = strip_digit_suffix x in
   if not (Types.StringSet.mem x0 !(ctx.existence)) then x0
   else begin
      let rec dogen hint =
         let x' = x0 ^ string_of_int hint in
         if not (Types.StringSet.mem x' !(ctx.existence)) then begin
             ctx.hints := Types.StringMap.add x0 (hint + 1) !(ctx.hints);
             x'
         end
         else dogen (hint + 1)
      in
      let hint0 =
         try Types.StringMap.find x0 !(ctx.hints)
         with Not_found -> 0
      in
      dogen hint0
   end

let alpha'bind ctx x =
   (* first, if it's not already there, use it directly *)
   let x' =
      if not (Types.StringSet.mem x !(ctx.existence)) then x
      else alpha'gen ctx x
   in
   ctx.existence := Types.StringSet.add x' !(ctx.existence);
   ctx.conversion := Types.StringMap.add x x' !(ctx.conversion);
   x'

let alpha'refer ctx x =
   try
      Types.StringMap.find x !(ctx.conversion)
   with Not_found ->
      Util.crash ("speccomp: alpha'refer: unbound variable " ^ x)

let rec alpha'expr ctx e =
   match e with
   | T.FORALL (x, e1) ->
        let x' = alpha'bind ctx x in
        let e1' = alpha'expr ctx e1 in
        T.FORALL (x', e1')
   | T.EXISTS (x, e1) ->
        let x' = alpha'bind ctx x in
        let e1' = alpha'expr ctx e1 in
        T.EXISTS (x', e1')
   | T.LET (e1, x, e2) ->
        let e1' = alpha'expr ctx e1 in
        let x' = alpha'bind ctx x in
        let e2' = alpha'expr ctx e2 in
        T.LET (e1', x', e2')
   | T.BOP (op, e1, e2) ->
        let e1' = alpha'expr ctx e1 in
        let e2' = alpha'expr ctx e2 in
        T.BOP (op, e1', e2')
   | T.UOP (op, e1) ->
        let e1' = alpha'expr ctx e1 in
        T.UOP (op, e1')
   | T.READQUALFUNC (x, y, arg) ->
        let x' = alpha'refer ctx x in
        let y' = alpha'refer ctx y in
        let arg' = alpha'expr ctx arg in
        T.READQUALFUNC (x', y', arg')
   | T.READQUAL (x, y) ->
        let x' = alpha'refer ctx x in
        let y' = alpha'refer ctx y in
        T.READQUAL (x', y')
   | T.READVAR x ->
        let x' = alpha'refer ctx x in
        T.READVAR x'
   | T.INTCONST k ->
        T.INTCONST k

let alpha'exprs ctx es =
   List.map (alpha'expr ctx) es

let alpha'operand ctx (field, es) =
   (field, alpha'exprs ctx es)

let alpha'insn ctx (opcode, operands) =
   (opcode, List.map (alpha'operand ctx) operands)

let alpha'sketchentry ctx se =
   match se with
   | T.ANYTHING -> T.ANYTHING
   | T.CHOICES insns -> T.CHOICES (List.map (alpha'insn ctx) insns)

let alpha'sketch ctx sketch =
   match sketch with
   | None -> None
   | Some entries -> Some (List.map (alpha'sketchentry ctx) entries)


(**************************************************************)
(* type inference *)

type inftype =
   | KNOWN of M.typename
   | TYVAR of int

(*
 * A type variable can refer to a specific concrete type or to
 *  - another type variable
 *  - a state type (register/cregister/fregister/fcregister)
 *  - an unsigned integer type
 *  - a signed integer type
 *  - any machine integer type
 *  - anything at all
 *
 * Note that type variables should only refer to (be the same as) other
 * type variables whose serial numbers are lower; this prevents cycles.
 *
 * XXX: this is wrong the way it is: it should be able to declare a
 * specific signedness, or refer to the signedness of another tyvar.
 * Also the width declaration should probably be separate.
 *
 * XXX: because widen isn't specific there's probably cases where a
 * sub-32-width int will be unresolved and we have no way to either
 * detect or handle this.
 *)
type tyvarinfo =
   | TV_CONCRETE of M.typename
   | TV_ALIAS of int
   | TV_STATE
   | TV_UINT of int (* minimum width *)
   | TV_SINT of int (* minimum width *)
   | TV_ANYINT of int (* minimum width *)
   | TV_ANY

type infctx = {
   vars: inftype Types.StringMap.t ref;
   funcs: (inftype * inftype) Types.StringMap.t ref;
   intvals: inftype Types.IntMap.t ref;
   nextintval: int ref;

   tyvars: tyvarinfo Types.IntMap.t ref;
   nexttyvar: int ref;
}

let newinfctx () =
   let ctx0 = {
      vars = ref Types.StringMap.empty;
      funcs = ref Types.StringMap.empty;
      intvals = ref Types.IntMap.empty;
      nextintval = ref 0;

      tyvars = ref Types.IntMap.empty;
      nexttyvar = ref 0;
   } in
   let doadd (x, ty, _e) =
      ctx0.vars := Types.StringMap.add x (KNOWN ty) !(ctx0.vars)
   in
   List.iter doadd builtin_exprsyms;
   let addfunc (x, _, _) =
      let v = (KNOWN M.UINT32, KNOWN M.CACHESTATE) in
      ctx0.funcs := Types.StringMap.add x v !(ctx0.funcs)
   in
   List.iter addfunc builtin_cacherefs;
   ctx0

let string_of_tyvarinfo info =
   match info with
   | TV_CONCRETE ty -> string_of_typename ty
   | TV_ALIAS n -> "<alias of type " ^ string_of_int n ^ ">"
   | TV_STATE -> "<state>"
   | TV_UINT n -> "<uint " ^ string_of_int n ^ ">"
   | TV_SINT n -> "<sint " ^ string_of_int n ^ ">"
   | TV_ANYINT n -> "<anyint " ^ string_of_int n ^ ">"
   | TV_ANY -> "<any>"

let inf'fresh' ctx =
   let n = !(ctx.nexttyvar) in
   ctx.nexttyvar := n + 1;
   ctx.tyvars := Types.IntMap.add n TV_ANY !(ctx.tyvars);
   n

let inf'fresh ctx = TYVAR (inf'fresh' ctx)

let inf'bind ctx x ty =
   ctx.vars := Types.StringMap.add x ty !(ctx.vars)

let string_of_inftype ctx ty =
   match ty with
   | KNOWN ty' -> string_of_typename ty'
   | TYVAR n -> string_of_tyvarinfo (Types.IntMap.find n !(ctx.tyvars))

let rec inf'refinestate ctx ty =
   let mismatch () =
      Util.crash ("Type mismatch: " ^ string_of_inftype ctx ty ^
		  " vs. " ^ string_of_tyvarinfo TV_STATE)
   in
   match ty with
   | KNOWN ty' -> begin
        match ty' with
        | M.REGISTER
        | M.CREGISTER
        | M.FREGISTER
        | M.FCREGISTER -> ()
        | _ -> mismatch ()
     end
   | TYVAR n ->
        match Types.IntMap.find n !(ctx.tyvars) with
        | TV_CONCRETE ty' -> inf'refinestate ctx (KNOWN ty')
        | TV_ALIAS n' -> inf'refinestate ctx (TYVAR n')
        | TV_STATE -> ()
        | TV_UINT _
        | TV_SINT _
        | TV_ANYINT _ -> mismatch ()
        | TV_ANY -> ctx.tyvars := Types.IntMap.add n TV_STATE !(ctx.tyvars)

let rec inf'refineanyint ctx ty w =
   let mismatch () =
      Util.crash ("Type mismatch: " ^ string_of_inftype ctx ty ^
		  " vs. " ^ string_of_tyvarinfo (TV_ANYINT w))
   in
   match ty with
   | KNOWN ty' -> begin
        let toowide what w =
           Util.crash ("Width of this type exceeds " ^ what ^ " " ^
                       string_of_int w ^ " bits")
        in
        match ty' with
	| M.UINT5 -> if w > 5 then toowide "unsigned" 5 else ()
	| M.UINT8 -> if w > 8 then toowide "unsigned" 8 else ()
	| M.UINT16 -> if w > 16 then toowide "unsigned" 16 else ()
	| M.UINT32 -> if w > 32 then toowide "unsigned" 32 else ()
	| M.SINT8 -> if w > 7 then toowide "signed" 8 else ()
	| M.SINT16 -> if w > 15 then toowide "signed" 16 else ()
	| M.SINT32 -> if w > 31 then toowide "signed" 32 else ()
        | _ -> mismatch ()
     end
   | TYVAR n ->
        let improve ty' =
           ctx.tyvars := Types.IntMap.add n ty' !(ctx.tyvars)
        in
        match Types.IntMap.find n !(ctx.tyvars) with
        | TV_CONCRETE ty' -> inf'refineanyint ctx (KNOWN ty') w
        | TV_ALIAS n' -> inf'refineanyint ctx (TYVAR n') w
        | TV_STATE -> mismatch ()
        | TV_UINT w' -> if w > w' then improve (TV_UINT w) else ()
        | TV_SINT w' -> if w + 1 > w' then improve (TV_SINT (w + 1)) else ()
        | TV_ANYINT w' -> if w > w' then improve (TV_ANYINT w) else ()
        | TV_ANY -> improve (TV_ANYINT w)

let inf'combineinfo info1 info2 =
   let mismatch () =
      Util.crash ("Type mismatch: " ^ string_of_tyvarinfo info1 ^
		  " vs. " ^ string_of_tyvarinfo info2)
   in
   let isstate ty =
      match ty with
      | M.REGISTER
      | M.CREGISTER
      | M.FREGISTER
      | M.FCREGISTER -> true
      | _ -> false
   in
   let isuint w ty =
      match ty with
      | M.UINT5 -> w <= 5
      | M.UINT8 -> w <= 8
      | M.UINT16 -> w <= 16
      | M.UINT32 -> w <= 32
      | _ -> false
   in
   let issint w ty =
      match ty with
      | M.SINT8 -> w <= 8
      | M.SINT16 -> w <= 16
      | M.SINT32 -> w <= 32
      | _ -> false
   in
   let isanyint w ty = isuint w ty || issint (w + 1) ty in
   let req p ty = if p ty then () else mismatch () in
   let reqstate = req isstate in
   let requint w = req (isuint w) in
   let reqsint w = req (issint w) in
   let reqanyint w = req (isanyint w) in

   if info1 = info2 then info1 else
   match info1, info2 with
   | TV_CONCRETE _, TV_CONCRETE _ -> mismatch ()
   | TV_CONCRETE ty1, TV_STATE -> reqstate ty1; info1
   | TV_CONCRETE ty1, TV_UINT w -> requint w ty1; info1
   | TV_CONCRETE ty1, TV_SINT w -> reqsint w ty1; info1
   | TV_CONCRETE ty1, TV_ANYINT w -> reqanyint w ty1; info1
   | TV_CONCRETE _, TV_ANY -> info1
   | TV_STATE, TV_CONCRETE ty2 -> reqstate ty2; info2
   | TV_UINT w, TV_CONCRETE ty2 -> requint w ty2; info2
   | TV_SINT w, TV_CONCRETE ty2 -> reqsint w ty2; info2
   | TV_ANYINT w, TV_CONCRETE ty2 -> reqanyint w ty2; info2
   | TV_ANY, TV_CONCRETE _ -> info2
   | TV_UINT w1, TV_ANYINT w2 -> TV_UINT (max w1 w2)
   | TV_SINT w1, TV_ANYINT w2 -> TV_SINT (max w1 (w2 + 1))
   | TV_ANYINT w1, TV_UINT w2 -> TV_UINT (max w1 w2)
   | TV_ANYINT w1, TV_SINT w2 -> TV_SINT (max (w1 + 1) w2)
   | TV_ALIAS _, _
   | _, TV_ALIAS _ ->
        (* should not happen *)
        Util.crash "speccomp: inf'combineinfo: unexpected alias"
   | _, _ -> mismatch ()
        

(*
 * ty1 is always the concrete constraint (when there is one)
 *)
let inf'unify ctx ty1 ty2 =
   let resolve n ty =
      let info = Types.IntMap.find n !(ctx.tyvars) in
      let info' = inf'combineinfo (TV_CONCRETE ty) info in
      ctx.tyvars := Types.IntMap.add n info' !(ctx.tyvars)
   in
   match ty1, ty2 with
   | KNOWN ty1', KNOWN ty2' when ty1' = ty2' -> ()
   | KNOWN ty1', KNOWN ty2' ->
        Util.crash ("Type mismatch: " ^ string_of_typename ty1' ^
                    " vs. " ^ string_of_typename ty2')
   | KNOWN ty1', TYVAR n2 -> resolve n2 ty1'
   | TYVAR n1, KNOWN ty2'  -> resolve n1 ty2'
   | TYVAR n1, TYVAR n2 ->
        let info1 = Types.IntMap.find n1 !(ctx.tyvars) in
        let info2 = Types.IntMap.find n2 !(ctx.tyvars) in
        let info12 = inf'combineinfo info1 info2 in
        if n1 < n2 then begin
           ctx.tyvars := Types.IntMap.add n2 (TV_ALIAS n1) !(ctx.tyvars);
           ctx.tyvars := Types.IntMap.add n1 info12 !(ctx.tyvars)
        end else begin
           ctx.tyvars := Types.IntMap.add n1 (TV_ALIAS n2) !(ctx.tyvars);
           ctx.tyvars := Types.IntMap.add n2 info12 !(ctx.tyvars)
        end

let rec inf'simplify ctx ty =
   match ty with
   | KNOWN _ -> ty
   | TYVAR n ->
        match Types.IntMap.find n !(ctx.tyvars) with
        | TV_CONCRETE ty' -> KNOWN ty'
        | TV_ALIAS n' -> inf'simplify ctx (TYVAR n')
        | _ -> TYVAR n

let inf'var ctx x =
   try
      inf'simplify ctx (Types.StringMap.find x !(ctx.vars))
   with Not_found ->
      Util.crash ("speccomp: inf'expr: unbound variable " ^ x)

let inf'func ctx f ty'arg =
   let (expected'arg, ty'ret) =
      try
         Types.StringMap.find f !(ctx.funcs)
      with Not_found ->
         Util.crash ("speccomp: inf'expr: unbound function " ^ f)
   in
   inf'unify ctx expected'arg ty'arg;
   ty'ret

let inf'bop ctx op ty1 ty2 =
   match op with
   | T.IMPLIES
   | T.LOGOR
   | T.LOGXOR
   | T.LOGAND ->
        inf'unify ctx (KNOWN M.BOOL) ty1;
        inf'unify ctx (KNOWN M.BOOL) ty2;
        KNOWN M.BOOL
   | T.BITOR
   | T.BITXOR
   | T.BITAND ->
        inf'unify ctx ty1 ty2;
        inf'simplify ctx ty1
   | T.EQ
   | T.NEQ ->
        inf'unify ctx ty1 ty2;
        KNOWN M.BOOL
   | T.LT
   | T.LTEQ
   | T.GT
   | T.GTEQ ->
        inf'unify ctx ty1 ty2;
        let ty1' = inf'simplify ctx ty1 in
        inf'refineanyint ctx ty1' 0;
        KNOWN M.BOOL
   | T.SHL
   | T.SHR ->
        inf'refineanyint ctx ty1 0;
        inf'unify ctx (KNOWN M.UINT5) ty2;
        inf'simplify ctx ty1
   | T.ADD
   | T.SUB
   | T.MUL
   | T.DIV
   | T.MOD ->
        inf'unify ctx ty1 ty2;
        let ty1' = inf'simplify ctx ty1 in
        inf'refineanyint ctx ty1' 0;
        inf'simplify ctx ty1'

let inf'uop ctx op ty1 =
   match op with
   | T.LOGNOT ->
        inf'unify ctx (KNOWN M.BOOL) ty1;
        KNOWN M.BOOL
   | T.BITNOT ->
        inf'refineanyint ctx ty1 0;
        inf'simplify ctx ty1
   | T.NEG ->
        inf'refineanyint ctx ty1 0;
        inf'simplify ctx ty1
   | T.WIDEN -> begin
        inf'refineanyint ctx ty1 0;
        let ty1' = inf'simplify ctx ty1 in
        match ty1' with
        | KNOWN M.UINT5
        | KNOWN M.UINT8
        | KNOWN M.UINT16 -> KNOWN M.UINT32
        | KNOWN M.SINT8
        | KNOWN M.SINT16 -> KNOWN M.SINT32
        | KNOWN M.UINT32
        | KNOWN M.SINT32 -> Util.crash ("misplaced widen (already 32 bit)")
        | KNOWN _ -> assert false (* already rejected by refineanyint *)
        | TYVAR n ->
             let any () =
                let ty' = inf'fresh ctx in
                inf'refineanyint ctx ty' 32;
                ty'
             in
             match Types.IntMap.find n !(ctx.tyvars) with
             | TV_CONCRETE _
             | TV_ALIAS _ -> assert false (* impossible *) 
             | TV_STATE -> assert false (* already rejected by refineanyint *)
             | TV_UINT _ -> KNOWN M.UINT32
             | TV_SINT _ -> KNOWN M.SINT32
             | TV_ANYINT w -> if w >= 32 then TYVAR n else any ()
             | TV_ANY -> any ()
     end
   | T.TOSIGNED ->
        inf'unify ctx (KNOWN M.UINT32) ty1;
        KNOWN M.SINT32
   | T.TOUNSIGNED ->
        inf'unify ctx (KNOWN M.SINT32) ty1;
        KNOWN M.UINT32

let rec inf'expr ctx e =
   let req'known ty =
      match ty with
      | KNOWN _ -> ()
      | TYVAR n ->
           Util.crash ("Cannot infer a type for tyvar " ^ string_of_int n)
   in
   match e with
   | T.FORALL (x, e1)
   | T.EXISTS (x, e1) ->
        let ty'x = inf'fresh ctx in
        inf'bind ctx x ty'x;
        let ty'e1 = inf'expr ctx e1 in
        inf'unify ctx (KNOWN M.BOOL) ty'e1;
        let ty'x = inf'simplify ctx ty'x in
        req'known ty'x;
        KNOWN M.BOOL
   | T.LET (e1, x, e2) ->
        let ty'e1 = inf'expr ctx e1 in
        inf'bind ctx x ty'e1;
        let ty'e2 = inf'expr ctx e2 in
        let ty'e2 = inf'simplify ctx ty'e2 in
        if ty'e1 = ty'e2 then ()
        else req'known ty'e1;
        ty'e2
   | T.BOP (op, e1, e2) ->
        let ty'e1 = inf'expr ctx e1 in
        let ty'e2 = inf'expr ctx e2 in
        inf'simplify ctx (inf'bop ctx op ty'e1 ty'e2)
   | T.UOP (op, e1) ->
        let ty'e1 = inf'expr ctx e1 in
        inf'simplify ctx (inf'uop ctx op ty'e1)
   | T.READQUALFUNC (x, y, arg) ->
        let ty'x = inf'var ctx x in
        let ty'arg = inf'expr ctx arg in
        inf'unify ctx (KNOWN M.PROGRAMPOINT) ty'x;
        inf'func ctx y ty'arg
   | T.READQUAL (x, y) ->
        let ty'x = inf'var ctx x in
        let ty'y = inf'var ctx y in
        inf'unify ctx (KNOWN M.PROGRAMPOINT) ty'x;
        inf'refinestate ctx ty'y;
        (*inf'simplify ctx ty'y*)
        KNOWN M.UINT32
   | T.READVAR x ->
        (* XXX: this should restrict the type to not be machine state *)
        inf'var ctx x
   | T.INTCONST k ->
        let ty = inf'fresh ctx in
        inf'refineanyint ctx ty k;
        let ty' = inf'simplify ctx ty in
        let id = !(ctx.nextintval) in
        ctx.nextintval := id + 1;
        ctx.intvals := Types.IntMap.add id ty' !(ctx.intvals);
        ty'

let inf'tyexprs ctx required'ty es =
   let tys'es = List.map (inf'expr ctx) es in
   let once ty =
      inf'unify ctx (KNOWN required'ty) ty
   in
   List.iter once tys'es

let inf'boolexprs ctx es =
   inf'tyexprs ctx M.BOOL es

let inf'operand ctx (field, es) =
   (*
    * XXX this table of field names should be in insns.ml or something
    *)
   let ty =
      match field with
      | "rd" -> M.REGISTER
      | "cd" -> M.CREGISTER
      | "fd" -> M.FREGISTER
      | "fcd" -> M.FCREGISTER
      | "rs" -> M.REGISTER
      | "cs" -> M.CREGISTER
      | "fs" -> M.FREGISTER
      | "fcs" -> M.FCREGISTER
      | "rt" -> M.REGISTER
      | "simm16" -> M.SINT16
      | "uimm16" -> M.UINT16
      | "uimm5" -> M.UINT5
(*
      | "mem" -> _
*)
      | "xcl"
      | "xdl"
      | "lcl"
      | "ldl" -> Util.crash "speccomp: sketching with labels doesn't work yet"
      | "whichcache" -> Util.crash "speccomp: cannot sketch cache ops"
      | _ -> Util.crash ("speccomp: Invalid instruction field " ^ field)
   in
   inf'tyexprs ctx ty es

let inf'sketchinsn ctx (_opcode, operands) =
   (*
    * XXX we should check that the operands are legal for the opcode,
    * instead of crashing later, but this is a pain given what we can
    * look up in Insns right now, so skip it. But do check that each
    * operand has the right type for its instruction field.
    *)
   List.iter (inf'operand ctx) operands

let inf'sketchentry ctx se =
   match se with
   | T.ANYTHING -> ()
   | T.CHOICES insns -> List.iter (inf'sketchinsn ctx) insns

let inf'sketch ctx sketch =
   match sketch with
   | None -> ()
   | Some entries -> List.iter (inf'sketchentry ctx) entries

let finishinfctx ctx =
   let polish ty =
      match inf'simplify ctx ty with
      | KNOWN ty' -> ty'
      | TYVAR n ->
           Util.crash ("Leftover unresolved type variable " ^ string_of_int n)
   in
   let vars = Types.StringMap.map polish !(ctx.vars) in
   let intvals = Types.IntMap.map polish !(ctx.intvals) in
   (vars, intvals)
      

(**************************************************************)
(* compile to M.expr *)

type cctx = {
   vars: M.typename Types.StringMap.t ref;
   intvals: M.typename Types.IntMap.t ref;
   nextintval: int ref;
   builtinexprs: M.expr Types.StringMap.t ref;
   builtinelts: MS.stateelement Types.StringMap.t ref;
   builtinfuncs: (MS.cache * MS.golems) Types.StringMap.t ref;
}

let newcctx vars intvals =
   let ctx0 = {
      vars = ref vars;
      intvals = ref intvals;
      nextintval = ref 0;
      builtinexprs = ref Types.StringMap.empty;
      builtinelts = ref Types.StringMap.empty;
      builtinfuncs = ref Types.StringMap.empty;
   } in
   let doadd (x, ty, e) =
      ctx0.vars := Types.StringMap.add x ty !(ctx0.vars);
      ctx0.builtinexprs := Types.StringMap.add x e !(ctx0.builtinexprs);
   in
   List.iter doadd builtin_exprsyms;
   let addelt (x, elt) =
      ctx0.builtinelts := Types.StringMap.add x elt !(ctx0.builtinelts);
   in
   List.iter addelt builtin_statesyms;
   let addfunc (x, c, g) =
      ctx0.builtinfuncs := Types.StringMap.add x (c, g) !(ctx0.builtinfuncs);
   in
   List.iter addfunc builtin_cacherefs;
   ctx0

let c'builtin (ctx: cctx) x =
   let ty'x = Types.StringMap.find x !(ctx.vars) in
   try
      let e = Types.StringMap.find x !(ctx.builtinexprs) in
(*
      let elt =
         try
             Some (Types.StringMap.find x !(ctx.builtinelts))
         with Not_found -> None
      in
*)
      (ty'x, e (*, elt*))
   with Not_found ->
      (ty'x, M.READVAR (M.PLAINVAR x, ty'x) (*, None*))

let c'cacheref (ctx: cctx) pp f arg =
   let (c, g) =
      try
         Types.StringMap.find f !(ctx.builtinfuncs)
      with Not_found ->
         (* should have already been rejected *)
         Util.crash "speccomp: c'cacheref: missing func"
   in
   let blue = MS.CACHE_BLUE in
   let green = MS.CACHE_GREEN in
   let make ce = M.WHICHCACHEENTRYVALUE (c, ce) in
   let readblue = M.READSTATE (make blue, pp, Z_ANY) in
   let readgreen = M.READSTATE (make green, pp, Z_ANY) in
   let cond = M.GOLEMIZE (M.READGOLEM (M.GOLEMVALUE g), arg) in
   (M.CACHESTATE, M.IF (cond, readblue, readgreen))

let rec c'bop ctx op ty'e1 e1 ty'e2 e2 =
   let not op' =
      let (ty, e') = c'bop ctx op' ty'e1 e1 ty'e2 e2 in
      (ty, M.LOGNOT e')
   in
   match op with
   | T.IMPLIES ->
        (ty'e1, M.IMPLIES (e1, e2))
   | T.LOGOR ->
        (ty'e1, M.LOGOR (e1, e2))
   | T.LOGXOR ->
        (ty'e1, M.LOGXOR (e1, e2))
   | T.LOGAND ->
        (ty'e1, M.LOGAND (e1, e2))
   | T.BITOR ->
        (ty'e1, M.MINTBOP (I.BITOR, e1, e2))
   | T.BITXOR ->
        (ty'e1, M.MINTBOP (I.BITXOR, e1, e2))
   | T.BITAND ->
        (ty'e1, M.MINTBOP (I.BITAND, e1, e2))
   | T.EQ ->
        let e' =
           if isint ty'e1 then M.MINTBOP (I.EQ, e1, e2)
           else M.LOGEQ (e1, e2)
        in
        (M.BOOL, e')
   | T.NEQ ->
        not T.EQ
   | T.LT ->
        let op' = if issigned ty'e1 then I.SLT else I.ULT in
        (M.BOOL, M.MINTBOP (op', e1, e2))
   | T.LTEQ ->
        not T.GT
   | T.GT ->
        let op' = if issigned ty'e1 then I.SLT else I.ULT in
        (M.BOOL, M.MINTBOP (op', e2, e1))
   | T.GTEQ ->
        not T.LT
   | T.SHL ->
        (ty'e1, M.MINTBOP (I.SHL, e1, e2))
   | T.SHR ->
        let op' = if issigned ty'e1 then I.SSHR else I.USHR in
        (ty'e1, M.MINTBOP (op', e1, e2))
   | T.ADD ->
        (ty'e1, M.MINTBOP (I.ADD, e1, e2))
   | T.SUB ->
        (ty'e1, M.MINTBOP (I.SUB, e1, e2))
   | T.MUL ->
        (* XXX need a way to access MULHI... *)
        (ty'e1, M.MINTBOP (I.MUL_LO, e1, e2))
   | T.DIV ->
        let op' = if issigned ty'e1 then I.SDIV else I.UDIV in
        (ty'e1, M.MINTBOP (op', e1, e2))
   | T.MOD ->
        let op' = if issigned ty'e1 then I.SMOD else I.UMOD in
        (ty'e1, M.MINTBOP (op', e1, e2))

let c'uop _ctx op ty'e1 e1 =
   match op with
   | T.LOGNOT -> (ty'e1, M.LOGNOT e1)
   | T.BITNOT -> (ty'e1, M.MINTUOP (I.BITNOT, e1))
   | T.NEG ->
        let z = M.INTVALUE (0, ty'e1) in
        (ty'e1, M.MINTBOP (I.SUB, z, e1))
   | T.WIDEN ->
        let ty', op' = match ty'e1 with
           | M.UINT5 -> M.UINT32, I.U5WIDEN
           | M.UINT8 -> M.UINT32, I.UBWIDEN
           | M.UINT16 -> M.UINT32, I.UHWIDEN
           | M.SINT8 -> M.SINT32, I.SBWIDEN
           | M.SINT16 -> M.SINT32, I.SHWIDEN
           | _ -> Util.crash "speccomp: c'uop: bad widen"
        in
        (ty', M.MINTUOP (op', e1))
   | T.TOSIGNED ->
        (M.SINT32, M.MINTUOP (I.TOSIGNED, e1))
   | T.TOUNSIGNED ->
        (M.UINT32, M.MINTUOP (I.TOUNSIGNED, e1))

let rec c'expr (ctx: cctx) e =
   match e with
   | T.FORALL (x, e1) ->
        let ty'x = Types.StringMap.find x !(ctx.vars) in
        let (ty'e1, e1') = c'expr ctx e1 in
        (ty'e1, M.FORALL (M.PLAINVAR x, ty'x, e1'))
   | T.EXISTS (x, e1) ->
        let ty'x = Types.StringMap.find x !(ctx.vars) in
        let (ty'e1, e1') = c'expr ctx e1 in
        (ty'e1, M.EXISTS (M.PLAINVAR x, ty'x, e1'))
   | T.LET (e1, x, e2) ->
        let (_ty'e1, e1') = c'expr ctx e1 in
        let ty'x = Types.StringMap.find x !(ctx.vars) in
        let (ty'e2, e2') = c'expr ctx e2 in
        (ty'e2, M.LET (M.PLAINVAR x, ty'x, e1', e2'))
   | T.BOP (op, e1, e2) ->
        let (ty'e1, e1') = c'expr ctx e1 in
        let (ty'e2, e2') = c'expr ctx e2 in
        c'bop ctx op ty'e1 e1' ty'e2 e2'
   | T.UOP (op, e1) ->
        let (ty'e1, e1') = c'expr ctx e1 in
        c'uop ctx op ty'e1 e1'
   | T.READQUALFUNC (x, y, arg) ->
        let (_, ppe) = c'builtin ctx x in
        let pp = match ppe with
           | M.PPVALUE pp -> pp
           | M.READVAR _ ->
                (* XXX: do we need to be able to do this? *)
                Util.crash "speccomp: c'expr: state at symbolic program point"
           | _ -> Util.crash "speccomp: c'expr: invalid ppe"
        in
        let (_ty'arg', arg') = c'expr ctx arg in
        c'cacheref ctx pp y arg'
   | T.READQUAL (x, y) ->
        let (_, ppe) = c'builtin ctx x in
        let pp = match ppe with
           | M.PPVALUE pp -> pp
           | M.READVAR _ ->
                (* XXX: do we need to be able to do this? *)
                Util.crash "speccomp: c'expr: state at symbolic program point"
           | _ -> Util.crash "speccomp: c'expr: invalid ppe"
        in
        let (_ty'elt, e' (*, elt*)) = c'builtin ctx y in
(*
        let ty'elt', e'' = match elt with
           | Some elt -> M.UINT32, M.READSTATE (elt, pp, Z_ANY)
           | None -> ty'elt, e'
        in
*)
        let ty'elt' = M.UINT32 in
        let e'' = M.READSTATE (e', pp, Z_ANY) in
        
        (ty'elt', e'')
   | T.READVAR x ->
        let (ty, e) = c'builtin ctx x in
        (ty, e)
   | T.INTCONST k ->
        (* this relies on the traversals being in the same order *)
        let id = !(ctx.nextintval) in
        ctx.nextintval := id + 1;
        let ty'k = Types.IntMap.find id !(ctx.intvals) in
        (ty'k, M.INTVALUE (k, ty'k))

let c'exprs ctx es =
   let once e =
      let (_, e') = c'expr ctx e in e'
   in
   List.map once es

let c'sketchoperand ctx (field, es) =
   (field, c'exprs ctx es)

let c'sketchinsn ctx (opcode, operands) =
   (opcode, List.map (c'sketchoperand ctx) operands)

let c'sketchentry ctx se =
   match se with
   | T.ANYTHING -> M.ANYTHING
   | T.CHOICES insns -> M.CHOICES (List.map (c'sketchinsn ctx) insns)

let c'sketch ctx sketch =
   match sketch with
   | None -> None
   | Some entries -> Some (List.map (c'sketchentry ctx) entries)


(**************************************************************)
(* remove ifs *)

type deify_result =
| NOIF of M.expr
| ISIF of (M.expr * M.expr) list

type nonbool_mode = NOCOND | DOCOND

let deify_iflift ctor r =
   match r with
   | NOIF e -> NOIF (ctor e)
   | ISIF cases -> ISIF (List.map (fun (c, e) -> (c, ctor e)) cases)

let deify_iflift_withcond ctor r =
   match r with
   | NOIF e -> NOIF (ctor e)
   | ISIF cases -> ISIF (List.map (fun (c, e) -> (ctor c, ctor e)) cases)

let deify_iflift_2 ctor r1 r2 =
   match r1, r2 with
   | NOIF e1, NOIF e2 -> NOIF (ctor e1 e2)
   | ISIF cases1, NOIF e2 ->
        let ins (c, e) = (c, ctor e e2) in
        ISIF (List.map ins cases1)
   | NOIF e1, ISIF cases2 ->
        let ins (c, e) = (c, ctor e1 e) in
        ISIF (List.map ins cases2)
   | ISIF cases1, ISIF cases2 ->
        let cases = Util.cartesian cases1 cases2 in
        let ins ((c1, e1), (c2, e2)) = (M.LOGAND (c1, c2), ctor e1 e2) in
        ISIF (List.map ins cases)

let deify_condlift c0 r =
   match r with
   | NOIF e -> [(c0, e)]
   | ISIF cases -> List.map (fun (c, e) -> (M.LOGAND (c0, c), e)) cases

let deify_finish r =
   match r with
   | NOIF e -> e
   | ISIF cases -> M.and_list (List.map (fun (c, e) -> M.IMPLIES (c, e)) cases)

let rec deify'nonbool e0 =
   match e0 with
     (* constants *)
   | M.TRUE
   | M.FALSE -> NOIF e0
     (* constructors *)
   | M.CACHESTATEVALUE _
   | M.INTVALUE _
   | M.REGVALUE _
   | M.CREGVALUE _
   | M.FREGVALUE _
   | M.FCREGVALUE _
   | M.HIREGVALUE
   | M.LOREGVALUE
   | M.GOLEMVALUE _
   | M.WHICHCACHEVALUE _
   | M.WHICHCACHEENTRYIDVALUE _
   | M.WHICHCACHEENTRYVALUE _
   | M.PPVALUE _
   | M.INSNVALUE _ -> NOIF e0
     (* access to processor state, as of a particular program point *)
   | M.READSTATE _
   | M.READGOLEM _
(*
   | M.READALLSTATE _ -> NOIF e0
   | M.UPDATEMAP _ -> Util.crash "speccomp: deify: updatemap unimplemented"
*)
     (* access to logic variables *)
   | M.READVAR _ -> NOIF e0
     (* access to logic functions *)
   | M.GOLEMIZE (g, e1) ->
        (*let g' = deify'nonbool g in*) (* XXX *)
        let e1' = deify'nonbool e1 in
        let e0' = deify_iflift (fun e -> GOLEMIZE (g, e)) e1' in
        NOIF (deify_finish e0')
     (* logical operators *)
   | M.FORALL (x, ty, e1) ->
        let e1' = deify'bool e1 in
        NOIF (FORALL (x, ty, e1'))
   | M.EXISTS (x, ty, e1) ->
        let e1' = deify'bool e1 in
        NOIF (EXISTS (x, ty, e1'))
   | M.LET (x, BOOL, e1, e2) ->
        let e1' = deify'bool e1 in
        let e2' = deify'nonbool e2 in
        deify_iflift_withcond (fun e -> LET (x, BOOL, e1', e)) e2'
   | M.LET (x, ty, e1, e2) ->
        let e1' = deify'nonbool e1 in
        let e2' = deify'nonbool e2 in
        begin
           match e1', e2' with
           | NOIF e1'', NOIF e2'' -> NOIF (M.LET (x, ty, e1'', e2''))
           | NOIF e1'', ISIF _ ->
                deify_iflift_withcond (fun e -> M.LET (x, ty, e1'', e)) e2'
           | ISIF _, NOIF e2'' ->
                deify_iflift (fun e -> M.LET (x, ty, e, e2'')) e1'
           | ISIF _, ISIF _ ->
		(* XXX implement this (the composition below doesn't work) *)
		Util.crash "speccomp: deify: double nonbool let unimplemented"
(*
                let ctor ea =
                   let ctor2 eb = M.LET (x, ty, ea, eb) in
                   deify_iflift_withcond ctor2 e2'
                in
                deify_iflift ctor e1'
*)
        end
   | M.IF (c, t, f) ->
        let c' = deify'bool c in
        let t' = deify'nonbool t in
        let f' = deify'nonbool f in
        let tcases = deify_condlift c' t' in
        let fcases = deify_condlift (LOGNOT c') f' in
        ISIF (tcases @ fcases)
   | M.IMPLIES (e1, e2) ->
        let e1' = deify'bool e1 in
        let e2' = deify'bool e2 in
        NOIF (M.IMPLIES (e1', e2'))
   | M.LOGNOT e1 ->
        let e1' = deify'bool e1 in
        NOIF (M.LOGNOT e1')
   | M.LOGAND (e1, e2) ->
        let e1' = deify'bool e1 in
        let e2' = deify'bool e2 in
        NOIF (M.LOGAND (e1', e2'))
   | M.LOGOR (e1, e2) ->
        let e1' = deify'bool e1 in
        let e2' = deify'bool e2 in
        NOIF (M.LOGOR (e1', e2'))
   | M.LOGXOR (e1, e2) ->
        let e1' = deify'bool e1 in
        let e2' = deify'bool e2 in
        NOIF (M.LOGXOR (e1', e2'))
   | M.LOGEQ (e1, e2) ->
        let e1' = deify'nonbool e1 in
        let e2' = deify'nonbool e2 in
        let e0' = deify_iflift_2 (fun ea eb -> LOGEQ (ea, eb)) e1' e2' in
        NOIF (deify_finish e0')
     (* other operators *)
   | M.MINTUOP (op, e1) ->
        let e1' = deify'nonbool e1 in
        deify_iflift (fun e -> MINTUOP (op, e)) e1'
   | M.MINTBOP (op, e1, e2) ->
        let e1' = deify'nonbool e1 in
        let e2' = deify'nonbool e2 in
        deify_iflift_2 (fun ea eb -> MINTBOP (op, ea, eb)) e1' e2'

and deify'bool e0 =
   match e0 with
     (* constants *)
   | M.TRUE
   | M.FALSE -> e0
     (* constructors *)
   | M.CACHESTATEVALUE _
   | M.INTVALUE _
   | M.REGVALUE _
   | M.CREGVALUE _
   | M.FREGVALUE _
   | M.FCREGVALUE _
   | M.HIREGVALUE
   | M.LOREGVALUE
   | M.GOLEMVALUE _
   | M.WHICHCACHEVALUE _
   | M.WHICHCACHEENTRYIDVALUE _
   | M.WHICHCACHEENTRYVALUE _
   | M.PPVALUE _
   | M.INSNVALUE _ -> e0
     (* access to processor state, as of a particular program point *)
   | M.READSTATE _
   | M.READGOLEM _
(*
   | M.READALLSTATE _ -> e0
   | M.UPDATEMAP _ -> Util.crash "speccomp: deify: updatemap unimplemented"
*)
     (* access to logic variables *)
   | M.READVAR _ -> e0
     (* access to logic functions *)
   | M.GOLEMIZE (g, e1) ->
        let e1' = deify'nonbool e1 in
        let e0' = deify_iflift (fun e -> GOLEMIZE (g, e)) e1' in
        deify_finish e0'
     (* logical operators *)
   | M.FORALL (x, ty, e1) ->
        let e1' = deify'bool e1 in
        M.FORALL (x, ty, e1')
   | M.EXISTS (x, ty, e1) ->
        let e1' = deify'bool e1 in
        M.EXISTS (x, ty, e1')
   | M.LET (x, BOOL, e1, e2) ->
        let e1' = deify'bool e1 in
        let e2' = deify'bool e2 in
        M.LET (x, BOOL, e1', e2')
   | M.LET (x, ty, e1, e2) ->
        let e1' = deify'nonbool e1 in
        let e2' = deify'bool e2 in
        let e0' = deify_iflift_withcond (fun e -> LET (x, ty, e, e2')) e1' in
        deify_finish e0'
   | M.IF (c, t, f) ->
        let c' = deify'bool c in
        let t' = deify'bool t in
        let f' = deify'bool f in
        let t'' = M.IMPLIES (c', t') in
        let f'' = M.IMPLIES (M.LOGNOT c', f') in
        M.LOGAND (t'', f'')
   | M.IMPLIES (e1, e2) ->
        let e1' = deify'bool e1 in
        let e2' = deify'bool e2 in
        M.IMPLIES (e1', e2')
   | M.LOGNOT e1 ->
        let e1' = deify'bool e1 in
        M.LOGNOT e1'
   | M.LOGAND (e1, e2) ->
        let e1' = deify'bool e1 in
        let e2' = deify'bool e2 in
        M.LOGAND (e1', e2')
   | M.LOGOR (e1, e2) ->
        let e1' = deify'bool e1 in
        let e2' = deify'bool e2 in
        M.LOGOR (e1', e2')
   | M.LOGXOR (e1, e2) ->
        let e1' = deify'bool e1 in
        let e2' = deify'bool e2 in
        LOGXOR (e1', e2')
   | M.LOGEQ (e1, e2) ->
        let e1' = deify'nonbool e1 in
        let e2' = deify'nonbool e2 in
        let e0' = deify_iflift_2 (fun ea eb -> LOGEQ (ea, eb)) e1' e2' in
        deify_finish e0'
     (* other operators *)
   | M.MINTUOP (op, e1) ->
        let e1' = deify'nonbool e1 in
        let e0' = deify_iflift (fun e -> MINTUOP (op, e)) e1' in
        deify_finish e0'
   | M.MINTBOP (op, e1, e2) ->
        let e1' = deify'nonbool e1 in
        let e2' = deify'nonbool e2 in
        let e0' = deify_iflift_2 (fun ea eb -> MINTBOP (op, ea, eb)) e1' e2' in
        deify_finish e0'

let deify'sketch sketch =
   let deify'oneexpr e =
      match deify'nonbool e with
      | NOIF e' -> [e']
      | ISIF cases ->
           (*
            * XXX: this is not really what we want; it converts "if x
            * then a else b" into just the choices "a" or "b" and
            * throws away the condition. But, people probably
            * shouldn't be slamming ifs into operands in sketches
            * anyway.
	    *)
           let once (_c, e) = e in
           List.map once cases
   in
   let deify'operand (field, es) =
      let es' = List.concat (List.map deify'oneexpr es) in
      (field, es')
   in
   let deify'insn (opcode, operands) =
      (opcode, List.map deify'operand operands)
   in
   let deify'entry se =
      match se with
      | M.ANYTHING -> M.ANYTHING
      | M.CHOICES insns -> M.CHOICES (List.map deify'insn insns)
   in
   match sketch with
   | None -> None
   | Some entries -> Some (List.map deify'entry entries)


(**************************************************************)
(* interface *)

let go es sketch =
   let alphactx = newalphactx () in
   let es = alpha'exprs alphactx es in
   let sketch = alpha'sketch alphactx sketch in

   let tyctx = newinfctx () in
   inf'boolexprs tyctx es;
   inf'sketch tyctx sketch;
   let (vars, intvals) = finishinfctx tyctx in

   let cctx = newcctx vars intvals in
   let es' = c'exprs cctx es in
   let sketch' = c'sketch cctx sketch in
   let es'' = List.map deify'bool es' in
   let sketch'' = deify'sketch sketch' in
   (es'', sketch'')
