(*
 * Copyright (c) 2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

module M = Mintops
open Smt

(**************************************************************)
(* sigh *)

module BigIntKey = struct
   type t = Big_int.big_int
   let compare = Big_int.compare_big_int
end

module BigIntSet = Set.Make(BigIntKey)
module BigIntMap = Map.Make(BigIntKey)


(**************************************************************)
(* tools *)

(*
 * XXX: these qualifiers are not supported by the representation any
 * more; this code will stuff a qualifier into the name portion of the
 * varname but the variable name extractor for solver responses won't
 * be able to process it.
 *
 * Will need adjustments if we ever want to bring back the boolization
 * code in here.
 *)

(*
let qualify_varname x k =
   match x.qual with
   | None -> { x with qual = Some k; }
   | Some _ -> Util.crash "smtcanon: qualify_varname: already qualified"
*)

let qualify_varname x k =
   let q x' =
      x' ^ "_QQ" ^ string_of_int k
   in
   match x with
   | PLAINVAR x' -> PLAINVAR (q x')
   | CONTROLVAR (k, x') -> CONTROLVAR (k, q x')
   | STATEVAR (s, k, x') -> STATEVAR (s, k, q x')
   | MACHINESTATE (s, pp, x') -> MACHINESTATE (s, pp, x')
   | GOLEMVAR (s, x') -> GOLEMVAR (s, q x')

let booleq e1 e2 =
   OR (AND (NOT e1, NOT e2), AND (e1, e2))

let rec collect_ands e1 e2 =
   let visit e =
      match e with
      | AND (ea, eb) -> collect_ands ea eb
      | _ -> [e]
   in
   visit e1 @ visit e2

let rec collect_ors e1 e2 =
   let visit e =
      match e with
      | OR (ea, eb) -> collect_ors ea eb
      | _ -> [e]
   in
   visit e1 @ visit e2


(**************************************************************)
(* base *)

(*
 * base expansions: eliminate LET, XOR, EQ on bools, IMPLIES
 * crash if we get a FORALL or EXISTS
 *)

let rec isbool typemap e =
   match e with
   | TRUE
   | FALSE -> true
   | XINTVAL _ -> false
   | MINTVAL _ -> false
   | BOOLARRAYVAL _ -> false
   | XINTARRAYVAL _ -> false
   | MINTARRAYVAL _ -> false
   | READVAR x -> begin
        try
           VarMap.find x typemap = BOOL
        with Not_found -> false
     end
   | READARRAY (x, _) -> begin
        try begin
           match VarMap.find x typemap with
           | ARRAY (_, BOOL) -> true
           | _ -> false
        end
        with Not_found -> false
     end
   | UPDATEARRAY _ -> false
   | LET (_x, _ty, _e1, e2) -> isbool typemap e2
   | IF (_c, t, _f) -> isbool typemap t (* typechecker makes t and f match *)
   | NOT _ -> true
   | AND _ -> true
   | OR _ -> true
   | XOR _ -> true
   | EQ (e1, e2) -> isbool typemap e1 && isbool typemap e2
   | IMPLIES _ -> true
   | FORALL _ -> true
   | EXISTS _ -> true
   | MINTUOP _ -> false
   | MINTBOP (M.EQ, _, _) -> true
   | MINTBOP (M.ULT, _, _) -> true
   | MINTBOP (M.SLT, _, _) -> true
   | MINTBOP (_, _, _) -> false

let rec base'expr typemap letmap e =
   match e with
   | TRUE
   | FALSE
   | XINTVAL _
   | MINTVAL _ -> e
   | BOOLARRAYVAL _
   | XINTARRAYVAL _
   | MINTARRAYVAL _ -> e
   | READVAR x -> begin
        try
           VarMap.find x letmap
        with Not_found -> e
     end
   | READARRAY (a, e1) ->
        let e1' = base'expr typemap letmap e1 in
        READARRAY (a, e1')
   | UPDATEARRAY (a, e1, e2) ->
        let e1' = base'expr typemap letmap e1 in
        let e2' = base'expr typemap letmap e2 in
        UPDATEARRAY (a, e1', e2')
   | LET (x, ty, e1, e2) ->
        let e1' = base'expr typemap letmap e1 in
        let typemap' = VarMap.add x ty typemap in
        let letmap' = VarMap.add x e1' letmap in
        base'expr typemap' letmap' e2
   | IF (c, t, f) ->
        let c' = base'expr typemap letmap c in
        let t' = base'expr typemap letmap t in
        let f' = base'expr typemap letmap f in
        if isbool typemap t' then
           let e' = AND (IMPLIES (c', t'), IMPLIES (NOT c', f')) in
           base'expr typemap letmap e'
        else
           IF (c', t', f')
   | NOT e1 ->
        let e1' = base'expr typemap letmap e1 in
        NOT e1'
   | AND (e1, e2) ->
        let e1' = base'expr typemap letmap e1 in
        let e2' = base'expr typemap letmap e2 in
        AND (e1', e2')
   | OR (e1, e2) ->
        let e1' = base'expr typemap letmap e1 in
        let e2' = base'expr typemap letmap e2 in
        OR (e1', e2')
   | XOR (e1, e2) ->
        let e1' = base'expr typemap letmap e1 in
        let e2' = base'expr typemap letmap e2 in
        AND (OR (e1', e2'), NOT (AND (e1', e2')))
   | EQ (e1, e2) ->
        let e1' = base'expr typemap letmap e1 in
        let e2' = base'expr typemap letmap e2 in
        if isbool typemap e1' then booleq e1' e2'
        else EQ (e1', e2')
   | IMPLIES (e1, e2) ->
        let e1' = base'expr typemap letmap e1 in
        let e2' = base'expr typemap letmap e2 in
        OR (NOT e1', e2')
   | FORALL _ -> Util.crash "canon: base: FORALL"
   | EXISTS _ -> Util.crash "canon: base: EXISTS"
   | MINTUOP (op, e1) ->
        let e1' = base'expr typemap letmap e1 in
        MINTUOP (op, e1')
   | MINTBOP (op, e1, e2) ->
        let e1' = base'expr typemap letmap e1 in
        let e2' = base'expr typemap letmap e2 in
        MINTBOP (op, e1', e2')

let base'decl typemap d =
   match d with
   | BIND (x, ty) -> typemap := VarMap.add x ty !typemap; d
   | COMMENT _ -> d
   | ASSERT e -> ASSERT (base'expr !typemap VarMap.empty e)

let base decls =
   List.map (base'decl (ref VarMap.empty)) decls

let boolsimplify e =
   base'expr VarMap.empty VarMap.empty e

(**************************************************************)
(* xints *)

(*
 * Turn all xint (math int) material into bools.
 *)

type fx_ctx = {
   xintvars: BigIntSet.t VarMap.t ref;
}

(*
 * for every integer var ("xint", that is, math ints, not machine ints)
 * deduce the allowable set of values.
 *
 * (we know that these exist and are easy to find because the smt has
 * been constructed that way.)
 *)

let fx'expr e =
   match e with
   | OR (e1, e2) -> begin
        let es = (collect_ors e1 e2) in
        let grind z e =
           match z with
           | None -> None
           | Some (mx0, set) ->
                match e with
                | EQ (READVAR x, XINTVAL (n, _w)) -> begin
                     let set' = BigIntSet.add n set in
                     match mx0 with
                     | None -> Some (Some x, set')
                     | Some x0 when x0 = x -> Some (Some x0, set')
                     | _ -> None
                  end
                | FALSE -> Some (mx0, set)
                | _ -> None
        in
        match List.fold_left grind (Some (None, BigIntSet.empty)) es with
        | None -> None
        | Some (None, _) -> None
        | Some (Some x0, set) -> Some (x0, set)
     end
   | _ -> None

let fx'decl ctx d =
   match d with
   | BIND (x, XINT _w) ->
        ctx.xintvars := VarMap.add x BigIntSet.empty !(ctx.xintvars)
   | BIND _ -> ()
   | ASSERT e -> begin
        match fx'expr e with
        | None -> ()
        | Some (x, set) -> begin
             let oldset = VarMap.find x !(ctx.xintvars) in
             let newset = BigIntSet.union set oldset in
             ctx.xintvars := VarMap.add x newset !(ctx.xintvars)
          end
     end
   | COMMENT _ -> ()

let rec ilog2x k =
   if k <= 1 then 0
   else 1 + (ilog2x (k / 2))

let rec mkbits n x =
   if n = 0 then []
   else (x mod 2 > 0) :: mkbits (n - 1) (x / 2)

let represent_xint valset =
   let n = BigIntSet.cardinal valset in
   let bits = ilog2x n in
   let valmap =
      let mkentry map (neue, olde) =
         BigIntMap.add olde (mkbits bits neue) map
      in
      let vals = Util.number (BigIntSet.elements valset) in
      List.fold_left mkentry BigIntMap.empty vals
   in
   (bits, valmap)

let test_xint ctx x k =
   let (_bits, valmap) = VarMap.find x ctx in
   if List.length (BigIntMap.bindings valmap) = 0 then begin
       Util.crash ("No value map for " ^ string_of_varname x)
   end;
   let k' = Util.number (BigIntMap.find k valmap) in
   let one (bitnum, bitval) =
      let read = READVAR (qualify_varname x bitnum) in
      if bitval then read else NOT read
   in
   Smt.and_list (List.map one k')

let rec rx'expr ctx e =
   match e with
   | TRUE
   | FALSE -> e
   | XINTVAL _ -> Util.crash "canon: rx'expr: loose XINTVAL"
   | MINTVAL _ -> e
   | BOOLARRAYVAL _ -> e
   | XINTARRAYVAL _ -> e
   | MINTARRAYVAL _ -> e
   | READVAR x -> begin
        try
           let _ = VarMap.find x ctx in
           Util.crash "canon: rx'expr: loose READVAR of xint"
        with Not_found -> e
     end
   | READARRAY (x, e1) ->
        let e1' = rx'expr ctx e1 in
        READARRAY (x, e1')
   | UPDATEARRAY (x, e1, e2) ->
        let e1' = rx'expr ctx e1 in
        let e2' = rx'expr ctx e2 in
        UPDATEARRAY (x, e1', e2')
   | LET _ -> Util.crash "canon: rx'expr: leftover LET"
   | IF (c, t, f) ->
        let c' = rx'expr ctx c in
        let t' = rx'expr ctx t in
        let f' = rx'expr ctx f in
        IF (c', t', f')
   | NOT e1 ->
        let e1' = rx'expr ctx e1 in
        NOT e1'
   | AND (e1, e2) ->
        let e1' = rx'expr ctx e1 in
        let e2' = rx'expr ctx e2 in
        AND (e1', e2')
   | OR (e1, e2) ->
        let e1' = rx'expr ctx e1 in
        let e2' = rx'expr ctx e2 in
        OR (e1', e2')
   | XOR _ -> Util.crash "canon: rx'expr: leftover XOR"
   | EQ (READVAR x, XINTVAL (k, _w)) ->
        test_xint ctx x k
   | EQ (XINTVAL (k, _w), READVAR x) ->
        test_xint ctx x k
   | EQ (e1, e2) ->
        let e1' = rx'expr ctx e1 in
        let e2' = rx'expr ctx e2 in
        EQ (e1', e2')
   | IMPLIES _ -> Util.crash "canon: rx'expr: leftover IMPLIES"
   | FORALL _ -> Util.crash "canon: rx'expr: leftover FORALL"
   | EXISTS _ -> Util.crash "canon: rx'expr: leftover EXISTS"
   | MINTUOP (op, e1) ->
        let e1' = rx'expr ctx e1 in
        MINTUOP (op, e1')
   | MINTBOP (op, e1, e2) ->
        let e1' = rx'expr ctx e1 in
        let e2' = rx'expr ctx e2 in
        MINTBOP (op, e1', e2')

let rx'decl ctx d =
   match d with
   | BIND (x, XINT _w) ->
        let (n, _valuemap) = VarMap.find x ctx in
        let make k = BIND (qualify_varname x k, BOOL) in
        List.init n make
   | BIND _ -> [d]
   | ASSERT e -> [ASSERT (rx'expr ctx e)]
   | COMMENT _ -> [d]

let process_xints decls =
   let ctx = { xintvars = ref VarMap.empty; } in
   List.iter (fx'decl ctx) decls;

   let xintvars' = VarMap.map represent_xint !(ctx.xintvars) in

   List.concat (List.map (rx'decl xintvars') decls)

(**************************************************************)
(* mints *)

(*
 * turn machine integers into lists of bools
 *)

let bitconstant k0 =
   let rec bitc depth k =
      if depth >= 32 then []
      else begin
         let two = Big_int.big_int_of_int 2 in
         let msbs = bitc (depth + 1) (Big_int.div_big_int k two) in
         (if (Big_int.gt_big_int (Big_int.mod_big_int k two) Big_int.zero_big_int) then TRUE else FALSE) :: msbs
      end
   in
   bitc 0 k0

let bitreadvar x =
   let mk i = READVAR (qualify_varname x i) in
   List.init 32 mk

let biteq e1s e2s =
   Smt.and_list (List.map (fun (a, b) -> booleq a b) (List.combine e1s e2s))

let rec bitult e1s e2s =
   match e1s, e2s with
   | [], [] -> FALSE
   | _ :: _, []
   | [], _ :: _ -> Util.crash "canon: bitult: mismatched lengths"
   | e1 :: e1s', e2 :: e2s' ->
        OR (bitult e1s' e2s', AND (biteq e1s' e2s', AND (NOT e1, e2)))

let bitslt e1s e2s =
   assert (List.length e1s = 32);
   let s1 = List.nth e1s 31 in
   let s2 = List.nth e1s 31 in
   (* s1 && !s2 -> true; !s1 && s2 -> false; s1 == s2 -> use bitult *)
   let x1 = IMPLIES (AND (s1, NOT s2), TRUE) in
   let x2 = IMPLIES (NOT (AND (NOT s1, s2)), FALSE) in
   let x3 = IMPLIES (EQ (s1, s2), bitult e1s e2s) in
   let x4 = AND (x1, AND (x2, x3)) in
   (* clean it up *)
   boolsimplify x4

let bitadd e1s e2s =
   (* XXX I forget how to do this properly *)
   let once (carry, rs) (e1, e2) =
      let r = XOR (carry, XOR (e1, e2)) in
      let carry' = OR (OR (carry, e1), OR (OR (carry, e2), OR (e1, e2))) in
      (carry', r :: rs)
   in
   let (_, rs) = List.fold_left once (FALSE, []) (List.combine e1s e2s) in
   List.rev rs

let bitnot e1s =
   List.map (fun e -> NOT e) e1s

let bitand e1s e2s =
   List.map2 (fun e1 e2 -> AND (e1, e2)) e1s e2s

let bitor e1s e2s =
   List.map2 (fun e1 e2 -> OR (e1, e2)) e1s e2s

let bitxor e1s e2s =
   List.map2 (fun e1 e2 -> boolsimplify (XOR (e1, e2))) e1s e2s

let bitneg e1s =
   bitadd (bitnot e1s) (bitconstant (Big_int.big_int_of_int 1))

let bitsignextend start e1s =
   let lsb = Util.take start e1s in
   let rest = Util.repeat (32 - start) (List.nth lsb (start - 1)) in
   lsb @ rest

let bitzeroextend start e1s =
   let lsb = Util.take start e1s in
   let rest = Util.repeat (32 - start) FALSE in
   lsb @ rest

let bittrunc start e1s =
   bitzeroextend start e1s

(*
let bitshl _e1s _e2s = Util.crash "canon: shift left"
let bitushr _e1s _e2s = Util.crash "canon: shift right"
let bitsshr _e1s _e2s = Util.crash "canon: shift right"
*)
(* XXX *)
let bitshl e1s _e2s = e1s
let bitushr e1s _e2s = e1s
let bitsshr e1s _e2s = e1s

let bituop op e1s =
   match op with
   | M.BITNOT -> bitnot e1s
   | M.SBWIDEN -> bitsignextend 8 e1s
   | M.SHWIDEN -> bitsignextend 16 e1s
   | M.UBWIDEN -> bitzeroextend 8 e1s
   | M.UHWIDEN -> bitzeroextend 16 e1s
   | M.U5WIDEN -> bitzeroextend 5 e1s
   | M.TRUNCU5 -> bittrunc 5 e1s
   | M.TOSIGNED -> e1s
   | M.TOUNSIGNED -> e1s

let bitbop op e1s e2s =
   match op with
   | M.EQ -> Util.crash "canon: bitbop: misplaced EQ"
   | M.ULT -> Util.crash "canon: bitbop: misplaced ULT"
   | M.SLT -> Util.crash "canon: bitbop: misplaced SLT"
   | M.ADD -> bitadd e1s e2s
   | M.SUB -> bitadd e1s (bitneg e2s)
   | M.SMUL_HI -> Util.crash "canon: bitbop: canna multiply yet"
   | M.UMUL_HI -> Util.crash "canon: bitbop: canna multiply yet"
   | M.MUL_LO -> Util.crash "canon: bitbop: canna multiply yet"
   | M.SDIV -> Util.crash "canon: bitbop: canna divide yet"
   | M.SMOD -> Util.crash "canon: bitbop: canna divide yet"
   | M.UDIV -> Util.crash "canon: bitbop: canna divide yet"
   | M.UMOD -> Util.crash "canon: bitbop: canna divide yet"
   | M.BITAND -> bitand e1s e2s
   | M.BITOR -> bitor e1s e2s
   | M.BITXOR -> bitxor e1s e2s
   | M.SHL -> bitshl e1s e2s
   | M.USHR -> bitushr e1s e2s
   | M.SSHR -> bitsshr e1s e2s

(*
 * XXX this logic no longer works, because adding IF means that
 * boolean expressions can be hidden under bitvector ones.
 *)

let rec xm'mexpr ctx e =
   match e with
   | TRUE -> Util.crash "canon: xm'mexpr: loose TRUE"
   | FALSE -> Util.crash "canon: xm'mexpr: loose FALSE"
   | XINTVAL _ -> Util.crash "canon: xm'mexpr: leftover XINTVAL"
   | MINTVAL k -> bitconstant k
   | BOOLARRAYVAL _
   | MINTARRAYVAL _
   | XINTARRAYVAL _ -> Util.crash "canon: xm'mexpr: array unsupported (FIXME)"
   | READVAR x ->
        if VarSet.mem x !ctx then
           bitreadvar x
        else Util.crash "canon: xm'mexpr: loose non-mint var"
   | READARRAY (_a, _e1) ->
        Util.crash "canon: xm'mexpr: READARRAY not supported (FIXME)"
   | UPDATEARRAY (_a, _e1, _e2) ->
        Util.crash "canon: xm'mexpr: UPDATEARRAY not supported (FIXME)"
   | LET _ -> Util.crash "canon: xm'mexpr: leftover LET"
   | IF _ -> Util.crash "canon: xm'mexpr: IF not supported (FIXME)"
   | NOT _ -> Util.crash "canon: xm'mexpr: loose NOT"
   | AND _ -> Util.crash "canon: xm'mexpr: loose AND"
   | OR _ -> Util.crash "canon: xm'mexpr: loose OR"
   | XOR _ -> Util.crash "canon: xm'mexpr: leftover XOR"
   | EQ _ -> Util.crash "canon: xm'mexpr: leftover EQ"
   | IMPLIES _ -> Util.crash "canon: xm'mexpr: leftover IMPLIES"
   | FORALL _ -> Util.crash "canon: xm'mexpr: leftover FORALL"
   | EXISTS _ -> Util.crash "canon: xm'mexpr: leftover EXISTS"
   | MINTUOP (op, e1) ->
        let e1' = xm'mexpr ctx e1 in
        bituop op e1'
   | MINTBOP (op, e1, e2) ->
        let e1' = xm'mexpr ctx e1 in
        let e2' = xm'mexpr ctx e2 in
        bitbop op e1' e2'

let rec xm'expr ctx e =
   match e with
   | TRUE
   | FALSE -> e
   | XINTVAL _ -> Util.crash "canon: xm'expr: leftover XINTVAL"
   | MINTVAL _ -> Util.crash "canon: xm'expr: loose MINTVAL"
   | BOOLARRAYVAL _
   | MINTARRAYVAL _
   | XINTARRAYVAL _ -> Util.crash "canon: xm'expr: array not supported (FIXME)"
   | READVAR x ->
        if VarSet.mem x !ctx then
           Util.crash "canon: xm'expr: loose mint var"
        else e
   | READARRAY _ ->
        Util.crash "canon: xm'expr: READARRAY not supported (FIXME)"
   | UPDATEARRAY _ ->
        Util.crash "canon: xm'expr: UPDATEARRAY not supported (FIXME)"
   | LET _ -> Util.crash "canon: xm'expr: leftover LET"
   | IF _ -> Util.crash "canon: xm'expr: IF not supported (FIXME)"
   | NOT e1 ->
        let e1' = xm'expr ctx e1 in
        NOT e1'
   | AND (e1, e2) ->
        let e1' = xm'expr ctx e1 in
        let e2' = xm'expr ctx e2 in
        AND (e1', e2')
   | OR (e1, e2) ->
        let e1' = xm'expr ctx e1 in
        let e2' = xm'expr ctx e2 in
        OR (e1', e2')
   | XOR _ -> Util.crash "canon: xm'expr: leftover XOR"
   | EQ _ -> Util.crash "canon: xm'expr: leftover EQ"
   | IMPLIES _ -> Util.crash "canon: xm'expr: leftover IMPLIES"
   | FORALL _ -> Util.crash "canon: xm'expr: leftover FORALL"
   | EXISTS _ -> Util.crash "canon: xm'expr: leftover EXISTS"
   | MINTUOP _ -> Util.crash "canon: xm'expr: loose MINTUOP"
   | MINTBOP (EQ, e1, e2) ->
        let e1' = xm'mexpr ctx e1 in
        let e2' = xm'mexpr ctx e2 in
        biteq e1' e2'
   | MINTBOP (ULT, e1, e2) ->
        let e1' = xm'mexpr ctx e1 in
        let e2' = xm'mexpr ctx e2 in
        bitult e1' e2'
   | MINTBOP (SLT, e1, e2) ->
        let e1' = xm'mexpr ctx e1 in
        let e2' = xm'mexpr ctx e2 in
        bitslt e1' e2'
   | MINTBOP _ -> Util.crash "canon: xm'expr: loose MINTBOP"

let xm'decl ctx d =
   match d with
   | BIND (x, MINT) ->
        ctx := VarSet.add x !ctx;
        let xs' =
           List.init 32 (qualify_varname x)
        in
        List.map (fun x' -> BIND (x', BOOL)) xs'
   | BIND _ -> [d]
   | ASSERT e -> [ASSERT (xm'expr ctx e)]
   | COMMENT _ -> [d]

let process_mints decls =
   let ctx = ref VarSet.empty in
   List.concat (List.map (xm'decl ctx) decls)


(**************************************************************)
(* canonicalize *)

(* push NOT out to the leaves *)
let rec canon'not e =
   match e with
   | TRUE
   | FALSE -> e
   | XINTVAL _ -> Util.crash "smtcanon: canon'not: leftover XINTVAL"
   | MINTVAL _ -> Util.crash "smtcanon: canon'not: leftover MINTVAL"
   | BOOLARRAYVAL _
   | MINTARRAYVAL _
   | XINTARRAYVAL _ -> Util.crash "canon: canon'not: array unsupported (FIXME)"
   | READVAR _ -> e
   | READARRAY (a, e1) -> READARRAY (a, canon'not e1)
   | UPDATEARRAY (a, e1, e2) -> UPDATEARRAY (a, canon'not e1, canon'not e2)
   | LET _ -> Util.crash "smtcanon: canon'not: leftover LET"
   | IF _ -> Util.crash "smtcanon: canon'not: leftover IF"
   | NOT TRUE -> FALSE
   | NOT FALSE -> TRUE
   | NOT (READVAR _) -> e
   | NOT (NOT e1) -> canon'not e1
   | NOT (AND (e1, e2)) -> OR (canon'not (NOT e1), canon'not (NOT e2))
   | NOT (OR (e1, e2)) -> AND (canon'not (NOT e1), canon'not (NOT e2))
   | NOT e1 -> NOT (canon'not e1)
   | AND (e1, e2) -> AND (canon'not e1, canon'not e2)
   | OR (e1, e2) -> OR (canon'not e1, canon'not e2)
   | XOR _ -> Util.crash "smtcanon: canon'not: leftover XOR"
   | EQ _ -> Util.crash "smtcanon: canon'not: leftover EQ"
   | IMPLIES _ -> Util.crash "smtcanon: canon'not: leftover IMPLIES"
   | FORALL _ -> Util.crash "smtcanon: canon'not: leftover FORALL"
   | EXISTS _ -> Util.crash "smtcanon: canon'not: leftover EXISTS"
   | MINTUOP _ -> Util.crash "smtcanon: canon'not: leftover MINTUOP"
   | MINTBOP _ -> Util.crash "smtcanon: canon'not: leftover MINTBOP"

(* convert to ANDs of ORs *)
let rec crush_orand e1 e2 =
   match e1, e2 with
   | AND (e1a, e1b), _ -> AND (crush_orand e1a e2, crush_orand e1b e2)
   | _, AND (e2a, e2b) -> AND (crush_orand e1 e2a, crush_orand e1 e2b)
   | _, _ -> OR (e1, e2)

let rec canon'and e =
   match e with
   | TRUE
   | FALSE
   | READVAR _
   | NOT (READVAR _) -> e
   | AND (e1, e2) ->
        let e1' = canon'and e1 in
        let e2' = canon'and e2 in
        AND (e1', e2')
   | OR (e1, e2) ->
        let e1' = canon'and e1 in
        let e2' = canon'and e2 in
        crush_orand e1' e2'
   | _ -> Util.crash "canon: canon'and: bogus"


let canon'decl d =
   match d with
   | BIND _ -> [d]
   | COMMENT _ -> [d]
   | ASSERT e ->
        match canon'and (canon'not e) with
        | AND (e1', e2') -> 
             let es' = collect_ands e1' e2' in
             List.map (fun e' -> ASSERT e') es'
        | _ as e' -> [ASSERT e']

let canonicalize decls =
   List.concat (List.map canon'decl decls)

(**************************************************************)
(* toplevel *)

let go decls =
   let decls = base decls in
   (* XXX there should be an if-eliminator here *)
   let decls = process_xints decls in

   let decls = process_mints decls in
   let r = canonicalize decls in
prerr_string ("zooby"); prerr_newline ();
prerr_string (Printsmt.print "   " decls);
r