(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

type stateid = INIT of int | STATE of int | VERIFY

(*
type namegolem = ISGOLEM | NOTGOLEM
type nameval = ISVALUE | NOTVALUE

type varname = {
   state: stateid option;
   pp: Ppoint.t option;
   golem: namegolem;
   isvalue: nameval;
   corename: string;
   qual: int option;
}
*)

type varname =
| PLAINVAR of string
| CONTROLVAR of int * string
| STATEVAR of stateid * int * string
| MACHINESTATE of stateid * Ppoint.t * string
| GOLEMVAR of stateid * string


type typename =
     BOOL
   | XINT of int(*bitsize*) (* logic/structure/modeling integer value *)
   | MINT (* machine integer value *)
   | ARRAY of typename * typename

type expr =
     TRUE
   | FALSE
   | XINTVAL of Big_int.big_int * int(*bitsize*)
   | MINTVAL of Big_int.big_int
   | BOOLARRAYVAL of bool Types.BigIntMap.t * bool *
                int(*keywidth*)
   | XINTARRAYVAL of Big_int.big_int Types.BigIntMap.t * Big_int.big_int *
		int(*keywidth*) * int (*valuewidth*)
   | MINTARRAYVAL of Big_int.big_int Types.BigIntMap.t * Big_int.big_int *
		int(*keywidth*)
   | READVAR of varname
   | READARRAY of varname * expr
   | UPDATEARRAY of varname * expr * expr
   | LET of varname * typename * expr * expr
   | IF of expr * expr * expr
   | NOT of expr
   | AND of expr * expr
   | OR of expr * expr
   | XOR of expr * expr
   | EQ of expr * expr
   | IMPLIES of expr * expr
   | FORALL of varname * typename * expr
   | EXISTS of varname * typename * expr
   | MINTUOP of Mintops.uop * expr
   | MINTBOP of Mintops.bop * expr * expr

type decl =
     BIND of varname * typename
   | ASSERT of expr
   | COMMENT of string

type smt = decl list

(*
 * maps for variable names
 *)

module VarKey = struct
   type t = varname
   let compare = compare
end
module VarMap = Map.Make(VarKey)
module VarSet = Set.Make(VarKey)

(*
 * assignments
 *)

type assignment =
   | ASSIGN of varname * typename * expr
   | ASSIGNARRAY of varname * typename * expr Types.BigIntMap.t * expr

type model = assignment list
type counterexample = expr list

let boolval k = if k then TRUE else FALSE

let xintval k w = XINTVAL (Big_int.big_int_of_int k, w)
let mintval k = MINTVAL (Big_int.big_int_of_int k)

let xinteq e1 e2 = EQ (e1, e2)
let minteq e1 e2 = MINTBOP (Mintops.EQ, e1, e2)

(*
 * and/or over lists
 *)
let and_list props =
   List.fold_right (fun prop z : expr -> AND (prop, z)) props TRUE
let or_list props =
   List.fold_right (fun prop z : expr -> OR (prop, z)) props FALSE

(*
 * variable name strings
 *)

(*
let string_of_varname { state; pp; golem; isvalue; corename; qual; } =
   let state' =
      match state with
      | None -> ""
      | Some (INIT i) -> "I" ^ string_of_int i
      | Some (STATE s) -> "S" ^ string_of_int s
      | Some (VERIFY) -> "V"
   in
   let pp' =
      match pp with
      | None -> ""
      | Some pp' -> Ppoint.prefix pp'
   in
   let golem' =
      match golem with
      | NOTGOLEM -> ""
      | ISGOLEM -> "golem_"
   in
   let isvalue' =
      match isvalue with
      | NOTVALUE -> ""
      | ISVALUE -> "val"
   in
   let qual' =
      match qual with
      | None -> ""
      | Some q -> "_QQ" ^ string_of_int q
   in
   state' ^ pp' ^ golem' ^ isvalue' ^ corename ^ qual'
*)

let string_of_varname x =
   let string_of_stateid s =
      match s with
      | INIT i -> "I" ^ string_of_int i
      | STATE s -> "S" ^ string_of_int s
      | VERIFY -> "V"
   in
   let string_of_symi k x' =
      "insn" ^ string_of_int k ^ "_" ^ x'
   in
   match x with
   | PLAINVAR x' -> x'
   | CONTROLVAR (k, x') -> string_of_symi k x'
   | STATEVAR (s, k, x') -> string_of_stateid s ^ string_of_symi k x'
   | MACHINESTATE (s, pp, x') -> string_of_stateid s ^ Ppoint.prefix pp ^ x'
   | GOLEMVAR (s, x') -> string_of_stateid s ^ "golem_" ^ x'

(*
let varname_of_string s0 =
   let wrong s =
      Util.crash ("smt: varname_of_string: wrong " ^ s ^ " in " ^ s0)
   in
   let isdigit c = c >= '0' && c <= '9' in
   let digitval c = Char.code c - Char.code '0' in
   let split s =
      let l = String.length s in
      if l = 0 then None
      else Some (String.get s 0, String.sub s 1 (l-1))
   in
   let split2 s =
      match split s with
      | None -> None
      | Some (c1, s) -> 
           match split s with
           | None -> None
           | Some (c2, s) -> Some (c1, c2, s)
   in
   let split3 s =
      match split2 s with
      | None -> None
      | Some (c1, c2, s) -> 
           match split s with
           | None -> None
           | Some (c3, s) -> Some (c1, c2, c3, s)
   in
   let split6 s =
      match split3 s with
      | None -> None
      | Some (c1, c2, c3, s) -> 
           match split3 s with
           | None -> None
           | Some (c4, c5, c6, s) -> Some (c1, c2, c3, c4, c5, c6, s)
   in
   let demand s c =
      match split s with
      | Some (c', s) when c' = c -> s
      | _ -> wrong s
   in
   let getnum s =
      let rec get s r =
         match split s with
         | Some (c, s) when isdigit c -> get s (10 * r + digitval c)
         | _ -> (s, r)
      in
      get s 0
   in

   let s = s0 in
   let s, state =
      match split s with
      | Some ('I', s) -> let (s, k) = getnum s in s, Some (INIT k)
      | Some ('S', s) -> let (s, k) = getnum s in s, Some (STATE k)
      | Some ('V', s) -> s, Some (VERIFY)
      | _ -> s, None
   in
   let s, pp =
      match split2 s with
      | Some ('p', 'p', s) ->
           let (s, k) = getnum s in (demand s '_'), Some (Ppoint.from_smt k)
      | _ -> s, None
   in
   let s, golem =
      match split6 s with
      | Some ('g', 'o', 'l', 'e', 'm', '_', s) -> s, ISGOLEM
      | _ -> s, NOTGOLEM
   in
   let s, isvalue =
      match split3 s with
      | Some ('v', 'a', 'l', s) -> s, ISVALUE
      | _ -> s, NOTVALUE
   in
   let s, qual =
      match String.rindex_opt s '_' with
      | None -> s, None
      | Some pos ->
           let s1 = String.sub s 0 pos in
           let s2 = String.sub s (pos + 1) (String.length s - pos - 1) in
           match split2 s2 with
           | Some ('Q', 'Q', s2) -> begin
                let (s2, k) = getnum s2 in
                match split s2 with
                | Some _ -> wrong s2
                | None -> s1, Some k
             end
           | _ -> s, None
   in
   { state; pp; golem; isvalue; corename = s; qual; }
*)

let varname_of_string s0 =
   let wrong msg =
      Util.crash ("smt: cannot interpret varname " ^ s0 ^ ": " ^ msg)
   in
   let isdigit c = c >= '0' && c <= '9' in
   let digitval c = Char.code c - Char.code '0' in
   let len = String.length s0 in

   let rec getnum' pos =
      let c = if pos < len then String.get s0 pos else 'a' in
      if isdigit c then
         let (pos', k) = getnum' (pos + 1) in
         (pos', k*10 + digitval c)
      else
         (pos, 0)
   in
   let getnum pos =
      let (pos', k) = getnum' pos in
      if pos' = pos then
         wrong ("Expected number at offset " ^ string_of_int pos)
      else
         (pos', k)
   in
   let demand pos c =
      if pos < len && String.get s0 pos = c then pos + 1
      else
         let c' = String.make 1 c in
         wrong ("Expected " ^ c' ^ " at offset " ^ string_of_int pos)
   in

   let needstate ctor pos state =
      let s = String.sub s0 pos (len - pos) in
      match state with
      | None -> wrong "no startstate"
      | Some state' -> ctor state' s
   in

   let (pos, state) =
      if s0 = "" then (0, None)
      else
         match String.get s0 0 with
         | 'I' -> let (pos, k) = getnum 1 in (pos, Some (INIT k))
         | 'S' -> let (pos, k) = getnum 1 in (pos, Some (STATE k))
         | 'V' -> (1, Some VERIFY)
         | _ -> (0, None)
   in

   if String.sub s0 pos 2 = "pp" then
      let (pos, k) = getnum (pos + 2) in
      let pos = demand pos '_' in
      needstate (fun st s -> MACHINESTATE (st, Ppoint.from_smt k, s)) pos state
   else if String.sub s0 pos 4 = "insn" then
      let (pos, k) = getnum (pos + 4) in
      let pos = demand pos '_' in
      let s = String.sub s0 pos (len - pos) in
      match state with
      | None -> CONTROLVAR (k, s)
      | Some state' -> STATEVAR (state', k, s)
   else if String.sub s0 pos 6 = "golem_" then
      needstate (fun st s -> GOLEMVAR (st, s)) (pos + 6) state
   else
      match state with
      | None -> PLAINVAR s0
      | Some _ -> wrong "unexpected startstate"
