(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

open Smt

let andp e =
   match e with
   | AND (e1, e2) -> Some (e1, e2)
   | _ -> None

let orp e =
   match e with
   | OR (e1, e2) -> Some (e1, e2)
   | _ -> None

let xorp e =
   match e with
   | XOR (e1, e2) -> Some (e1, e2)
   | _ -> None

let neverp _e = None

let collect samep e1 e2 =
   let rec collectone e =
      match samep e with
      | Some (e1', e2') -> collectone e1' @ collectone e2'
      | None -> [e]
   in
   collectone e1 @ collectone e2


let rec print_typename ty =
   match ty with
   | BOOL -> "bool"
   | XINT w -> "xint" ^ string_of_int w
   | MINT -> "mint"
   | ARRAY (t1, t2) ->
        "array " ^ print_typename t1 ^ " -> " ^ print_typename t2

let indent e' =
   List.map (fun txt -> "   " ^ txt) e'

let tailindent e' =
   match e' with
   | [] -> []
   | x :: more -> x :: indent more

let fold elts =
   let (most, last) = Util.untailcons elts in
   let most' = List.map (String.concat " ") most in
   let last' = String.concat " " last in

   let most'' = String.concat " " most' in
   let all = most'' ^ " " ^ last' in

   if String.length all < 72 then [all]
   else if String.length most'' < 72 then most'' :: last
   else if List.for_all (fun s -> String.length s < 72) most' then
      most' @ last
   else
      List.concat (List.map tailindent most) @ last

let pwrap precout (precin, e') =
   if precin > precout then e'
   else match e' with
   | [] -> [] (* notreached *)
   | [txt] -> ["(" ^ txt ^ ")"]
   | _ -> "(" :: indent e' @ [")"]

let rec print_expr' e =
   let bistr = Big_int.string_of_big_int in
   let binary prec samep op e1 e2 =
      let es = collect samep e1 e2 in
      let es' = List.map (fun e0 -> pwrap prec (print_expr' e0)) es in
(*
      let e1' = pwrap prec (print_expr' e1) in
      let e2' = pwrap prec (print_expr' e2) in
      (prec, fold [e1'; [op]; e2'])
*)
      (prec, fold (Util.intersperse [op] es'))
   in
   let array print m d =
      let doprint (k, v) = bistr k ^ " -> " ^ print v in
      let m' = List.map doprint (Types.BigIntMap.bindings m) in
      let d' = "_ -> " ^ print d in
      "[" :: m' @ [d'; "]"]
   in

   match e with
   | TRUE -> (99, ["true"])
   | FALSE -> (99, ["false"])
   | XINTVAL (n, w) -> (99, [bistr n ^ "x" ^ string_of_int w])
   | MINTVAL n -> (99, [bistr n ^ "m"])
   | BOOLARRAYVAL (m, d, _kw) -> (99, array (fun b -> string_of_bool b) m d)
   | XINTARRAYVAL (m, d, _kw, _vw) -> (99, array (fun n -> bistr n ^ "x") m d)
   | MINTARRAYVAL (m, d, _kw) -> (99, array (fun n -> bistr n ^ "m") m d)
   | READVAR x -> (99, [string_of_varname x])
   | READARRAY (a, e1) ->
        let e1' = pwrap 0 (print_expr' e1) in
        (99, [string_of_varname a ^ "["] @ e1' @ ["]"])
   | UPDATEARRAY (a, e1, e2) ->
        let e1' = pwrap 0 (print_expr' e1) in
        let e2' = pwrap 0 (print_expr' e2) in
        (99, [string_of_varname a ^ "["] @ e1' @ [" -> "] @ e2' @ ["]"])
   | LET (sym, ty, e1, e2) ->
        let sym' = string_of_varname sym in
        let symty = "let " ^ sym' ^ " :: " ^ print_typename ty ^ " :=" in
        let e1' = pwrap 5 (print_expr' e1) in
        let e2' = pwrap 5 (print_expr' e2) in
        (5, fold [[symty]; e1'; ["in"]; e2'])
   | IF (c, t, f) ->
        let c' = pwrap 5 (print_expr' c) in
        let t' = pwrap 5 (print_expr' t) in
        let f' = pwrap 15 (print_expr' f) in
        (15, fold [["if"]; c'; ["then"]; t'; ["else"]; f'])
   | FORALL (sym, ty, e1) ->
        let sym' = string_of_varname sym in
        let e1' = pwrap 10 (print_expr' e1) in
        (10, fold [["forall " ^ sym' ^ " :: " ^ print_typename ty ^ ","]; e1'])
   | EXISTS (sym, ty, e1) ->
        let sym' = string_of_varname sym in
        let e1' = pwrap 10 (print_expr' e1) in
        (10, fold [["exists " ^ sym' ^ " :: " ^ print_typename ty ^ ","]; e1'])
   | IMPLIES (e1, e2) ->
        binary 20 neverp "->" e1 e2
   | NOT e1 ->
        let e1' = pwrap 90 (print_expr' e1) in
        (90, fold [["!"]; e1'])
   | OR (e1, e2) ->
        binary 30 orp "||" e1 e2
   | XOR (e1, e2) ->
        binary 31 xorp "^^" e1 e2
   | AND (e1, e2) ->
        binary 32 andp "&&" e1 e2
   | EQ (e1, e2) ->
        binary 40 neverp "==x" e1 e2
   | MINTBOP (EQ, e1, e2) ->
        binary 40 neverp "==m" e1 e2
   | MINTUOP (op, e1) ->
        let name = Mintops.name_of_uop op in
        let es' = List.map (fun e -> pwrap 0 (print_expr' e)) [e1] in
        let es'' = Util.intersperse [","] es' in
        (99, fold ([[name ^ "("]] @ es'' @ [[")"]]))
   | MINTBOP (op, e1, e2) ->
        let name = Mintops.name_of_bop op in
        let es' = List.map (fun e -> pwrap 0 (print_expr' e)) [e1; e2] in
        let es'' = Util.intersperse [","] es' in
        (99, fold ([[name ^ "("]] @ es'' @ [[")"]]))

let print_decl d =
   match d with
   | BIND (sym, ty) ->
        ["bind " ^ string_of_varname sym ^ " :: " ^ print_typename ty]
   | ASSERT e ->
        let (_prec, e') = print_expr' e in
        fold [["assert"]; e']
   | COMMENT txt -> ["# " ^ txt]

let print_expr prefix e =
   let (_prec, e') = print_expr' e in
   prefix ^ String.concat ("\n" ^ prefix) e'

let print prefix decls =
   let strings = List.concat (List.map print_decl decls) in
   prefix ^ String.concat ("\n" ^ prefix) strings ^ "\n"

