(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 * Written by David A. Holland.
 *
 * Copyright (c) 2020 David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

open Symbolic
module MS = Mipsstate
module M = Mips
module MT = Mipstools

let readspec pathname =
   let ((rawpre, rawpost), numinsns, sketch) = Speclex.read pathname in
   let (pre, post) as spec, sketch =
      match Speccomp.go [rawpre; rawpost] sketch with
      | ([pre; post], sketch) -> ((pre, post), sketch)
      | _ -> Util.crash "main: garbage speccomp result"
   in
   let prestr = Printlogic.print "   " pre in
   let poststr = Printlogic.print "   " post in
   Log.dcrawsay ("Spec:\n");
   Log.dcrawsay ("pre:\n");
   Log.dcrawsay prestr;
   Log.dcrawsay ("post:\n");
   Log.dcrawsay poststr;
   Log.dcrawsay ("\n");
   Log.flush ();
   (spec, numinsns, sketch)

let typecheck_spec (pre, post) =
   let one e what =
      Typecheck.go M.VarMap.empty e BOOL what
   in
   one pre "precondition";
   one post "postcondition"


(*
let myspec_true = (M.TRUE, M.TRUE)

let myspec_startval_true =
   let v0 = M.READSTATE (MS.REG (CONC MS.V0), Ppoint.pre, M.Z_ZERO) in
   (M.MINTBOP (EQ, v0, M.INTVALUE (1, M.UINT32)), M.TRUE)

let myspec_add2 =
   let v0pre = M.READSTATE (MS.REG (CONC MS.V0), Ppoint.pre, M.Z_ZERO) in
   let v1pre = M.READSTATE (MS.REG (CONC MS.V1), Ppoint.pre, M.Z_ZERO) in
   let v0post = M.READSTATE (MS.REG (CONC MS.V0), Ppoint.post, M.Z_ZERO) in
   (M.TRUE, M.MINTBOP (EQ, v0post, M.MINTBOP (ADD, v0pre, v1pre)))

let myspec_add3 =
   let v0pre = M.READSTATE (MS.REG (CONC MS.V0), Ppoint.pre, M.Z_ZERO) in
   let v1pre = M.READSTATE (MS.REG (CONC MS.V1), Ppoint.pre, M.Z_ZERO) in
   let t0pre = M.READSTATE (MS.REG (CONC MS.T0), Ppoint.pre, M.Z_ZERO) in
   let t1pre = M.READSTATE (MS.REG (CONC MS.T1), Ppoint.pre, M.Z_ZERO) in
   let v0post = M.READSTATE (MS.REG (CONC MS.V0), Ppoint.post, M.Z_ZERO) in
   let t1post = M.READSTATE (MS.REG (CONC MS.T1), Ppoint.post, M.Z_ZERO) in
   let sum = M.MINTBOP (ADD, M.MINTBOP (ADD, v0pre, v1pre), t0pre) in
   (M.TRUE, M.LOGAND (M.MINTBOP (EQ, v0post, sum), M.MINTBOP (EQ, t1pre, t1post)))

let myspec_add4 =
   let v0pre = M.READSTATE (MS.REG (CONC MS.V0), Ppoint.pre, M.Z_ZERO) in
   let v1pre = M.READSTATE (MS.REG (CONC MS.V1), Ppoint.pre, M.Z_ZERO) in
   let t0pre = M.READSTATE (MS.REG (CONC MS.T0), Ppoint.pre, M.Z_ZERO) in
   let t1pre = M.READSTATE (MS.REG (CONC MS.T1), Ppoint.pre, M.Z_ZERO) in
   let v0post = M.READSTATE (MS.REG (CONC MS.V0), Ppoint.post, M.Z_ZERO) in
   (M.TRUE, M.MINTBOP (EQ, v0post,
                           M.MINTBOP (ADD, M.MINTBOP (ADD, v0pre, v1pre),
                                           M.MINTBOP (ADD, t0pre, t1pre))))
*)

(*
 * Get non-directory part of path.
 *)
let basename s =
    try
      let n = String.rindex s '/' in
      String.sub s (n+1) (String.length s - (n+1))
    with Not_found -> s

(*
 * Get the non-suffix part of a filename.
 *
 * Note: assumes it doesn't have a directory part; if so and that
 * contains a period, may misbehave.
 *)
let unsuffix s =
    try
       let n = String.rindex s '.' in
       String.sub s 0 n
    with Not_found -> s

let usage () =
   Util.say ("Usage: penguin [options] spec.pen");
   Util.say ("   -o outputfile");
   Util.say ("   -t timingfile");
   Util.say ("   -d                   (enable dumps)");
   Util.say ("   -q                   (quiet)");
   Util.say ("   -v                   (verbose)");
   exit 2

let main argv =
   let outfile = ref None in
   let timingfile = ref None in
   let dodumps = ref false in
   let quiet = ref false in
   let verbose = ref false in

   let options = [
      ("o", Getopt.HASARG (fun _ arg -> outfile := Some arg; None));
      ("t", Getopt.HASARG (fun _ arg -> timingfile := Some arg; None));
      ("d", Getopt.NOARG (fun _ -> dodumps := true; None));
      ("q", Getopt.NOARG (fun _ -> quiet := true; None));
      ("v", Getopt.NOARG (fun _ -> verbose := true; None));
   ] in
   let argv' =
      match Getopt.getopt options argv with
      | Getopt.SUCCESS (_, argv') -> argv'
      | Getopt.FAILURE msg -> Util.say msg; usage ()
   in

   let path =
      match argv' with
      | [_argv0; specfile] -> specfile
      | _ -> usage ()
   in
   let base = unsuffix (basename path) in

   let outfile =
      match !outfile with
      | None -> base ^ ".asm"
      | Some file -> file
   in

   Log.setup (base ^ ".log") !timingfile !dodumps !quiet !verbose;
   Solver.setup (!dodumps && !verbose);

   let (spec, numinsns, sketch) = readspec path in

(*
   let spec = myspec_add4 in
   let numinsns = 3 in
   let spec = myspec_add3 in
   let numinsns = 2 in
   let spec = myspec_add2 in
   let numinsns = 1 in
*)

   typecheck_spec spec;
   Enabled.autoset ();

(* Don't use this; it isn't better. Refer to the code so it gets linked though.
   let code = Synthdirect.synthesize spec numinsns sketch in
*)
   let _ = Synthdirect.synthesize in

   if !dodumps then begin
      let _ = Synthcegis.synthesize spec numinsns sketch true(*dryrun*) in ()
   end else ();

   let code = Synthcegis.synthesize spec numinsns sketch false(*dryrun*) in
   let codestr = Printasm.print "" code in

   let out = open_out outfile in
   output_string out codestr;
   close_out out;
   Log.crawsay codestr;

   Solver.shutdown ();
   Log.shutdown ()

let _ = main Sys.argv
