(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

open Symbolic
module MS = Mipsstate
module M = Mips
module MT = Mipstools

(*
 * Create the operands for a symbolic instruction whose logic variable
 * is INSNNAME; add variables to LENV.
 *
 * If SKINSNS is not None, constrain it to be one of the instructions
 * described therein.
 *)
let populate_symbolic_insn logicenv insnnum insnname skinsns =
   (*
    * First, make logic variables for the operands. Reuse the variables
    * for instructions that use the same kind of thing, in order to have
    * fewer variables.
    *)
   let (bindings, opset) = Insns.mkoperandset insnnum in
   let logicenv' =
      let bindvar env (fullname, ty) =
         M.VarMap.add fullname ty env
      in
      List.fold_left bindvar logicenv bindings
   in

   (*
    * Now we need a gigantic OR over all the possible instructions.
    *)
   let insns = Insns.insns opset in

   let readinsn = M.READVAR (insnname, M.INSN) in
   let testexpr insn = M.LOGEQ (readinsn, M.INSNVALUE insn) in
   let insnor = M.or_list (List.map testexpr insns) in

   (*
    * Now add additional constraints on the operands implied by
    * selecting any given instruction. Originally I'd set up the types
    * so we don't need any, but nowadays some of the possible cache
    * instructions are invalid.
    *)
   let constraints =
      (*
       * instruction-specific restriction
       *)
      let restrict insn e =
         [M.IMPLIES (testexpr insn, e)]
      in

      (*
       * The cache selected (c) and addressing (byindex) must match the
       * choice of cache golem (gsym). Note that gsym is always a symbol;
       * in practice c is also.
       *
       * Because there are only two cases of this, corresponding to
       * the two kinds of addressing (byindex), and also by
       * construction the symbol used for c is the same for all
       * instructions, we can issue this logic once for each case,
       * using the symbols from the opset, and not per-instruction.
       * We'll put a boolean in the opset that corresponds to byindex
       * and bind it per instruction below.
       *
       * If we get an instruction with a concrete cache (this is not
       * currently possible but it might happen in the future with
       * more advanced mechanisms applied to the instruction list)
       * bind the c variable from the opset to it per instruction,
       * below.
       *
       * We'll also explictly avoid issuing this material if the cache
       * state isn't enabled.
       *)
      let csym =
         match opset.whichcache with
         | SYM csym -> M.CONTROLVAR csym
         | CONC _ -> Util.crash "syminsns: populate: concrete whichcache"
      in
      let constrain_cachegolem byindex =
         let gsym = M.CONTROLVAR opset.cachegolemsym in
         let byindexsym = M.CONTROLVAR opset.byindexsym in
         let mkgolem c = if byindex then MS.G_INDEX c else MS.G_VADDR c in
         let mkgeq c =
            M.eq_logic (READVAR (gsym, GOLEM)) (GOLEMVALUE (mkgolem c))
         in
         let mkceq c =
            M.eq_logic (READVAR (csym, WHICHCACHE)) (WHICHCACHEVALUE c)
         in
         let mk c = M.LOGAND (mkgeq c, mkceq c) in
         let conc = M.or_list (List.map mk (Mipstools.all_caches_raw ())) in
         let readbyindex = M.READVAR (byindexsym, M.BOOL) in
         let prem = if byindex then readbyindex else (M.LOGNOT readbyindex) in
         M.IMPLIES (prem, conc)
      in
      let cachegolem_asserts =
         if !Enabled.use_cache then [
            constrain_cachegolem true;
            constrain_cachegolem false;
         ] else []
      in
      let specify_byindex byindex =
         let byindexsym = M.CONTROLVAR opset.byindexsym in
         let byindex' = if byindex then M.TRUE else M.FALSE in
         M.eq_logic (M.READVAR (byindexsym, M.BOOL)) byindex'
      in
      let concretize_whichcache c =
         M.eq_logic (READVAR (csym, WHICHCACHE)) (WHICHCACHEVALUE c)
      in
      let try_concretize_whichcache sc =
         match sc with
         | CONC c -> [concretize_whichcache c]
         | SYM _ -> []
      in

      (*
       * You can't wb or wbinv the icache.
       *)
      let noticache c =
         let read =
            match c with
            | CONC c' -> M.WHICHCACHEVALUE c'
            | SYM sym -> M.READVAR (M.CONTROLVAR sym, WHICHCACHE)
         in
         M.neq_logic read (WHICHCACHEVALUE ICACHE)
      in

      let once insn =
         match insn with
         | M.CACHE_INDEX_WBINV (sc, _, _gsym) ->
              let ps =
                 specify_byindex true
                 :: try_concretize_whichcache sc
              in
              restrict insn (M.and_list ps)
         | M.CACHE_VADDR_WB (sc, _, _gsym) ->
              let ps =
                 specify_byindex false
                 :: noticache sc
                 :: try_concretize_whichcache sc
              in
              restrict insn (M.and_list ps)
         | M.CACHE_VADDR_WBINV (sc, _, _gsym) ->
              let ps =
                 specify_byindex false
                 :: noticache sc
                 :: try_concretize_whichcache sc
              in
              restrict insn (M.and_list ps)
         | M.CACHE_VADDR_INV (sc, _, _gsym) ->
              let ps =
                 specify_byindex false
                 :: try_concretize_whichcache sc
              in
              restrict insn (M.and_list ps)
         | _ -> []
      in
      let perinsn = List.concat (List.map once insns) in
      cachegolem_asserts @ perinsn
   in

   (*
    * Now apply SKINSNS, if any.
    *)
   let insnor2, moreconstraints =
      match skinsns with
      | None -> None, []
      | Some insns ->
           (*
            * 1. INSNS is a list of possible instruction choices.
            * If it doesn't contain "anything" (which appears here as -1),
            * we'll use it constrain the choice of opcodes instead of
            * the overall list. XXX: this assumes that all registers
            * in the sketch are enabled; the gating logic ought to
            * be checking the sketch but currently doesn't.
            *)
           let opcodechoices =
              let oneinsn z (opcode, _operands) =
                 if opcode = -1 then None else
                 match z with
                 | None -> None
                 | Some e ->
                      let insn = M.INSNVALUE (Insns.decode opset opcode) in
                      let readvar = M.READVAR (insnname, M.INSN) in
                      let e' = M.LOGOR (e, M.eq_logic readvar insn) in
                      Some e'
              in
              List.fold_left oneinsn (Some M.FALSE) insns
           in
           (*
            * 2. The choice of instruction may come with choices for
            * operands/instruction fields, so apply those as implications.
            *)
           let moreconstraints =
              let oneinsn (opcode, operands) =
                 let oneoperand (field, values) =
                    let field', ty' = Insns.get_fieldname_sym field opset in
                    let field'' = M.READVAR (M.CONTROLVAR field', ty') in
                    let oneval v =
                       M.eq_logic field'' v
                    in
                    M.or_list (List.map oneval values)
                 in
                 let operands' = M.and_list (List.map oneoperand operands) in
                 if opcode = -1 then operands' else
                 let insn = M.INSNVALUE (Insns.decode opset opcode) in
                 let readvar = M.READVAR (insnname, M.INSN) in
                 let eq = M.eq_logic readvar insn in
                 M.IMPLIES (eq, operands')
              in
              List.map oneinsn insns
           in
           opcodechoices, moreconstraints
   in
   let insnor =
      (* insnor2 takes priority if present *)
      match insnor2 with
      | None -> insnor
      | Some e -> e
   in
   (logicenv', M.and_list (insnor :: constraints @ moreconstraints))


(*
 * Create n symbolic instructions.
 *
 * Produce a logicenv that holds declarations of the logic variables
 * needed to describe these instructions. These are logic-expression
 * level, not smt-expression level. Note that we do not need to declare
 * processor state at this point.
 *)
let make_symbolic_insns' prebody =
   let ppenv = Ppoint.generate_prep () in

   let prebody = Util.number prebody in

   (* make all the instructions first *)

   let insnnames =
      List.map (fun (num, skinsns) -> (num, (num, "opcode"), skinsns)) prebody
   in
   let insns =
      let mk (_num, sym, _skinsns) =
         (Ppoint.generate ppenv, SYM sym)
      in
      List.map mk insnnames
   in
   let code = (insns, Ppoint.generate ppenv) in
   let allpps = Ppoint.generated ppenv in

   (* now make the logic variables *)

   let logicenv =
      let add env (_num, name, _skinsns) =
         M.VarMap.add (M.CONTROLVAR name) M.INSN env
      in
      List.fold_left add M.VarMap.empty insnnames
   in

   let (logicenv', props) =
      let once (lenv, props) (num, name, skinsns) =
         let sym = M.CONTROLVAR name in       
         let desc = "structural assertions for insn " ^ string_of_int num in
         let (lenv', insnprop) = populate_symbolic_insn lenv num sym skinsns in
         (lenv', (desc, insnprop) :: props)
      in
      List.fold_left once (logicenv, []) insnnames
   in

   (logicenv', List.rev (props : (string * Mips.expr) list), allpps, code)

(*
 * Create n symbolic instructions and apply a sketch
 *
 * XXX: this does not handle all symmetry cases of distributing
 * instructions across two or more ANYTHING sections; if the number of
 * instructions doesn't divide evenly it will always place the
 * leftover k instructions one each in the first k ANYTHING sections.
 * Fixing this would require being able to return multiple symbolic
 * programs though (and then schedule them somehow upstream) and that
 * seems problematic. Probably the right fix for this is to pass a
 * more structured configuration in than just N and so that upstream
 * can do the scheduling/planning first and then request the
 * combination it currently wants. But that's also nontrivial. For the
 * moment this doesn't matter for any of the things we want the sketch
 * for so it's not worth worrying about.
 *)
let make_symbolic_insns n sketch =
   match sketch with
   | None -> Some (make_symbolic_insns' (List.init n (fun _ -> None)))
   | Some entries ->
        let (numanything, numchoices) =
           let once (na, nc) se =
              match se with
              | M.ANYTHING -> (na + 1, nc)
              | M.CHOICES _ -> (na, nc + 1)
           in
           List.fold_left once (0, 0) entries
        in
        (* fail if the sketch doesn't allow as many as n instructions *)
        if numanything = 0 && numchoices < n then None else

        (*
         * Each CHOICES is one instruction; if n < numchoices we'll
         * take only the first n CHOICES in the sketch (and drop the
         * rest; oh well)... if n > numchoices we can apportion the
         * remaining number of instructions across the ANYTHING blocks.
         *)
        let base_numanything, extra_numanything =
           if n <= numchoices then 0, 0
           else if numanything = 0 then 0, 0
           else begin
              let anything_total = n - numchoices in
              let base = anything_total / numanything in
              let extra = anything_total mod numanything in
              (base, extra)
           end
        in
        let rec doassign extra_left choices_left ents =
           match ents with
           | [] -> []
           | ent :: more ->
                let n, extra_left', choices_left' =
                   match ent with
                   | M.ANYTHING ->
                        if extra_left > 0 then
                           base_numanything + 1, extra_left - 1, choices_left
                        else
                           base_numanything, extra_left, choices_left
                   | M.CHOICES _ ->
                        if choices_left > 0 then
                           1, extra_left, choices_left - 1
                        else
                           0, extra_left, choices_left
                in
                (n, ent) :: doassign extra_left' choices_left' more
        in
        let assignments = doassign extra_numanything numchoices entries in
        let generate (n, ent) =
           match n, ent with
           | 0, _ -> []
           | _, M.ANYTHING -> List.init n (fun _ -> None)
           | 1, M.CHOICES insns -> [Some insns]
           | _, M.CHOICES _ -> assert false
        in
        let prebody = List.concat (List.map generate assignments) in
        let body = make_symbolic_insns' prebody in
        Some body
