(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

open Symbolic
open Mipsstate
open Mips
module M = Mips (* XXX remove later *)
module T = Mipstools
module D = Dumptools

let linemax = 72

let s'cachestate s =
   match s with
   | SYM sym -> "<<" ^ string_of_symi sym ^ ">>"
   | CONC cs ->
	match cs with
	| CACHE_PRESENT_DIRTY -> "present_dirty"
	| CACHE_PRESENT_CLEAN -> "present_clean"
	| CACHE_ABSENT_VALID -> "absent_valid"
	| CACHE_ABSENT_INVALID -> "absent_invalid"

let s'reg r =
   match r with
   | CONC r' -> "$" ^ string_of_reg r'
   | SYM sym -> "$<" ^ string_of_symi sym ^ ">"

let s'creg cr =
   match cr with
   | CONC cr' -> "$c0." ^ string_of_creg cr'
   | SYM sym -> "$c0.<" ^ string_of_symi sym ^ ">"

let s'freg fr =
   match fr with
   | CONC fr' -> "$f" ^ string_of_freg fr'
   | SYM sym -> "$f<" ^ string_of_symi sym ^ ">"

let s'fcreg fcr =
   match fcr with
   | CONC fcr' -> "$fc." ^ string_of_fcreg fcr'
   | SYM sym -> "$fc.<" ^ string_of_symi sym ^ ">"

let s'golem g =
   match g with
   | CONC g' -> string_of_golem g'
   | SYM sym -> "$g.<" ^ string_of_symi sym ^ ">"

let s'whichcache c =
   match c with
   | CONC c' -> string_of_cache c'
   | SYM sym -> "<" ^ string_of_symi sym ^ ">"

let s'whichcacheentryid ce =
   match ce with
   | CONC ce' -> string_of_cacheentryid ce'
   | SYM sym -> "<" ^ string_of_symi sym ^ ">"

let s'imm imm =
   match imm with
   | CONC n -> string_of_int n
   | SYM sym -> "<" ^ string_of_symi sym ^ ">"

let s'simm = s'imm
let s'uimm = s'imm

let s'extcodelabel l =
   match l with
   | CONC txt -> txt ^ "@code"
   | SYM sym -> "<" ^ string_of_symi sym ^ ">" ^ "@code"

let s'extdatalabel l =
   match l with
   | CONC txt -> txt ^ "@data"
   | SYM sym -> "<" ^ string_of_symi sym ^ ">" ^ "@data"

let s'localcodelabel l =
   match l with
   | CONC pp -> Ppoint.to_string pp
   | SYM sym -> "pp.<" ^ string_of_symi sym ^ ">"

(*
let s'whichcacheentry (c, ce) =
   "$cacheentry." ^ s'whichcache c ^ "." ^ s'whichcacheentryid ce

let s'stateelement se =
   match se with
   | REG r -> s'reg r
   | CREG cr -> s'creg cr
   | FREG fr -> s'freg fr
   | FCREG fcr -> s'fcreg fcr
   | HI -> "$hi"
   | LO -> "$lo"
   | CACHEENTRY ce -> s'whichcacheentry ce
*)

let rec s'typename ty =
   match ty with
   | BOOL -> "bool"
   | CACHESTATE -> "cachestate"
   | UINT5 -> "uint5"
   | SINT8 -> "sint8"
   | UINT8 -> "uint8"
   | SINT16 -> "sint16"
   | UINT16 -> "uint16"
   | SINT32 -> "sint32"
   | UINT32 -> "uint32"
   | REGISTER -> "register"
   | CREGISTER -> "cregister"
   | FREGISTER -> "fregister"
   | FCREGISTER -> "fcregister"
   | HIREGISTER -> "hiregister"
   | LOREGISTER -> "loregister"
   | GOLEM -> "golem"
   | WHICHCACHE -> "whichcache"
   | WHICHCACHEENTRYID -> "whichcacheentryid"
   | WHICHCACHEENTRY -> "whichcacheentry"
   | PROGRAMPOINT -> "programpoint"
   | INSN -> "insn"
   | MAP (t1, t2) -> "map(" ^ s'typename t1 ^ " -> " ^ s'typename t2 ^ ")"

let s'insn insn =
   let modify m txt =
      match m with
      | Some mo -> "%" ^ mo ^ "(" ^ txt ^ ")"
      | None -> txt
   in

   let rrx op rd rs x =
      "[" ^ op ^ " " ^ s'reg rd ^ ", " ^ s'reg rs ^ ", " ^ x ^ "]"
   in
   let rrr op rd rs rt = rrx op rd rs (s'reg rt) in
   let rrs op rd rs simm = rrx op rd rs (s'simm simm) in
   let rru op rd rs uimm = rrx op rd rs (s'uimm uimm) in
   let rrxc op rd rs mo xc = rrx op rd rs (modify mo (s'extcodelabel xc)) in
   let rrxd op rd rs mo xd = rrx op rd rs (modify mo (s'extdatalabel xd)) in
   let rrlc op rd rs mo lc = rrx op rd rs (modify mo (s'localcodelabel lc))in

   let rx op rd x =
      "[" ^ op ^ " " ^ s'reg rd ^ ", " ^ x ^ "]"
   in
   let rr op rd rs = rx op rd (s'reg rs) in
   let ru op rd uimm = rx op rd (s'uimm uimm) in
   let rxc op rd mo xc = rx op rd (modify mo (s'extcodelabel xc)) in
   let rxd op rd mo xd = rx op rd (modify mo (s'extdatalabel xd)) in
   let rlc op rd mo lc = rx op rd (modify mo (s'localcodelabel lc)) in

   let xm op x m =
      let MEMORY (rs, imm) = m in
      "[" ^ op ^ " " ^ x ^ ", " ^ s'imm imm ^ "(" ^ s'reg rs ^ ")]"
   in
   let rm op r m = xm op (s'reg r) m in
   let fm op f m = xm op (s'freg f) m in

   let cache action c m =
      xm "cache" (action ^ "@" ^ s'whichcache c) m
   in

   match insn with
   (* arithmetic *)
   | M.ADDU (rd, rs, rt) ->   rrr "addu" rd rs rt
   | M.ADDIU (rd, rs, imm) -> rrs "addu" rd rs imm
   | M.DIV (rs, rt) ->        rr "div" rs rt
   | M.DIVU (rs, rt) ->       rr "divu" rs rt
   | M.LUI (rd, imm) ->       ru "lui" rd imm
   | M.MULT (rs, rt) ->       rr "mult" rs rt
   | M.MULTU (rs, rt) ->      rr "multu" rs rt
   | M.SUBU (rd, rs, rt) ->   rrr "subu" rd rs rt
   | M.LUI_extcode_hi (rd, l) ->         rxc "lui"    rd    (Some "hi") l
   | M.ADDIU_extcode_lo (rd, rs, l) ->   rrxc "addiu" rd rs (Some "lo") l
   | M.LUI_extdata_hi (rd, l) ->         rxd "lui"    rd    (Some "hi") l
   | M.ADDIU_extdata_lo (rd, rs, l) ->   rrxd "addiu" rd rs (Some "lo") l
   | M.LUI_localcode_hi (rd, l) ->       rlc "lui"    rd    (Some "hi") l
   | M.ADDIU_localcode_lo (rd, rs, l) -> rrlc "addiu" rd rs (Some "lo") l
   (* comparisons *)
   | M.SLT (rd, rs, rt) ->    rrr "slt" rd rs rt
   | M.SLTI (rd, rs, imm) ->  rrs "slti" rd rs imm
   | M.SLTU (rd, rs, rt) ->   rrr "sltu" rd rs rt
   | M.SLTIU (rd, rs, imm) -> rrs "sltiu" rd rs imm
   (* bitwise logic *)
   | M.AND (rd, rs, rt) ->    rrr "and" rd rs rt
   | M.ANDI (rd, rs, imm) ->  rru "andi" rd rs imm
   | M.NOR (rd, rs, rt) ->    rrr "nor" rd rs rt
   | M.OR (rd, rs, rt) ->     rrr "or" rd rs rt
   | M.ORI (rd, rs, imm) ->   rru "ori" rd rs imm
   | M.XOR (rd, rs, rt) ->    rrr "xor" rd rs rt
   | M.XORI (rd, rs, imm) ->  rru "xori" rd rs imm
   (* shifts *)
   | M.SLL (rd, rs, imm) ->   rru "sll" rd rs imm
   | M.SLLV (rd, rs, rt) ->   rrr "sllv" rd rs rt
   | M.SRA (rd, rs, imm) ->   rru "sra" rd rs imm
   | M.SRAV (rd, rs, rt) ->   rrr "srav" rd rs rt
   | M.SRL (rd, rs, imm) ->   rru "srl" rd rs imm
   | M.SRLV (rd, rs, rt) ->   rrr "srlv" rd rs rt
   (* transfers *)
   | M.B l ->                 "[b " ^ s'localcodelabel l ^ "]"
   | M.BEQ (rd, rs, l) ->     rrlc "beq" rd rs None l
   | M.BNE (rd, rs, l) ->     rrlc "bne" rd rs None l
   | M.J l ->                 "[j " ^ s'extcodelabel l ^ "]"
   | M.JAL l ->               "[jal " ^ s'extcodelabel l ^ "]"
   | M.JALR rs ->             "[jalr " ^ s'reg rs ^ "]"
   | M.JR rs ->               "[jr " ^ s'reg rs ^ "]"
   (* moves *)
   | M.CFC1 (rd, fcs) ->      "[cfc1 " ^ s'reg rd ^ ", " ^ s'fcreg fcs ^ "]"
   | M.CTC1 (rd, fcs) ->      "[ctc1 " ^ s'reg rd ^ ", " ^ s'fcreg fcs ^ "]"
   | M.MFC0 (rd, cs) ->       "[mfc0 " ^ s'reg rd ^ ", " ^ s'creg cs ^ "]"
   | M.MFC1 (rd, fs) ->       "[mfc1 " ^ s'reg rd ^ ", " ^ s'freg fs ^ "]"
   | M.MFHI rd ->             "[mfhi " ^ s'reg rd ^ "]"
   | M.MFLO rd ->             "[mflo " ^ s'reg rd ^ "]"
   | M.MTC0 (rd, cs) ->       "[mtc0 " ^ s'reg rd ^ ", " ^ s'creg cs ^ "]"
   | M.MTC1 (rd, fs) ->       "[mtc1 " ^ s'reg rd ^ ", " ^ s'freg fs ^ "]"
   | M.MTHI rs ->             "[mthi " ^ s'reg rs ^ "]"
   | M.MTLO rs ->             "[mthi " ^ s'reg rs ^ "]"
   | M.MOVN (rd, rs, rt) ->   rrr "movn" rd rs rt
   | M.MOVZ (rd, rs, rt) ->   rrr "movz" rd rs rt
   (* loads *)
   | M.LB (rd, mem) ->        rm "lb" rd mem
   | M.LBU (rd, mem) ->       rm "lbu" rd mem
   | M.LDC1 (fd, mem) ->      fm "ldc1" fd mem
   | M.LH (rd, mem) ->        rm "lh" rd mem
   | M.LHU (rd, mem) ->       rm "lhu" rd mem
   | M.LW (rd, mem) ->        rm "lw" rd mem
   | M.LWC1 (fd, mem) ->      fm "lwc1" fd mem
   (* stores *)
   | M.SB (rd, mem) ->        rm "sb" rd mem
   | M.SDC1 (fd, mem) ->      fm "sdc1" fd mem
   | M.SH (rd, mem) ->        rm "sh" rd mem
   | M.SW (rd, mem) ->        rm "sw" rd mem
   | M.SWC1 (fd, mem) ->      fm "swc1" fd mem
   (* system control *)
   | M.CACHE_INDEX_WBINV (c, mem, _) -> cache "wbinv" c mem
   | M.CACHE_VADDR_WB (c, mem, _) ->    cache "wb" c mem
   | M.CACHE_VADDR_WBINV (c, mem, _) -> cache "wbinv" c mem
   | M.CACHE_VADDR_INV (c, mem, _) ->   cache "inv" c mem
   | M.TLBP ->                "tlbp"
   | M.TLBR ->                "tlbr"
   | M.TLBWI ->               "tlbwi"
   | M.TLBWR ->               "tlbwr"


(*
 * slightly mangy prettyprinter
 *)

let isor e =
   match e with
   | LOGOR (e1, e2) -> Some (e1, e2)
   | _ -> None

let isxor e =
   match e with
   | LOGXOR (e1, e2) -> Some (e1, e2)
   | _ -> None

let isand e =
   match e with
   | LOGAND (e1, e2) -> Some (e1, e2)
   | _ -> None

let collect p e1 e2 =
   let rec visit e =
      match p e with
      | Some (e1, e2) -> visit e1 @ visit e2
      | None -> [e]
   in
   visit e1 @ visit e2

let indent e' =
   let once txt = "   " ^ txt in
   List.map once e'

let parts_se s0 e1 =
   let s1 = String.concat " " e1 in
   let s = String.concat " " [s0; s1] in
   if String.length s < linemax then [s]
   else s0 :: indent e1

let parts_ese e0 s1 e2 =
   let s0 = String.concat " " e0 in
   let s2 = String.concat " " e2 in
   let s = String.concat " " [s0; s1; s2] in
   if String.length s < linemax then [s]
   else indent e0 @ s1 :: indent e2

let parts_sese s0 e1 s2 e3 =
   let s1 = String.concat " " e1 in
   let s3 = String.concat " " e3 in
   let s = String.concat " " [s0; s1; s2; s3] in
   if String.length s < linemax then [s]
   else begin
      let s' = String.concat " " [s0; s1; s2] in
      if String.length s < linemax then s' :: e3
      else s0 :: indent e1 @ [s2] @ e3
   end

let parts_sesese s0 e1 s2 e3 s4 e5 =
   let pair a b b' =
      let ab = String.concat " " [a; b] in
      if String.length ab < linemax then [ab] else a :: indent b'
   in

   let s1 = String.concat " " e1 in
   let s3 = String.concat " " e3 in
   let s5 = String.concat " " e5 in
   let s05 = String.concat " " [s0; s1; s2; s3; s4; s5] in
   if String.length s05 < linemax then [s05]
   else begin
      let s45 = pair s4 s5 e5 in
      let s03 = String.concat " " [s0; s1; s2; s3] in
      if String.length s03 < linemax then s03 :: s45
      else begin
         let s01 = pair s0 s1 e1 in
         let s23 = pair s2 s3 e3 in
         s01 @ s23 @ s45
      end
   end

let pwrap precout (precin, e') =
   if precin > precout then e'
   else match e' with
      | [] -> [] (* notreached *)
      | [txt] -> ["(" ^ txt ^ ")"]
      | _ -> "(" :: indent e' @ [")"]

let print'var x ty =
   let x' = string_of_varname x in
   x' ^ " :: " ^ s'typename ty

let rec print'expr e =
   let collectable_bop prec pred op opall e1 e2 =
      let es = collect pred e1 e2 in
      if List.length es > 4 then
         let es' = List.map (fun e -> pwrap prec (print'expr e)) es in
         let es'' = List.concat (List.map indent es') in
         (prec, opall :: indent es'')
      else
         let e1' = pwrap prec (print'expr e1) in
         let e2' = pwrap prec (print'expr e2) in
         (prec, parts_ese e1' op e2')
   in

   match e with
     (* constants *)
   | TRUE -> (99, ["true"])
   | FALSE -> (99, ["false"])
   | CACHESTATEVALUE cs -> (99, [s'cachestate (CONC cs)])

     (* constructors *)
   | INTVALUE (n, ty) ->
        (99, [string_of_int n ^ " :: " ^ s'typename ty])
   | REGVALUE r -> (99, [s'reg (CONC r)])
   | CREGVALUE cr -> (99, [s'creg (CONC cr)])
   | FREGVALUE fr -> (99, [s'freg (CONC fr)])
   | FCREGVALUE fcr -> (99, [s'fcreg (CONC fcr)])
   | HIREGVALUE -> (99, ["$hi"])
   | LOREGVALUE -> (99, ["$lo"])
   | GOLEMVALUE g -> (99, [s'golem (CONC g)])
   | WHICHCACHEVALUE c -> (99, [s'whichcache (CONC c)])
   | WHICHCACHEENTRYIDVALUE ce -> (99, [s'whichcacheentryid (CONC ce)])
   | WHICHCACHEENTRYVALUE (c, ce) ->
        let c' = s'whichcache (CONC c) in
        let ce' = s'whichcacheentryid (CONC ce) in
        (99, ["$cacheentry(" ^ c' ^ ", " ^ ce' ^ ")"])
   | PPVALUE pp -> (99, [Ppoint.to_string pp])
   | INSNVALUE insn -> (99, [s'insn insn])

     (* access to processor state, as of a particular program point *)
   | READSTATE (se, pp, zerosem) ->
        let zstr =
           match zerosem with
           | Z_ZERO -> " (z0_0)"
           | Z_ANY -> " (z0_*)"
        in
        let se' = pwrap 90 (print'expr se) in
        (99, se' @ [" @ " ^ Ppoint.to_string pp ^ zstr])
   | READGOLEM g ->
        let g' = pwrap 90 (print'expr g) in
        (* I _think_ & is nethack for golems... *)
        (99, "&" :: g')
(*
   | READALLSTATE (ty, pp) ->
        (99, ["all " ^ s'typename ty ^ " @ " ^ Ppoint.to_string pp])
   | UPDATEMAP (m, k, v) ->
        (* XXX 99? *)
        let m' = pwrap 90 (print'expr m) in
        let k' = pwrap 90 (print'expr k) in
        let v' = pwrap 90 (print'expr v) in
        (99, ["["] @ m' @ [" with "] @ k' @ [" -> "] @ v' @ ["]"])
*)

     (* access to logic variables *)
   | READVAR (x, ty) ->
        (99, [string_of_varname x ^ " :: " ^ s'typename ty])

     (* access to logic functions *)
   | GOLEMIZE (g, e1) ->
        let g' = pwrap 100 (print'expr g) in
        let e1' = pwrap 100 (print'expr e1) in
        (99, g' @ e1')

     (* logical operators *)
   | FORALL (x, ty, e1) ->
        let x' = print'var x ty in
        let e1' = pwrap 5 (print'expr e1) in
        (5, parts_se ("forall " ^ x' ^ ",") e1')
   | EXISTS (x, ty, e1) ->
        let x' = print'var x ty in
        let e1' = pwrap 5 (print'expr e1) in
        (5, parts_se ("exists " ^ x' ^ ",") e1')
   | LET (x, ty, e1, e2) ->
        let x' = print'var x ty in
        let e1' = pwrap 5 (print'expr e1) in
        let e2' = pwrap 5 (print'expr e2) in
        (5, parts_sese ("let " ^ x' ^ " :=") e1' "in" e2')
   | IF (c, t, f) ->
        let c' = pwrap 5 (print'expr c) in
        let t' = pwrap 5 (print'expr t) in
        let f' = pwrap 70 (print'expr f) in
        (70, parts_sesese "if" c' "then" t' "else" f')
   | IMPLIES (e1, e2) ->
        let e1' = pwrap 10 (print'expr e1) in
        let e2' = pwrap 10 (print'expr e2) in
        (10, parts_ese e1' "->" e2')
   | LOGNOT e1 ->
        let e1' = pwrap 90 (print'expr e1) in
        (90, parts_se "!" e1')
   | LOGOR (e1, e2) ->
        collectable_bop 20 isor "||" "||all" e1 e2
   | LOGXOR (e1, e2) ->
        collectable_bop 21 isxor "^^" "^^all" e1 e2
   | LOGAND (e1, e2) ->
        collectable_bop 22 isand "&&" "&&all" e1 e2
   | LOGEQ (e1, e2) ->
        let e1' = pwrap 40 (print'expr e1) in
        let e2' = pwrap 40 (print'expr e2) in
        (40, parts_ese e1' "==x" e2')

     (* other operators *)
   | MINTUOP (op, e1) ->
        let e1' = pwrap 90 (print'expr e1) in
        (90, parts_se (Mintops.string_of_uop op) e1')
   | MINTBOP (op, e1, e2) ->
        let prec = Mintops.prec_of_bop op in
        let e1' = pwrap prec (print'expr e1) in
        let e2' = pwrap prec (print'expr e2) in
        (prec, parts_ese e1' (Mintops.string_of_bop op) e2')

let print prefix e =
   let (_prec, strings) = print'expr e in
   prefix ^ String.concat ("\n" ^ prefix) strings ^ "\n"

let string_of_typename = s'typename
