(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

(*
 * Print the assembly for a code block.
 *)

open Symbolic
open Mipsstate
open Mips

let get_labeled_points (insns, _ppost) =
   let labelcount = ref 0 in
   let ret = ref Ppoint.Map.empty in
   let once (_pp, insn) =
      match insn with
      | SYM _ -> ()
      | CONC cinsn ->
	   match Mipstools.pp_of_insn cinsn with
	   | None -> ()
	   | Some pp ->
		let num = !labelcount in
		labelcount := 1 + num;
		let name = ".L" ^ string_of_int num in
		ret := Ppoint.Map.add pp name !ret
   in
   List.iter once insns;
   !ret


(**************************************************************)
(* convert instructions to strings *)

let crash i msg =
   Util.crash ("Instruction " ^ string_of_int i ^ ": " ^ msg)

let symcrash i sym =
   crash i ("Leftover symbolic value; symbol was " ^ string_of_symi sym)

let string_of_reg i r =
   match r with
   | SYM sym -> symcrash i sym
   | CONC r -> "$" ^ string_of_int (int_of_reg r)

let string_of_creg i r =
   match r with
   | SYM sym -> symcrash i sym
   | CONC cr -> "$" ^ string_of_int (int_of_creg cr)

let string_of_freg i r =
   match r with
   | SYM sym -> symcrash i sym
   | CONC fr -> "$f" ^ string_of_int (int_of_freg fr)

let string_of_fcreg i r =
   match r with
   | SYM sym -> symcrash i sym
   | CONC fcr -> "$f" ^ string_of_int (int_of_fcreg fcr)

let string_of_immed i lb ub imm =
   match imm with
   | SYM sym -> symcrash i sym
   | CONC n ->
        let ns = string_of_int n in
        if n < lb then
           let lbs = string_of_int lb in
           crash i ("Immediate out of range: " ^ ns ^ " < " ^ lbs)
        else if n > ub then
           let ubs = string_of_int ub in
           crash i ("Immediate out of range: " ^ ns ^ " > " ^ ubs)
        else ns

let string_of_localcodelabel labelenv i l =
   match l with
   | SYM sym -> symcrash i sym
   | CONC pp ->
        try Ppoint.Map.find pp labelenv
        with Not_found ->
           crash i ("No label for program point: " ^ Ppoint.to_string pp)

let string_of_extcodelabel i l =
   match l with
   | SYM sym -> symcrash i sym
   | CONC txt -> txt

let string_of_extdatalabel i l =
   match l with
   | SYM sym -> symcrash i sym
   | CONC txt -> txt

let string_of_cacheop i action c =
   (*
    * In the assembly language, the operation is chosen by 5 bits
    * written as a magic number; the top 3 are the action and the
    * bottom 2 identify the cache.
    *
    * actions:
    *    000   writeback and invalidate by index/way
    *    001   read tag (into special reg) by index/way
    *    010   store tag (from special reg) by index/way 
    *    011   vendor-dependent
    *    100   invalidate by vaddr
    *    101   writeback and invalidate by vaddr (but: fill for icache)
    *    110   writeback by vaddr
    *    111   fill and lock by vaddr
    *)
   assert (action >= 0 && action < 8);
   match c with
   | SYM csym -> symcrash i csym
   | CONC c' -> string_of_int (action * 4 + int_of_cache c')

let string_of_concreteinsn labelenv i insn =
   let reg r = string_of_reg i r in
   let creg r = string_of_creg i r in
   let freg r = string_of_freg i r in
   let fcreg r = string_of_fcreg i r in
   let regs = List.map reg in
(*
   let cregs = List.map creg in
   let fregs = List.map freg in
   let fcregs = List.map fcreg in
*)
   let simm16 imm = string_of_immed i (-32768) 32767 imm in
   let uimm16 imm = string_of_immed i 0 65535 imm in
   let uimm5 imm = string_of_immed i 0 31 imm in
   let xc l = string_of_extcodelabel i l in
   let xd l = string_of_extdatalabel i l in
   let lc l = string_of_localcodelabel labelenv i l in
   let hi s = "%hi(" ^ s ^ ")" in
   let lo s = "%lo(" ^ s ^ ")" in
   let mem (MEMORY (r, imm)) = simm16 imm ^ "(" ^ reg r ^ ")" in
   let cache action c = string_of_cacheop i action c in
   let op opcode operands =
      opcode ^ " " ^ String.concat ", " operands
   in
   match insn with
     (* arithmetic *)
   | ADDU (rd, rs, rt) ->     op "addu" (regs [rd; rs; rt])
   | ADDIU (rd, rs, imm) ->   op "addiu" (regs [rd; rs] @ [simm16 imm])
   | DIV (rs, rt) ->          op "div" (regs [CONC Z0; rs; rt])
   | DIVU (rs, rt) ->         op "divu" (regs [CONC Z0; rs; rt])
   | LUI (rd, imm) ->         op "lui" [reg rd; uimm16 imm]
   | MULT (rs, rt) ->         op "mult" (regs [CONC Z0; rs; rt])
   | MULTU (rs, rt) ->        op "multu" (regs [CONC Z0; rs; rt])
   | SUBU (rd, rs, rt) ->     op "subu" (regs [rd; rs; rt])
   | LUI_extcode_hi (rd, l)       -> op "lui" [reg rd; hi (xc l)]
   | ADDIU_extcode_lo (rd, rs, l) -> op "addiu" (regs [rd; rs] @ [lo (xc l)])
   | LUI_extdata_hi (rd, l)       -> op "lui" [reg rd; hi (xd l)]
   | ADDIU_extdata_lo (rd, rs, l) -> op "addiu" (regs [rd; rs] @ [lo (xd l)])
   | LUI_localcode_hi (rd, l)       -> op "lui" [reg rd; hi (lc l)]
   | ADDIU_localcode_lo (rd, rs, l) -> op "addiu" (regs [rd; rs] @ [lo (lc l)])
     (* comparisons *)
   | SLT (rd, rs, rt) ->      op "slt" (regs [rd; rs; rt])
   | SLTI (rd, rs, imm) ->    op "slti" (regs [rd; rs] @ [simm16 imm])
   | SLTU (rd, rs, rt) ->     op "sltu" (regs [rd; rs; rt])
   | SLTIU (rd, rs, imm) ->   op "sltiu" (regs [rd; rs] @ [simm16 imm])
     (* bitwise logic *)
   | AND (rd, rs, rt) ->      op "and" (regs [rd; rs; rt])
   | ANDI (rd, rs, imm) ->    op "andi" (regs [rd; rs] @ [uimm16 imm])
   | NOR (rd, rs, rt) ->      op "nor" (regs [rd; rs; rt])
   | OR (rd, rs, rt) ->       op "or" (regs [rd; rs; rt])
   | ORI (rd, rs, imm) ->     op "ori" (regs [rd; rs] @ [uimm16 imm])
   | XOR (rd, rs, rt) ->      op "xor" (regs [rd; rs; rt])
   | XORI (rd, rs, imm) ->    op "xori" (regs [rd; rs] @ [uimm16 imm])
     (* shifts *)
   | SLL (rd, rs, imm) ->     op "sll" (regs [rd; rs] @ [uimm5 imm])
   | SLLV (rd, rs, rt) ->     op "sllv" (regs [rd; rs; rt])
   | SRA (rd, rs, imm) ->     op "sra" (regs [rd; rs] @ [uimm5 imm])
   | SRAV (rd, rs, rt) ->     op "srav" (regs [rd; rs; rt])
   | SRL (rd, rs, imm) ->     op "srl" (regs [rd; rs] @ [uimm5 imm])
   | SRLV (rd, rs, rt) ->     op "srlv" (regs [rd; rs; rt])
     (* transfers *)
   | B l ->                   op "b" [lc l]
   | BEQ (rd, rs, l) ->       op "beq" (regs [rd; rs] @ [lc l])
   | BNE (rd, rs, l) ->       op "bne" (regs [rd; rs] @ [lc l])
   | J l ->                   op "j" [xc l]
   | JAL l ->                 op "jal" [xc l]
   | JALR rs ->               op "jalr" [reg rs]
   | JR rs ->                 op "jr" [reg rs]
     (* moves *)
   | CFC1 (rd, fcs) ->        op "cfc1" [reg rd; fcreg fcs]
   | CTC1 (rd, fcs) ->        op "ctc1" [reg rd; fcreg fcs]
   | MFC0 (rd, cs) ->         op "mfc0" [reg rd; creg cs]
   | MFC1 (rd, fs) ->         op "mfc1" [reg rd; freg fs]
   | MFHI rd ->               op "mfhi" [reg rd]
   | MFLO rd ->               op "mfhi" [reg rd]
   | MTC0 (rd, cs) ->         op "mtc0" [reg rd; creg cs]
   | MTC1 (rd, fs) ->         op "mtc1" [reg rd; freg fs]
   | MTHI rs ->               op "mthi" [reg rs]
   | MTLO rs ->               op "mtlo" [reg rs]
   | MOVN (rd, rs, rt) ->     op "movn" (regs [rd; rs; rt])
   | MOVZ (rd, rs, rt) ->     op "movz" (regs [rd; rs; rt])
     (* loads *)
   | LB (rd, m) ->            op "lb" [reg rd; mem m]
   | LBU (rd, m) ->           op "lbu" [reg rd; mem m]
   | LDC1 (fd, m) ->          op "ldc1" [freg fd; mem m]
   | LH (rd, m) ->            op "lh" [reg rd; mem m]
   | LHU (rd, m) ->           op "lhu" [reg rd; mem m]
   | LW (rd, m) ->            op "lw" [reg rd; mem m]
   | LWC1 (rd, m) ->          op "lwc1" [freg rd; mem m]
     (* stores *)
   | SB (rd, m) ->            op "sb" [reg rd; mem m]
   | SDC1 (fd, m) ->          op "sdc1" [freg fd; mem m]
   | SH (rd, m) ->            op "sh" [reg rd; mem m]
   | SW (rd, m) ->            op "sw" [reg rd; mem m]
   | SWC1 (fd, m) ->          op "swc1" [freg fd; mem m]
     (* system control *)
   | CACHE_INDEX_WBINV (c, m, _g) ->  op "cache" [cache 0 c; mem m]
   | CACHE_VADDR_WB (c, m, _g) ->     op "cache" [cache 6 c; mem m]
   | CACHE_VADDR_WBINV (c, m, _g) ->  op "cache" [cache 5 c; mem m]
   | CACHE_VADDR_INV (c, m, _g) ->    op "cache" [cache 4 c; mem m]
   | TLBP ->                  "tlbp"
   | TLBR ->                  "tlbr"
   | TLBWI ->                 "tlbwi"
   | TLBWR ->                 "tlbwr"

let string_of_insn labelenv i insn =
   match insn with
   | SYM sym -> symcrash i sym
   | CONC cinsn -> string_of_concreteinsn labelenv i cinsn

(**************************************************************)
(* toplevel *)

let print prefix code =
   let labelenv = get_labeled_points code in
   let printpp pp =
      try [Ppoint.Map.find pp labelenv ^ ":"]
      with Not_found -> []
   in
   let printpair (i, (pp, insn)) =
      printpp pp @ [string_of_insn labelenv i insn]
   in
   let (pairs, ppost) = code in
   let pairs' = Util.number pairs in
   let strings = List.concat (List.map printpair pairs') @ printpp ppost in
   prefix ^ String.concat ("\n" ^ prefix) strings ^ "\n"

