(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

open Symbolic
open Mipsstate
open Mips

(* Get the precondition program point for a code block *)
let prepoint (insns, post) =
   match insns with
   | [] -> post (* not supposed to happen, but whatever *)
   | (pre, _) :: _ -> pre

(* Get the postcondition program point for a code block *)
let postpoint (_, post) = post

(* Replace uses of opp with npp in e *)
let rec changepp e opp npp =
   let dopp pp =
      if pp = opp then npp else pp
   in
   match e with
   | TRUE
   | FALSE
   | CACHESTATEVALUE _
   | INTVALUE _
   | REGVALUE _
   | CREGVALUE _
   | FREGVALUE _
   | FCREGVALUE _
   | HIREGVALUE
   | LOREGVALUE
   | GOLEMVALUE _
   | WHICHCACHEVALUE _
   | WHICHCACHEENTRYIDVALUE _ -> e
   | WHICHCACHEENTRYVALUE _ -> e
   | PPVALUE pp -> PPVALUE (dopp pp)
   | INSNVALUE _ -> e
   | READSTATE (se, pp, zerosem) ->
        let se' = changepp se opp npp in
        READSTATE (se', dopp pp, zerosem)
   | READGOLEM g ->
        let g' = changepp g opp npp in
        READGOLEM g'
(*
   | READALLSTATE _ -> e
   | UPDATEMAP (m, k, v) ->
        let m' = changepp m opp npp in
        let k' = changepp k opp npp in
        let v' = changepp v opp npp in
        UPDATEMAP (m', k', v')
*)
   | READVAR _ -> e
   | GOLEMIZE (g, e1) ->
        let g' = changepp g opp npp in
        let e1' = changepp e1 opp npp in
        GOLEMIZE (g', e1')
   | FORALL (x, ty, e1) ->
        let e1' = changepp e1 opp npp in
        FORALL (x, ty, e1')
   | EXISTS (x, ty, e1) ->
        let e1' = changepp e1 opp npp in
        EXISTS (x, ty, e1')
   | LET (x, ty, e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        LET (x, ty, e1', e2')
   | IF (c, t, f) ->
        let c' = changepp c opp npp in
        let t' = changepp t opp npp in
        let f' = changepp f opp npp in
        IF (c', t', f')
   | IMPLIES (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        IMPLIES (e1', e2')
   | LOGNOT e1 ->
        let e1' = changepp e1 opp npp in
        LOGNOT e1'
   | LOGAND (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        LOGAND (e1', e2')
   | LOGOR (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        LOGOR (e1', e2')
   | LOGXOR (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        LOGXOR (e1', e2')
   | LOGEQ (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        LOGEQ (e1', e2')
   | MINTUOP (op, e1) ->
        let e1' = changepp e1 opp npp in
        MINTUOP (op, e1')
   | MINTBOP (op, e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        MINTBOP (op, e1', e2')
(*
   | INT_ULT (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_ULT (e1', e2')
   | INT_SLT (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_SLT (e1', e2')
   | INT_ADD (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_ADD (e1', e2')
   | INT_SUB (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_SUB (e1', e2')
   | INT_SMUL_HI (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_SMUL_HI (e1', e2')
   | INT_UMUL_HI (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_UMUL_HI (e1', e2')
   | INT_MUL_LO (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_MUL_LO (e1', e2')
   | INT_SDIV (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_SDIV (e1', e2')
   | INT_SMOD (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_SMOD (e1', e2')
   | INT_UDIV (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_UDIV (e1', e2')
   | INT_UMOD (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_UMOD (e1', e2')
   | INT_BITNOT e1 ->
        let e1' = changepp e1 opp npp in
        INT_BITNOT e1'
   | INT_BITAND (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_BITAND (e1', e2')
   | INT_BITOR (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_BITOR (e1', e2')
   | INT_BITXOR (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_BITXOR (e1', e2')
   | INT_SHL (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_SHL (e1', e2')
   | INT_USHR (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_USHR (e1', e2')
   | INT_SSHR (e1, e2) ->
        let e1' = changepp e1 opp npp in
        let e2' = changepp e2 opp npp in
        INT_SSHR (e1', e2')
   | INT_SBWIDEN e1 ->
        let e1' = changepp e1 opp npp in
        INT_SBWIDEN e1'
   | INT_SHWIDEN e1 ->
        let e1' = changepp e1 opp npp in
        INT_SHWIDEN e1'
   | INT_UBWIDEN e1 ->
        let e1' = changepp e1 opp npp in
        INT_UBWIDEN e1'
   | INT_UHWIDEN e1 ->
        let e1' = changepp e1 opp npp in
        INT_UHWIDEN e1'
   | INT_U5WIDEN e1 ->
        let e1' = changepp e1 opp npp in
        INT_U5WIDEN e1'
   | INT_TRUNCU5 e1 ->
        let e1' = changepp e1 opp npp in
        INT_TRUNCU5 e1'
   | INT_TOSIGNED e1 ->
        let e1' = changepp e1 opp npp in
        INT_TOSIGNED e1'
   | INT_TOUNSIGNED e1 ->
        let e1' = changepp e1 opp npp in
        INT_TOUNSIGNED e1'
*)

(*
 * extract the program point of a concrete local code label
 *)
let concrete_pp_of_localcodelabel l =
   match l with
   | CONC pp -> Some pp
   | SYM _ -> None

(*
 * extract the program point referenced by the insn, if any
 * (which only happens in local code labels)
 *)
let pp_of_insn insn =
   match insn with
   | LUI_localcode_hi (_, l) -> concrete_pp_of_localcodelabel l
   | ADDIU_localcode_lo (_, _, l) -> concrete_pp_of_localcodelabel l
   | B l -> concrete_pp_of_localcodelabel l
   | BEQ (_, _, l) -> concrete_pp_of_localcodelabel l
   | BNE (_, _, l) -> concrete_pp_of_localcodelabel l
   | _ -> None

(*
 * get all the things
 * (for instructions see insns.ml)
 *)

let all_cachestates_raw () =
   if !Enabled.use_cache then [
      CACHE_PRESENT_DIRTY;
      CACHE_PRESENT_CLEAN;
      CACHE_ABSENT_VALID;
      CACHE_ABSENT_INVALID;
   ] else []

let nonz0_regs_raw () =
   if !(Enabled.use_allregs) then
      [
	    AT; V0; V1; A0; A1; A2; A3;
	T0; T1; T2; T3; T4; T5; T6; T7;
	S0; S1; S2; S3; S4; S5; S6; S7;
        T8; T9; K0; K1; GP; SP; FP; RA;
      ]
   else
      [ V0; V1; T0; T1; ]

let all_regs_raw () =
   Z0 :: nonz0_regs_raw ()

let all_cregs_raw () =
   let full = [
      C0_STATUS;
      C0_INDEX;
      C0_RANDOM;
      C0_ENTRYLO;
      C0_ENTRYHI;
      C0_CONTEXT;
      C0_BADVADDR;
      C0_EPC;
      C0_COUNT;
      C0_COMPARE;
      C0_PRID;
   ] in
   List.filter Enabled.creg_enabled full

let all_fregs_raw () =
   if !Enabled.use_fpu then
      [
        F0; F1; F2; F3; F4; F5; F6; F7;
        F8; F9; F10; F11; F12; F13; F14; F15;
        F16; F17; F18; F19; F20; F21; F22; F23;
        F24; F25; F26; F27; F28; F29; F30; F31;
      ]
   else []

let all_fcregs_raw () =
   if !Enabled.use_fpu then [
      FC_FIR;
      FC_FCSR;
      FC_FCCR;
      FC_FEXR;
      FC_FENR;
   ] else []

let all_cacheentryids_raw () =
   if !Enabled.use_cache then [
      CACHE_GREEN;
      CACHE_BLUE;
   ] else []

let all_caches_raw () =
   if !Enabled.use_cache then [
      ICACHE;
      DCACHE;
      L2CACHE;
      L3CACHE;
   ] else []

let all_cacheentries_raw () =
   Util.cartesian (all_caches_raw ()) (all_cacheentryids_raw ())

let all_golems_raw () =
   let cachegolems =
      if !Enabled.use_cache then
         let caches = all_caches_raw () in
         let onepair c = [
            G_VADDR c;
            G_INDEX c;
         ] in
         List.concat (List.map onepair caches)
      else []
   in
   cachegolems (* @ ... *)

let all_cachestates () =
   List.map (fun c -> CONC c) (all_cachestates_raw ())

let nonz0_regs () =
   List.map (fun r -> CONC r) (nonz0_regs_raw ())

let all_regs () =
   List.map (fun r -> CONC r) (all_regs_raw ())

let all_cregs () =
   List.map (fun cr -> CONC cr) (all_cregs_raw ())

let all_fregs () =
   List.map (fun f -> CONC f) (all_fregs_raw ())

let all_fcregs () =
   List.map (fun fcr -> CONC fcr) (all_fcregs_raw ())

let all_cacheentries () =
   List.map (fun (c, ce) -> (CONC c, CONC ce)) (all_cacheentries_raw ())

let all_caches () =
   List.map (fun c -> CONC c) (all_caches_raw ())

let all_golems () =
   List.map (fun g -> CONC g) (all_golems_raw ())

let string_of_cregs cr =
   match cr with
   | C0_STATUS -> "status"
   | C0_INDEX -> "index"
   | C0_RANDOM -> "random"
   | C0_ENTRYLO -> "entrylo"
   | C0_ENTRYHI -> "entryhi"
   | C0_CONTEXT -> "context"
   | C0_BADVADDR -> "badvaddr"
   | C0_EPC -> "epc"
   | C0_COUNT -> "count"
   | C0_COMPARE -> "compare"
   | C0_PRID -> "prid"

let string_of_fcregs fcr =
   match fcr with
   | FC_FIR -> "fir"
   | FC_FCSR -> "fcsr"
   | FC_FCCR -> "fccr"
   | FC_FEXR -> "fexr"
   | FC_FENR -> "fenr"

let all_stateelements () =
   List.map (fun r -> REG r) (nonz0_regs ()) @
   List.map (fun r -> CREG r) (all_cregs ()) @
   List.map (fun r -> FREG r) (all_fregs ()) @
   List.map (fun r -> FCREG r) (all_fcregs ()) @
   if !Enabled.use_hilo then [HI; LO] else [] @
   List.map (fun ce -> CACHEENTRY ce) (all_cacheentries ())
