(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

open Symbolic

(*
 * Elements of processor state.
 *)

type regs =
| Z0 | AT				(* 0-1 *)
| V0 | V1				(* 2-3 *)
| A0 | A1 | A2 | A3			(* 4-7 *)
| T0 | T1 | T2 | T3 | T4 | T5 | T6 | T7	(* 8-15 *)
| S0 | S1 | S2 | S3 | S4 | S5 | S6 | S7	(* 16-23 *)
| T8 | T9				(* 24-25 *)
| K0 | K1				(* 26-27 *)
| GP					(* 28 *)
| SP					(* 29 *)
| FP (* or S8 *)			(* 30 *)
| RA					(* 31 *)

type cregs =
| C0_STATUS	(* 12 *)
| C0_INDEX	(* 0 *)
| C0_RANDOM	(* 1 *)
| C0_ENTRYLO (* 2 *)
| C0_ENTRYHI (* 10 *)
| C0_CONTEXT	(* 4 *)
| C0_BADVADDR (* 8 *)
| C0_EPC	(* 14 *)
| C0_COUNT	(* 9 *)
| C0_COMPARE	(* 11 *)
| C0_PRID	(* 31 *)

type fregs =
| F0  | F1  | F2  | F3  | F4  | F5  | F6  | F7
| F8  | F9  | F10 | F11 | F12 | F13 | F14 | F15
| F16 | F17 | F18 | F19 | F20 | F21 | F22 | F23
| F24 | F25 | F26 | F27 | F28 | F29 | F30 | F31

type fcregs =
| FC_FIR (* 0 *)
| FC_FCSR (* 31 *)
| FC_FCCR (* 25 *)
| FC_FEXR (* 26 *)
| FC_FENR (* 28 *)

let register_bitwidth = 5

(*
 * Cache model:
 *
 * Each cache is represented by two cache entries (green and blue)
 * which hold metadata of type cachestate. The entry can be selected
 * by either vaddr or index and way. Instead of modeling the actual
 * cache workings, the (trivial) functions that choose the entry based
 * on the selection use the corresponding golem state variable for the
 * cache in order to make them reliably different.
 *
 * This is believed to be sufficient to make the solver choose the
 * correct cache operation and addressing mode, which is all we really
 * need to synthesize cache control ops.
 *)

type cachestate =
| CACHE_PRESENT_DIRTY
| CACHE_PRESENT_CLEAN
| CACHE_ABSENT_VALID
| CACHE_ABSENT_INVALID

type cacheentryid =
| CACHE_GREEN
| CACHE_BLUE

type cache =
| ICACHE
| DCACHE
| L2CACHE
| L3CACHE

type cacheentry = cache * cacheentryid

let cachestate_bitwidth = 2
let whichcache_bitwidth = 2
let whichcacheentryid_bitwidth = 1
let whichcacheentry_bitwidth = 3

(*
 * Golem state:
 *
 * "Golem" state is artificial machine state introduced to create
 * simple models of complicated machine components. The premise is
 * that any two predicates (even if simple and not the least opaque)
 * can be made universally distinguishable to the solver by inserting
 * a dummy boolean into them and universally quantifying. To get the
 * quantification the dummy state is treated as machine state in the
 * CEGIS loop; but it's not real machine state because we don't ever
 * modify it. (This means it isn't indexed by program point, so the
 * output logic is simpler.) The name is a weak joke on "skolem" and
 * on it being artificial.
 *)

type golems =
| G_INDEX of cache		(* cache index -> cache entry *)
| G_VADDR of cache		(* vaddr -> cache entry *)

let golem_bitwidth = whichcache_bitwidth + 1

(*
 * Symbolic versions
 *)

type sireg =   regs symbolic    (* general register *)
type sicreg =  cregs symbolic   (* control register *)
type sifreg =  fregs symbolic   (* fpu register *)
type sifcreg = fcregs symbolic  (* fpu control register *)
type sigolem = golems symbolic  (* golem state *)

(* which cache in simple cache model *)
type sicache = cache symbolic
(* which entry in simple cache model *)
type sicacheentry = (cache symbolic * cacheentryid symbolic)
(* golem state for simple cache model *)
type sicachegolem = (cache symbolic * sigolem)

(*
 * Labels for addressing processor state
 *
 * Also note that it's important to be able to read a symbolic state
 * element in a logic expression, so it's not an accident that this
 * uses the sreg/screg/sfreg/sfcreg types that can be symbolic.
 *)

(*
type stateelement =
| REG of sreg
| CREG of screg
| FREG of sfreg
| FCREG of sfcreg
| HI
| LO
| CACHEENTRY of scacheentry
*)

type stateelement =
| REG of sireg
| CREG of sicreg
| FREG of sifreg
| FCREG of sifcreg
| HI
| LO
| CACHEENTRY of sicacheentry


(*
 * Tools
 *)

let int_of_cachestate cs =
   match cs with
   | CACHE_PRESENT_DIRTY -> 0
   | CACHE_PRESENT_CLEAN -> 1
   | CACHE_ABSENT_VALID -> 2
   | CACHE_ABSENT_INVALID -> 3

let string_of_cachestate cs =
   match cs with
   | CACHE_PRESENT_DIRTY -> "CACHE_PRESENT_DIRTY"
   | CACHE_PRESENT_CLEAN -> "CACHE_PRESENT_CLEAN"
   | CACHE_ABSENT_VALID -> "CACHE_ABSENT_VALID"
   | CACHE_ABSENT_INVALID -> "CACHE_ABSENT_INVALID"

let int_of_reg r =
   match r with
   | Z0 -> 0
   | AT -> 1
   | V0 -> 2
   | V1 -> 3
   | A0 -> 4
   | A1 -> 5
   | A2 -> 6
   | A3 -> 7
   | T0 -> 8
   | T1 -> 9
   | T2 -> 10
   | T3 -> 11
   | T4 -> 12
   | T5 -> 13
   | T6 -> 14
   | T7 -> 15
   | S0 -> 16
   | S1 -> 17
   | S2 -> 18
   | S3 -> 19
   | S4 -> 20
   | S5 -> 21
   | S6 -> 22
   | S7 -> 23
   | T8 -> 24
   | T9 -> 25
   | K0 -> 26
   | K1 -> 27
   | GP -> 28
   | SP -> 29
   | FP -> 30
   | RA -> 31

let reg_of_int n =
   match n with
   | 0 -> Z0
   | 1 -> AT
   | 2 -> V0
   | 3 -> V1
   | 4 -> A0
   | 5 -> A1
   | 6 -> A2
   | 7 -> A3
   | 8 -> T0
   | 9 -> T1
   | 10 -> T2
   | 11 -> T3
   | 12 -> T4
   | 13 -> T5
   | 14 -> T6
   | 15 -> T7
   | 16 -> S0
   | 17 -> S1
   | 18 -> S2
   | 19 -> S3
   | 20 -> S4
   | 21 -> S5
   | 22 -> S6
   | 23 -> S7
   | 24 -> T8
   | 25 -> T9
   | 26 -> K0
   | 27 -> K1
   | 28 -> GP
   | 29 -> SP
   | 30 -> FP
   | 31 -> RA
   | _ -> Util.crash ("reg_of_int: " ^ string_of_int n)

let string_of_reg r =
   match r with
   | Z0 -> "z0"
   | AT -> "AT"
   | V0 -> "v0"
   | V1 -> "v1"
   | A0 -> "a0"
   | A1 -> "a1"
   | A2 -> "a2"
   | A3 -> "a3"
   | T0 -> "t0"
   | T1 -> "t1"
   | T2 -> "t2"
   | T3 -> "t3"
   | T4 -> "t4"
   | T5 -> "t5"
   | T6 -> "t6"
   | T7 -> "t7"
   | S0 -> "s0"
   | S1 -> "s1"
   | S2 -> "s2"
   | S3 -> "s3"
   | S4 -> "s4"
   | S5 -> "s5"
   | S6 -> "s6"
   | S7 -> "s7"
   | T8 -> "t8"
   | T9 -> "t9"
   | K0 -> "k0"
   | K1 -> "k1"
   | GP -> "gp"
   | SP -> "sp"
   | FP -> "fp"
   | RA -> "ra"

let int_of_creg cr =
   match cr with
   | C0_STATUS -> 12
   | C0_INDEX -> 0
   | C0_RANDOM -> 1
   | C0_ENTRYLO -> 2
   | C0_ENTRYHI -> 10
   | C0_CONTEXT -> 4
   | C0_BADVADDR -> 8
   | C0_EPC -> 14
   | C0_COUNT -> 9
   | C0_COMPARE -> 11
   | C0_PRID -> 31

let creg_of_int n =
   match n with
   | 12 -> C0_STATUS
   | 0 -> C0_INDEX
   | 1 -> C0_RANDOM
   | 2 -> C0_ENTRYLO
   | 10 -> C0_ENTRYHI
   | 4 -> C0_CONTEXT
   | 8 -> C0_BADVADDR
   | 14 -> C0_EPC
   | 9 -> C0_COUNT
   | 11 -> C0_COMPARE
   | 31 -> C0_PRID
   | _ -> Util.crash ("creg_of_int: " ^ string_of_int n)

let string_of_creg cr =
   match cr with
   | C0_STATUS -> "c0_status"
   | C0_INDEX -> "c0_index"
   | C0_RANDOM -> "c0_random"
   | C0_ENTRYLO -> "c0_entrylo"
   | C0_ENTRYHI -> "c0_entryhi"
   | C0_CONTEXT -> "c0_context"
   | C0_BADVADDR -> "c0_badvaddr"
   | C0_EPC -> "c0_epc"
   | C0_COUNT -> "c0_count"
   | C0_COMPARE -> "c0_compare"
   | C0_PRID -> "c0_prid"

let int_of_freg fr =
   match fr with
   | F0 -> 0
   | F1 -> 1
   | F2 -> 2
   | F3 -> 3
   | F4 -> 4
   | F5 -> 5
   | F6 -> 6
   | F7 -> 7
   | F8 -> 8
   | F9 -> 9
   | F10 -> 10
   | F11 -> 11
   | F12 -> 12
   | F13 -> 13
   | F14 -> 14
   | F15 -> 15
   | F16 -> 16
   | F17 -> 17
   | F18 -> 18
   | F19 -> 19
   | F20 -> 20
   | F21 -> 21
   | F22 -> 22
   | F23 -> 23
   | F24 -> 24
   | F25 -> 25
   | F26 -> 26
   | F27 -> 27
   | F28 -> 28
   | F29 -> 29
   | F30 -> 30
   | F31 -> 31

let freg_of_int n =
   match n with
   | 0 -> F0
   | 1 -> F1
   | 2 -> F2
   | 3 -> F3
   | 4 -> F4
   | 5 -> F5
   | 6 -> F6
   | 7 -> F7
   | 8 -> F8
   | 9 -> F9
   | 10 -> F10
   | 11 -> F11
   | 12 -> F12
   | 13 -> F13
   | 14 -> F14
   | 15 -> F15
   | 16 -> F16
   | 17 -> F17
   | 18 -> F18
   | 19 -> F19
   | 20 -> F20
   | 21 -> F21
   | 22 -> F22
   | 23 -> F23
   | 24 -> F24
   | 25 -> F25
   | 26 -> F26
   | 27 -> F27
   | 28 -> F28
   | 29 -> F29
   | 30 -> F30
   | 31 -> F31
   | _ -> Util.crash ("freg_of_int: " ^ string_of_int n)

let string_of_freg fr =
   "f" ^ string_of_int (int_of_freg fr)

let int_of_fcreg fcr =
   match fcr with
   | FC_FIR -> 0
   | FC_FCSR -> 31
   | FC_FCCR -> 25
   | FC_FEXR -> 26
   | FC_FENR -> 28

let fcreg_of_int n =
   match n with
   | 0 -> FC_FIR
   | 31 -> FC_FCSR
   | 25 -> FC_FCCR
   | 26 -> FC_FEXR
   | 28 -> FC_FENR
   | _ -> Util.crash ("fcreg_of_int: " ^ string_of_int n)

let string_of_fcreg fcr =
   match fcr with
   | FC_FIR -> "fir"
   | FC_FCSR -> "fcsr"
   | FC_FCCR -> "fccr"
   | FC_FEXR -> "fexr"
   | FC_FENR -> "fenr"

let int_of_cacheentryid ce =
   match ce with
   | CACHE_GREEN -> 0
   | CACHE_BLUE -> 1

let string_of_cacheentryid ce =
   match ce with
   | CACHE_GREEN -> "green"
   | CACHE_BLUE -> "blue"

let int_of_cache c =
   match c with
   | ICACHE -> 0
   | DCACHE -> 1
   | L2CACHE -> 3 (* yes, L2 is 3 and L3 is 2... go figure *)
   | L3CACHE -> 2

let cache_of_int n =
   match n with
   | 0 -> ICACHE
   | 1 -> DCACHE
   | 2 -> L3CACHE
   | 3 -> L2CACHE
   | _ -> Util.crash ("cache_of_int: " ^ string_of_int n)

let string_of_cache c =
   match c with
   | ICACHE -> "I"
   | DCACHE -> "D"
   | L2CACHE -> "L2"
   | L3CACHE -> "L3"

let int_of_cacheentry (c, ce) =
   int_of_cache c * 2 + int_of_cacheentryid ce

let string_of_cacheentry (c, ce) =
   string_of_cache c ^ "_" ^ string_of_cacheentryid ce

let int_of_golem g =
   match g with
   | G_VADDR c -> 2 * int_of_cache c + 0
   | G_INDEX c -> 2 * int_of_cache c + 1

let string_of_golem g =
   match g with
   | G_VADDR c -> string_of_cache c ^ "_vaddr"
   | G_INDEX c -> string_of_cache c ^ "_index"
