(*
 * Copyright (c) 2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

(*
 * Evaluate a logic expression to a constant, if possible; otherwise None.
 *)

module M = Mintops
open Mips
module MT = Mipstools

module PpsymMap = Ppoint.SymMap


type valuesdata =
   | ALLVALS of expr list
   | SOMEVALS of expr list
   | NOVALS

let rec intvals ty lo hi =
   if lo < hi then INTVALUE (lo, ty) :: intvals ty (lo + 1) hi
   else []

let getallvalues ty =
   match ty with
   | BOOL -> [TRUE; FALSE]
   | UINT5 -> intvals UINT5 0 31
   | SINT8 -> intvals SINT8 (-128) 127
   | UINT8 -> intvals UINT8 0 255
   | REGISTER -> List.map (fun r -> REGVALUE r) (MT.all_regs_raw ())
   | CREGISTER -> List.map (fun r -> CREGVALUE r) (MT.all_cregs_raw ())
   | FREGISTER -> List.map (fun r -> FREGVALUE r) (MT.all_fregs_raw ())
   | FCREGISTER -> List.map (fun r -> FCREGVALUE r) (MT.all_fcregs_raw ())
   | HIREGISTER -> [HIREGVALUE]
   | LOREGISTER -> [LOREGVALUE]
   | CACHESTATE -> List.map (fun s -> CACHESTATEVALUE s) (MT.all_cachestates_raw ())
   | GOLEM -> List.map (fun g -> GOLEMVALUE g) (MT.all_golems_raw ())
   | WHICHCACHE -> List.map (fun c -> WHICHCACHEVALUE c) (MT.all_caches_raw ())
   | WHICHCACHEENTRYID -> List.map (fun c -> WHICHCACHEENTRYIDVALUE c) (MT.all_cacheentryids_raw ())
   | WHICHCACHEENTRY ->
        let mk (c, ce) =
           WHICHCACHEENTRYVALUE (c, ce)
        in
        List.map mk (MT.all_cacheentries_raw ())
   | PROGRAMPOINT -> [] (* XXX *)
   | SINT16
   | UINT16
   | SINT32
   | UINT32
   | INSN -> Util.crash "lconsteval: getallvalues: too many"
   | MAP _ -> Util.crash "lconsteval: getallvalues: map"

let getsomevalues ty =
   match ty with
   | BOOL
   | CACHESTATE
   | UINT5
   | SINT8
   | UINT8
   | REGISTER
   | CREGISTER
   | FREGISTER
   | FCREGISTER 
   | HIREGISTER 
   | LOREGISTER 
   | GOLEM
   | WHICHCACHE
   | WHICHCACHEENTRYID
   | WHICHCACHEENTRY
   | PROGRAMPOINT -> Util.crash "lconsteval: getsomevalues: not enough"
   | SINT16
   | UINT16
   | SINT32
   | UINT32 -> intvals ty (-2) 5
   | INSN -> []
   | MAP _ -> Util.crash "lconsteval: getsomevalues: map"

(**************************************************************)
(* search *)

type searchexpr =
   | SEARCHBASE of int * bool (* expr k produces result b -> useful *)
   | SEARCHAND of searchexpr * searchexpr
   | SEARCHOR of searchexpr * searchexpr
   | SEARCHNOT of searchexpr

type searchresult = searchexpr * searchexpr  (* for true/false result resp. *)

type ctx = {
   nextid: int ref;
   exprs: expr Types.IntMap.t ref;
   vars: searchresult list VarMap.t;
}

let newctx () = {
   nextid = ref 0;
   exprs = ref Types.IntMap.empty;
   vars = VarMap.empty;
}

let ctx_addexpr ctx e =
   let id = !(ctx.nextid) in
   ctx.nextid := id + 1;
   ctx.exprs := Types.IntMap.add id e !(ctx.exprs);
   id

let ctx_addvar ctx x results =
   let vars = VarMap.add x results ctx.vars in
   { ctx with vars; }

let feasible_to_exhaust ty =
   match ty with
   | BOOL
   | CACHESTATE
   | UINT5
   | SINT8
   | UINT8 -> true
   | SINT16
   | UINT16
   | SINT32
   | UINT32 -> false
   | REGISTER
   | CREGISTER
   | FREGISTER
   | FCREGISTER
   | HIREGISTER
   | LOREGISTER
   | GOLEM
   | WHICHCACHE
   | WHICHCACHEENTRYID
   | WHICHCACHEENTRY -> true
   | PROGRAMPOINT -> false (* XXX: currently we don't get the list here *)
   | INSN -> false (* too many *)
   | MAP _ -> false

let qand ((r1t, r1f), (r2t, r2f)) =
   let t = SEARCHAND (r1t, r2t) in
   let f = SEARCHOR (r1f, r2f) in
   (t, f)

let qor ((r1t, r1f), (r2t, r2f)) =
   let t = SEARCHOR (r1t, r2t) in
   let f = SEARCHAND (r1f, r2f) in
   (t, f)

let qxor ((r1t, r1f), (r2t, r2f)) =
   let t = SEARCHOR (SEARCHAND (r1t, r2f), SEARCHAND (r1f, r2t)) in
   let f = SEARCHOR (SEARCHAND (r1t, r2t), SEARCHAND (r1f, r2f)) in
   (t, f)

let qnot (rt, rf) =
   (rf, rt)

(*
 * Search for a forall or exists we might be able to shortcut.
 * Returns the subexpression to experiment on and the values the
 * overall expression will return if we can shortcut it.
 *
 * Note that because the logic expressions don't have if-then-else,
 * there can be no boolean expressions buried under expressions of
 * other types, so we can stop looking when we find a non-boolean
 * type.
 *
 * XXX: this is no longer true; need to rework the logic...
 * (for now at least though the only way to get an IF is from the
 * cache reference logic in speccomp, which by design doesn't produce
 * anything tractable here)
 *)
let rec search ctx e =
   match e with
   | TRUE
   | FALSE
   | CACHESTATEVALUE _
   | INTVALUE _
   | REGVALUE _
   | CREGVALUE _
   | FREGVALUE _
   | FCREGVALUE _
   | HIREGVALUE
   | LOREGVALUE
   | GOLEMVALUE _
   | WHICHCACHEVALUE _
   | WHICHCACHEENTRYIDVALUE _
   | WHICHCACHEENTRYVALUE _
   | PPVALUE _
   | INSNVALUE _
   | READSTATE _ -> []
   | READGOLEM _ -> []
(*
   | READALLSTATE _ -> []
   | UPDATEMAP (_m, _k, _v) -> [] (* XXX? *)
*)
(*
   | READMEM _ -> []
*)
   | READVAR (x, BOOL) -> begin
        try
           VarMap.find x ctx.vars
        with Not_found -> []
     end
   | READVAR (_x, _ty) -> []
   | GOLEMIZE (_g, _e1) -> []
   | FORALL _ ->
        let id = ctx_addexpr ctx e in
        [(SEARCHBASE (id, false), SEARCHBASE (id, true))]
   | EXISTS _ ->
        let id = ctx_addexpr ctx e in
        [(SEARCHBASE (id, false), SEARCHBASE (id, true))]
   | LET (x, BOOL, e1, e2) ->
        let results = search ctx e1 in
        search (ctx_addvar ctx x results) e2
   | LET (_x, _ty, _e1, e2) -> search ctx e2
   | IF (_c, _t, _f) -> [] (* XXX implement me *)
   | IMPLIES _ -> Util.crash "lconsteval: IMPLIES not removed by lbaseopt"
   | LOGNOT e1 ->
        let r1 = search ctx e1 in
        List.map qnot r1
   | LOGAND (e1, e2) ->
        let r1 = search ctx e1 in
        let r2 = search ctx e2 in
        List.map qand (Util.cartesian r1 r2)
   | LOGOR (e1, e2) ->
        let r1 = search ctx e1 in
        let r2 = search ctx e2 in
        List.map qor (Util.cartesian r1 r2)
   | LOGXOR (e1, e2) ->
        let r1 = search ctx e1 in
        let r2 = search ctx e2 in
        List.map qxor (Util.cartesian r1 r2)
   | LOGEQ (e1, e2) ->
        let r1 = search ctx e1 in
        let r2 = search ctx e2 in
        List.map qnot (List.map qxor (Util.cartesian r1 r2))
   | MINTUOP _ -> []
   | MINTBOP _ -> []

let searchcheck info result =
   let rec checkse se =
      match se with
      | SEARCHBASE (id, tf) ->
           let (tfeasible, ffeasible) = Types.IntMap.find id info in
           if (tf && tfeasible) || ((not tf) && ffeasible) then Some se
           else None
      | SEARCHAND (se1, se2) -> begin
           match checkse se1, checkse se2 with
           | Some se1', Some se2' -> Some (SEARCHAND (se1', se2'))
           | _ -> None
        end
      | SEARCHOR (se1, se2) -> begin
           match checkse se1, checkse se2 with
           | Some se1', Some se2' -> Some (SEARCHOR (se1', se2'))
           | Some se1', None -> Some se1'
           | None, Some se2' -> Some se2'
           | None, None -> None
        end
      | SEARCHNOT se1 -> begin
           match checkse se1 with
           | Some se1' -> Some (SEARCHNOT se1')
           | None -> None
        end
   in
   let (fortrue, forfalse) = result in
   let conv se tf =
      match checkse se with
      | Some se' -> [(se', tf)]
      | None -> []
   in
   conv fortrue true @ conv forfalse false

let searchprune map results =
   let mkinfo e =
      match e with
      | FORALL (_, ty, _) ->
           (feasible_to_exhaust ty, true)
      | EXISTS (_, ty, _) ->
           (true, feasible_to_exhaust ty)
      | _ -> Util.crash "lconsteval: searchprune: mkinfo: garbage"
   in
   let info = Types.IntMap.map mkinfo map in
   List.concat (List.map (searchcheck info) results)

let dosearch e =
   let ctx = newctx () in
   let results = search ctx e in
   let results = searchprune !(ctx.exprs) results in
   (!(ctx.exprs), results)


(**************************************************************)
(* rescue *)

let weigh'type ty =
   match ty with
   | BOOL -> 2
   | CACHESTATE -> 4
   | UINT5 -> 32
   | SINT8 -> 256
   | UINT8 -> 256
   | SINT16 -> 800
   | UINT16 -> 800
   | SINT32 -> 1600
   | UINT32 -> 1600
   | REGISTER -> 32
   | CREGISTER -> 32
   | FREGISTER -> 32
   | FCREGISTER -> 32
   | HIREGISTER -> 1
   | LOREGISTER -> 1
   | GOLEM -> 2
   | WHICHCACHE -> 4
   | WHICHCACHEENTRYID -> 2
   | WHICHCACHEENTRY -> 8
   | PROGRAMPOINT -> 10
   | INSN -> 10000
   | MAP _ -> 10000

let alltrue xs = List.for_all (fun x -> x = TRUE) xs
let allfalse xs = List.for_all (fun x -> x = FALSE) xs
let anytrue xs = List.exists (fun x -> x = TRUE) xs
let anyfalse xs = List.exists (fun x -> x = FALSE) xs

let tryvalue x e v =
   let env = VarMap.add x v VarMap.empty in
   Lbaseopt.withenv env e

let testexpr e tf =
   match e with
   | FORALL (x, ty, e1) ->
        let values =
           if feasible_to_exhaust ty then getallvalues ty
           else if tf = false then getsomevalues ty
           else []
        in
        let results = List.map (tryvalue x e1) values in
        if feasible_to_exhaust ty && alltrue results then Some true
        else if anyfalse results then Some false
        else None
   | EXISTS (x, ty, e1) ->
        let values =
           if feasible_to_exhaust ty then getallvalues ty
           else if tf = true then getsomevalues ty
           else []
        in
        let results = List.map (tryvalue x e1) values in
        if feasible_to_exhaust ty && allfalse results then Some false
        else if anytrue results then Some true
        else None
   | _ -> Util.crash "lconsteval: testexpr: garbage"

let rescue exprs results =
   let weigh'expr e =
      match e with
      | FORALL (_, ty, _) -> weigh'type ty
      | EXISTS (_, ty, _) -> weigh'type ty
      | _ -> Util.crash "lconsteval: getweights: weigh'expr: garbage"
   in
   let exprweights = ref (Types.IntMap.map weigh'expr exprs) in

   let rec weigh'result se =
      match se with
      | SEARCHBASE (id, _res) -> Types.IntMap.find id !exprweights
      | SEARCHAND (se1, se2) -> weigh'result se1 + weigh'result se2
      | SEARCHOR (se1, se2) -> min (weigh'result se1) (weigh'result se2)
      | SEARCHNOT se1 -> weigh'result se1
   in
   let weigh'results rs =
      List.map (fun (se, tf) -> (weigh'result se, se, tf)) rs
   in
   let sort'results rs =
      List.sort (fun (w1, _, _) (w2, _, _) -> compare w1 w2) rs
   in
   let unweigh'results rs =
      List.map (fun (_w, se, tf) -> (se, tf)) rs
   in

   let answers = ref Types.IntMap.empty in

   let rec check_se se =
      match se with
      | SEARCHBASE (id, tfneeded) -> begin
           let answer =
              try
                 (* already did this one *)
                 Some (Types.IntMap.find id !answers)
              with Not_found ->
                 match testexpr (Types.IntMap.find id exprs) tfneeded with
                 | None -> None
                 | Some tf ->
                      answers := Types.IntMap.add id tf !answers;
                      exprweights := Types.IntMap.add id 0 !exprweights;
                      Some tf
           in
           match answer with
           | Some tf when tf = tfneeded -> true
           | _ -> false
        end
      | SEARCHAND (se1, se2) -> check_se se1 && check_se se2
      | SEARCHOR (se1, se2) -> check_se se1 || check_se se2
      | SEARCHNOT se1 -> not (check_se se1)
   in
   let check_result (_w, se, tfresult) =
      if check_se se then Some tfresult else None
   in

   let rec process rs =
      match sort'results (weigh'results rs) with
      | [] -> None
      | r :: rs ->
           match check_result r with
           | Some result -> Some result
           | None -> process (unweigh'results rs)
   in
   process results


(**************************************************************)
(* top level *)

let common e =
   (* Use baseopt to do all available constant folding *)
   let e' = Lbaseopt.go e in e'

let go_bool e =
   match common e with
   | TRUE -> Some true
   | FALSE -> Some false
   | _ ->
        let (searchexprs, searchresults) = dosearch e in
        rescue searchexprs searchresults

let go_int e =
   match common e with
   | INTVALUE (k, _ty) -> Some k
   | _ -> None

