(*
 * Copyright (c) 2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

(*
 * Basic optimizations on logic expressions.
 *
 * Note: this should be run only after typechecking; ill-typed
 * expressions might be mishandled, and in particular might turn into
 * well-typed expressions.
 *)

module M = Mintops
open Mips

type optresult =
   | Done of expr
   | Again of expr

let ispow2 k =
   k land (k - 1) <> 0

let rec ilog2 k =
   match k with
   | 0 -> Util.crash "lbaseopt: ilog2 0"
   | 1 -> 0
   | _ -> 1 + ilog2 (k / 2)

let isconst e =
   match e with
     (* constants *)
   | TRUE
   | FALSE -> true
     (* constructors *)
   | INTVALUE _
   | REGVALUE _
   | CREGVALUE _
   | FREGVALUE _
   | FCREGVALUE _
   | PPVALUE _ -> true
   | INSNVALUE _ ->
        (* we could/should check that it's not got any symbols in it. but... *)
        false
   | _ -> false

let rec eq e1_0 e2_0 =
   let boolval b = if b then Some TRUE else Some FALSE in
   match e1_0, e2_0 with
   | INTVALUE (k1, _ty1), INTVALUE (k2, _ty2) -> boolval (k1 = k2)
   | REGVALUE r1, REGVALUE r2 -> boolval (r1 = r2)
   | CREGVALUE r1, CREGVALUE r2 -> boolval (r1 = r2)
   | FREGVALUE r1, FREGVALUE r2 -> boolval (r1 = r2)
   | FCREGVALUE r1, FCREGVALUE r2 -> boolval (r1 = r2)
   | PPVALUE pp1, PPVALUE pp2 -> boolval (pp1 = pp2)
   | INSNVALUE i1, INSNVALUE i2 -> boolval (i1 = i2)
   | READSTATE (se1, pp1, zs1), READSTATE (se2, pp2, zs2)
        when se1 = se2 && pp1 = pp2 && zs1 = zs2 -> Some TRUE
(* XXX notyet
   | READMEM (ptr1, off1, pp1), READMEM (ptr2, off2, pp2)
        when ptr1 = ptr2 && eq off1 off2 = Some TRUE && pp1 = pp2 -> Some TRUE
*)
   | READVAR (x1, _ty1), READVAR (x2, _ty2) when x1 = x2 -> Some TRUE
   | FORALL (_x1, _ty1, e1), FORALL (_x2, _ty2, e2)
        (* XXX: should match READVAR x1 vs. READVAR x2 and doesn't *)
        when eq e1 e2 = Some TRUE -> Some TRUE
   | EXISTS (_x1, _ty1, e1), FORALL (_x2, _ty2, e2)
        (* XXX: should match READVAR x1 vs. READVAR x2 and doesn't *)
        when eq e1 e2 = Some TRUE -> Some TRUE
   | LET (_x1, _ty1, e1a, e1b), LET (_x2, _ty2, e2a, e2b)
        (* XXX: should match READVAR x1 vs. READVAR x2 and doesn't *)
        when eq e1a e2a = Some TRUE && eq e1b e2b = Some TRUE -> Some TRUE
   | IMPLIES (e1a, e1b), IMPLIES (e2a, e2b)
        when eq e1a e2a = Some TRUE && eq e1b e2b = Some TRUE -> Some TRUE
   | LOGNOT e1, LOGNOT e2
        when eq e1 e2 = Some TRUE -> Some TRUE
   | LOGAND (e1a, e1b), LOGAND (e2a, e2b)
        when eq e1a e2a = Some TRUE && eq e1b e2b = Some TRUE -> Some TRUE
   | LOGOR (e1a, e1b), LOGOR (e2a, e2b)
        when eq e1a e2a = Some TRUE && eq e1b e2b = Some TRUE -> Some TRUE
   | LOGXOR (e1a, e1b), LOGXOR (e2a, e2b)
        when eq e1a e2a = Some TRUE && eq e1b e2b = Some TRUE -> Some TRUE
   | LOGEQ (e1a, e1b), LOGEQ (e2a, e2b)
        when eq e1a e2a = Some TRUE && eq e1b e2b = Some TRUE -> Some TRUE
   | MINTUOP (op1, e1), MINTUOP (op2, e2)
        when op1 = op2 && eq e1 e2 = Some TRUE -> Some TRUE
   | MINTBOP (op1, e1a, e1b), MINTBOP (op2, e2a, e2b)
        when op1 = op2 &&
             eq e1a e2a = Some TRUE && eq e1b e2b = Some TRUE -> Some TRUE
   | _, _ -> None

let baseopt'expr'here constenv e = 
   let boolval b = if b then Done TRUE else Done FALSE in
   match e with
   | READVAR (x, _ty) -> begin
        try
           Done (VarMap.find x constenv)
        with Not_found -> Done e
     end
   | FORALL (_x, _ty, TRUE) -> Done TRUE
   | FORALL (_x, _ty, FALSE) -> Done FALSE (* assume ty is not an empty type *)
   | FORALL (x, BOOL, e1) ->
        let e1t = LET (x, BOOL, TRUE, e1) in
        let e1f = LET (x, BOOL, FALSE, e1) in
        Done (LOGAND (e1t, e1f))
   | EXISTS (_x, _ty, TRUE) -> Done TRUE (* assume ty is not an empty type *)
   | EXISTS (_x, _ty, FALSE) -> Done FALSE
   | EXISTS (x, BOOL, e1) ->
        let e1t = LET (x, BOOL, TRUE, e1) in
        let e1f = LET (x, BOOL, FALSE, e1) in
        Done (LOGOR (e1t, e1f))
   | IMPLIES (e1, e2) -> Done (LOGOR (LOGNOT e1, e2))
   | LOGNOT TRUE -> Done FALSE
   | LOGNOT FALSE -> Done TRUE
   | LOGNOT (LOGNOT e1) -> Done e1
   | LOGAND (FALSE, _) -> Done FALSE
   | LOGAND (_, FALSE) -> Done FALSE
   | LOGAND (TRUE, e2) -> Done e2
   | LOGAND (e1, TRUE) -> Done e1
   | LOGOR (TRUE, _) -> Done TRUE
   | LOGOR (_, TRUE) -> Done TRUE
   | LOGOR (FALSE, e2) -> Done e2
   | LOGOR (e1, FALSE) -> Done e1
   | LOGXOR (e1, FALSE) -> Done e1
   | LOGXOR (FALSE, e2) -> Done e2
   | LOGXOR (TRUE, TRUE) -> Done FALSE
   | LOGXOR (e1, TRUE) -> Again (LOGNOT e1)
   | LOGXOR (TRUE, e2) -> Again (LOGNOT e2)
   | LOGEQ (e1, TRUE) -> Done e1
   | LOGEQ (TRUE, e2) -> Done e2
   | LOGEQ (FALSE, FALSE) -> Done TRUE
   | LOGEQ (e1, FALSE) -> Again (LOGNOT e1)
   | LOGEQ (FALSE, e2) -> Again (LOGNOT e2)
   | LOGEQ (e1, e2) when eq e1 e2 = Some TRUE -> Done TRUE
   | LOGEQ (e1, e2) when eq e1 e2 = Some FALSE -> Done FALSE
     (* XXX what's the width of this lnot? *)
   | MINTUOP (M.BITNOT, INTVALUE (k, ty)) -> Done (INTVALUE (lnot k, ty))
   | MINTUOP (M.SBWIDEN, INTVALUE (k, _ty)) -> Done (INTVALUE (k, SINT32))
   | MINTUOP (M.SHWIDEN, INTVALUE (k, _ty)) -> Done (INTVALUE (k, SINT32))
   | MINTUOP (M.UBWIDEN, INTVALUE (k, _ty)) -> Done (INTVALUE (k, UINT32))
   | MINTUOP (M.UHWIDEN, INTVALUE (k, _ty)) -> Done (INTVALUE (k, UINT32))
   | MINTUOP (M.U5WIDEN, INTVALUE (k, _ty)) -> Done (INTVALUE (k, UINT32))
   | MINTUOP (M.TRUNCU5, INTVALUE (k, _ty)) ->
        Done (INTVALUE (k land 31, UINT5))
   | MINTUOP (M.TOSIGNED, INTVALUE (k, _ty)) -> Done (INTVALUE (k, SINT32))
   | MINTUOP (M.TOUNSIGNED, INTVALUE (k, _ty)) -> Done (INTVALUE (k, UINT32))
   | MINTUOP (M.U5WIDEN, MINTUOP (M.TRUNCU5, e1)) ->
        Again (MINTBOP (M.BITAND, e1, INTVALUE (31, UINT32)))
   | MINTUOP (M.TRUNCU5, MINTUOP (M.U5WIDEN, e1)) -> Done e1
   | MINTUOP (M.TOSIGNED, MINTUOP (M.TOUNSIGNED, e1)) -> Done e1
   | MINTUOP (M.TOUNSIGNED, MINTUOP (M.TOSIGNED, e1)) -> Done e1
   | MINTUOP (M.BITNOT, MINTUOP (M.BITNOT, e1)) -> Done e1
   | MINTBOP (M.EQ, e1, e2) when eq e1 e2 = Some TRUE -> Done TRUE
   | MINTBOP (M.EQ, e1, e2) when eq e1 e2 = Some FALSE -> Done FALSE
     (* unsigned less than 0 is always false *)
   | MINTBOP (M.ULT, _, INTVALUE (0, _ty)) -> Done FALSE
     (* adding 0 *)
   | MINTBOP (M.ADD, e1, INTVALUE (0, _ty)) -> Done e1
   | MINTBOP (M.ADD, INTVALUE (0, _ty), e2) -> Done e2
     (* subtracting 0 *)
   | MINTBOP (M.SUB, e1, INTVALUE (0, _ty)) -> Done e1
     (* multiplying by 0 *)
   | MINTBOP (M.SMUL_HI, _, INTVALUE (0, _ty)) -> Done (INTVALUE (0, SINT32))
   | MINTBOP (M.UMUL_HI, _, INTVALUE (0, _ty)) -> Done (INTVALUE (0, UINT32))
   | MINTBOP (M.MUL_LO, _, INTVALUE (0, ty)) -> Done (INTVALUE (0, ty))
   | MINTBOP (M.SMUL_HI, INTVALUE (0, _ty), _) -> Done (INTVALUE (0, SINT32))
   | MINTBOP (M.UMUL_HI, INTVALUE (0, _ty), _) -> Done (INTVALUE (0, UINT32))
   | MINTBOP (M.MUL_LO, INTVALUE (0, ty), _) -> Done (INTVALUE (0, ty))
     (* multiplying by 1 *)
   | MINTBOP (M.SMUL_HI, _, INTVALUE (1, _ty)) -> Done (INTVALUE (0, SINT32))
   | MINTBOP (M.UMUL_HI, _, INTVALUE (1, _ty)) -> Done (INTVALUE (0, UINT32))
   | MINTBOP (M.SMUL_HI, INTVALUE (1, _ty), _) -> Done (INTVALUE (0, SINT32))
   | MINTBOP (M.UMUL_HI, INTVALUE (1, _ty), _) -> Done (INTVALUE (0, UINT32))
   | MINTBOP (M.MUL_LO, e1, INTVALUE (1, _ty)) -> Done e1
   | MINTBOP (M.MUL_LO, INTVALUE (1, _ty), e2) -> Done e2
     (* multiplying by -1 *)
   | MINTBOP (M.MUL_LO, e1, INTVALUE (-1, ty)) ->
        Done (MINTBOP (M.SUB, INTVALUE (0, ty), e1))
     (* multiplying by power of 2 *)
   | MINTBOP (M.SMUL_HI, e1, INTVALUE (k, _ty)) when k > 0 && ispow2 k ->
        Again (MINTBOP (M.SSHR, e1, INTVALUE (32 - (ilog2 k), UINT5)))
   | MINTBOP (M.UMUL_HI, e1, INTVALUE (k, _ty)) when k > 0 && ispow2 k ->
        Again (MINTBOP (M.USHR, e1, INTVALUE (32 - (ilog2 k), UINT5)))
   | MINTBOP (M.MUL_LO, e1, INTVALUE (k, _ty)) when k > 0 && ispow2 k ->
        Again (MINTBOP (M.SHL, e1, INTVALUE (ilog2 k, UINT5)))
   | MINTBOP (M.SMUL_HI, INTVALUE (k, _ty), e2) when k > 0 && ispow2 k ->
        Again (MINTBOP (M.SSHR, e2, INTVALUE (32 - (ilog2 k), UINT5)))
   | MINTBOP (M.UMUL_HI, INTVALUE (k, _ty), e2) when k > 0 && ispow2 k ->
        Again (MINTBOP (M.USHR, e2, INTVALUE (32 - (ilog2 k), UINT5)))
   | MINTBOP (M.MUL_LO, INTVALUE (k, _ty), e2) when k > 0 && ispow2 k ->
        Again (MINTBOP (M.SHL, e2, INTVALUE (ilog2 k, UINT5)))
     (* dividing by 0... *)
   | MINTBOP (M.SDIV as op, _, INTVALUE (0, ty))
   | MINTBOP (M.UDIV as op, _, INTVALUE (0, ty))
   | MINTBOP (M.SMOD as op, _, INTVALUE (0, ty))
   | MINTBOP (M.UMOD as op, _, INTVALUE (0, ty)) ->
        Done (MINTBOP (op, INTVALUE (1, ty), INTVALUE (0, ty)))
     (* dividing by 1 *)
   | MINTBOP (M.SDIV, e1, INTVALUE (1, _ty))
   | MINTBOP (M.UDIV, e1, INTVALUE (1, _ty)) -> Done e1
     (* mod by 1 *)
   | MINTBOP (M.SMOD, _, INTVALUE (1, ty))
   | MINTBOP (M.UMOD, _, INTVALUE (1, ty)) -> Done (INTVALUE (0, ty))
     (* dividing by -1 *)
   | MINTBOP (M.SDIV, e1, INTVALUE (-1, ty)) ->
        Done (MINTBOP (M.SUB, INTVALUE (0, ty), e1))
   | MINTBOP (M.UDIV, e1, INTVALUE (-1, ty)) ->
        let one = INTVALUE (1, ty) in
        let ult = MINTBOP (M.ULT, e1, INTVALUE (-1, ty)) in
        Again (MINTBOP (M.BITXOR, ult, one))
     (* divide by power of two *)
   | MINTBOP (M.SDIV, e1, INTVALUE (k, ty)) when k > 0 && ispow2 k ->
        Again (MINTBOP (M.USHR, e1, INTVALUE (ilog2 k, ty)))
   | MINTBOP (M.UDIV, e1, INTVALUE (k, ty)) when k > 0 && ispow2 k ->
        Again (MINTBOP (M.SSHR, e1, INTVALUE (ilog2 k, ty)))
     (* mod by power of two *)
   | MINTBOP (M.UMOD, e1, INTVALUE (k, ty)) when k > 0 && ispow2 k ->
        Again (MINTBOP (M.BITAND, e1, INTVALUE (k - 1, ty)))
     (* bitwise ops with 0 *)
   | MINTBOP (M.BITAND, _, INTVALUE (0, ty)) -> Done (INTVALUE (0, ty))
   | MINTBOP (M.BITAND, INTVALUE (0, ty), _) -> Done (INTVALUE (0, ty))
   | MINTBOP (M.BITOR, e1, INTVALUE (0, _ty)) -> Done e1
   | MINTBOP (M.BITOR, INTVALUE (0, _ty), e2) -> Done e2
   | MINTBOP (M.BITXOR, e1, INTVALUE (0, _ty)) -> Done e1
   | MINTBOP (M.BITXOR, INTVALUE (0, _ty), e2) -> Done e2
   | MINTBOP (M.SHL, e1, INTVALUE (0, _ty)) -> Done e1
   | MINTBOP (M.USHR, e1, INTVALUE (0, _ty)) -> Done e1
   | MINTBOP (M.SSHR, e1, INTVALUE (0, _ty)) -> Done e1
     (* bitwise ops with -1 *)
   | MINTBOP (M.BITAND, e1, INTVALUE (-1, _ty)) -> Done e1
   | MINTBOP (M.BITAND, INTVALUE (-1, _ty), e2) -> Done e2
   | MINTBOP (M.BITOR, _, INTVALUE (-1, ty)) -> Done (INTVALUE (-1, ty))
   | MINTBOP (M.BITOR, INTVALUE (-1, ty), _) -> Done (INTVALUE (-1, ty))
   | MINTBOP (M.BITXOR, e1, INTVALUE (-1, _ty)) ->
        Again (MINTUOP (M.BITNOT, e1))
   | MINTBOP (M.BITXOR, INTVALUE (-1, _ty), e2) -> 
        Again (MINTUOP (M.BITNOT, e2))
     (* constant folding (M.EQ already covered) *)
   | MINTBOP (M.ULT, INTVALUE (k1, _ty1), INTVALUE (k2, _ty2)) ->
        (* bloody ocaml has no unsigned *)
        if (k1 < 0 && k2 >= 0) then Done FALSE
        else if (k1 >= 0 && k2 < 0) then Done TRUE
        else boolval (k1 < k2)
   | MINTBOP (M.SLT, INTVALUE (k1, _ty1), INTVALUE (k2, _ty2)) ->
        boolval (k1 < k2)
   | MINTBOP (M.ADD, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) ->
        Done (INTVALUE (k1 + k2, ty1))
   | MINTBOP (M.SUB, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) ->
        Done (INTVALUE (k1 - k2, ty1))
   (* XXX multiply needs real ints *)
   | MINTBOP (M.SDIV, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) when k2 <> 0 ->
        Done (INTVALUE (k1 / k2, ty1))
   | MINTBOP (M.UDIV, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) when k2 <> 0 ->
        (* bloody ocaml has no unsigned *)
        let k =
           if (k1 < 0 && k2 < 0) then (-k1) / (-k2)
           else if (k1 < 0) then -((-k1) / k2)
           else if (k2 < 0) then -(k1 / (-k2))
           else k1 / k2
        in
        Done (INTVALUE (k, ty1))
   | MINTBOP (M.SMOD, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) when k2 <> 0 ->
        Done (INTVALUE (k1 mod k2, ty1))
   (* XXX unsigned mod requires real ints *)
   | MINTBOP (M.BITAND, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) ->
        (* note: bitwise ops are "logical" because ocaml *)
        Done (INTVALUE (k1 land k2, ty1))
   | MINTBOP (M.BITOR, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) ->
        Done (INTVALUE (k1 lor k2, ty1))
   | MINTBOP (M.BITXOR, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) ->
        Done (INTVALUE (k1 lxor k2, ty1))
   | MINTBOP (M.SHL, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) ->
        Done (INTVALUE (k1 lsl k2, ty1))
   | MINTBOP (M.USHR, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) ->
        Done (INTVALUE (k1 lsr k2, ty1))
   | MINTBOP (M.SSHR, INTVALUE (k1, ty1), INTVALUE (k2, _ty2)) ->
        Done (INTVALUE (k1 asr k2, ty1))
   | _ -> Done e

let rec baseopt'expr constenv e0 =
   (* try matching before recursing *)
   let rec prematch e =
      match baseopt'expr'here constenv e with
      | Done e' -> e'
      | Again e' -> prematch e'
   in
   let e = prematch e0 in

   (* now recurse *)
   let e = match e with
     (* constants *)
   | TRUE
   | FALSE
   | CACHESTATEVALUE _
     (* constructors *)
   | INTVALUE _
   | REGVALUE _
   | CREGVALUE _
   | FREGVALUE _
   | FCREGVALUE _
   | HIREGVALUE
   | LOREGVALUE
   | GOLEMVALUE _
   | WHICHCACHEVALUE _
   | WHICHCACHEENTRYIDVALUE _ -> e
   | WHICHCACHEENTRYVALUE _ -> e
   | PPVALUE _ -> e
   | INSNVALUE _ -> e

     (* access to processor state, as of a particular program point *)
   | READSTATE (e1, pp, zs) ->
        let e1' = baseopt'expr constenv e1 in
        READSTATE (e1', pp, zs)
   | READGOLEM g ->
        let g' = baseopt'expr constenv g in
        READGOLEM g'
(*
   | READALLSTATE _ -> e
   | UPDATEMAP (m, k, v) ->
        let m' = baseopt'expr constenv m in
        let k' = baseopt'expr constenv k in
        let v' = baseopt'expr constenv v in
        UPDATEMAP (m', k', v')
*)
(*
   | READMEM (ptr, off, pp) ->
        let off' = baseopt'expr constenv off in
        READMEM (ptr, off', pp)
*)

     (* access to logic variables *)
   | READVAR _ -> e

     (* access to logic functions *)
   | GOLEMIZE (g, e1) ->
        let g' = baseopt'expr constenv g in
        let e1' = baseopt'expr constenv e1 in
        GOLEMIZE (g', e1')

     (* logical operators *)
   | FORALL (x, ty, e1) ->
        let e1' = baseopt'expr constenv e1 in
        FORALL (x, ty, e1')
   | EXISTS (x, ty, e1) ->
        let e1' = baseopt'expr constenv e1 in
        EXISTS (x, ty, e1')
   | LET (x, ty, e1, e2) ->
        let e1' = baseopt'expr constenv e1 in
        if isconst e1' then
           baseopt'expr (VarMap.add x e1' constenv) e2
        else
           let e2' = baseopt'expr constenv e2 in
           LET (x, ty, e1', e2')
   | IF (c, t, f) ->
        let c' = baseopt'expr constenv c in
        let t' = baseopt'expr constenv t in
        let f' = baseopt'expr constenv f in
        if isconst c' then begin
           match c with
           | TRUE -> t'
           | FALSE -> f'
           | _ -> Util.crash "lbaseopt: if: constant boolean not boolean"
        end
        else IF (c', t', f')
   | IMPLIES _ -> (* removed above *) Util.crash "lbaseopt: IMPLIES"
   | LOGNOT e1 ->
        let e1' = baseopt'expr constenv e1 in
        LOGNOT e1'
   | LOGAND (e1, e2) ->
        let e1' = baseopt'expr constenv e1 in
        let e2' = baseopt'expr constenv e2 in
        LOGAND (e1', e2')
   | LOGOR (e1, e2) ->
        let e1' = baseopt'expr constenv e1 in
        let e2' = baseopt'expr constenv e2 in
        LOGOR (e1', e2')
   | LOGXOR (e1, e2) ->
        let e1' = baseopt'expr constenv e1 in
        let e2' = baseopt'expr constenv e2 in
        LOGXOR (e1', e2')
   | LOGEQ (e1, e2) ->
        let e1' = baseopt'expr constenv e1 in
        let e2' = baseopt'expr constenv e2 in
        LOGEQ (e1', e2')

     (* other operators *)
   | MINTUOP (op, e1) ->
        let e1' = baseopt'expr constenv e1 in
        MINTUOP (op, e1')
   | MINTBOP (op, e1, e2) ->
        let e1' = baseopt'expr constenv e1 in
        let e2' = baseopt'expr constenv e2 in
        MINTBOP (op, e1', e2')
   in
   (* now try matching again *)
   match baseopt'expr'here constenv e with
   | Done e' -> e'
   | Again e' -> (* recurse again *) baseopt'expr constenv e'

let withenv env e =
   baseopt'expr env e

let go e =
   withenv VarMap.empty e
