(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

(*
 * list out all the possible instructions
 *
 * Would be nice if there were an automated way to iterate over all
 * the constructors of concreteinsn.
 *)

open Symbolic
module MS = Mipsstate
module M = Mips
module E = Enabled

(*
 * Note: this does not strictly adhere to the machine-level usage of
 * the rd, rs, and rt instruction fields, because we don't know or
 * care about instruction encodings. We use "rd" as the destination
 * for basically everything and the real CPU is nowhere near so tidy.
 *
 * The coprocessor versions of the rd and rs fields are different only
 * for type safety.
 *
 * The important thing to take note of at this level is that in the
 * assembly language the move-to-coprocessor instructions move left to
 * right even though everything else, including move-from-coprocessor,
 * moves right to left.
 *) 

type operandset = {
   (* registers - rd field *)
   rd: MS.sireg;		(* destination register for most ops *)
   cd: MS.sicreg;		(* when it's a control register *)
   fd: MS.sifreg;		(* when it's an fpu register *)
   fcd: MS.sifcreg;		(* when it's an fpu control register *)
   (* registers - rs field *)
   rs: MS.sireg;		(* first source register *)
   cs: MS.sicreg;		(* when a control regster *)
   fs: MS.sifreg;		(* when an fpu register *)
   fcs: MS.sifcreg;		(* when an fpu control register *)
   (* registers - rt field *)
   rt: MS.sireg;		(* second source register *)
   (* other machine state handles *)
   whichcache: MS.sicache;
   (* immediates *)
   simm16: M.imm;
   uimm16: M.imm;
   uimm5: M.imm;
   (* labels *)   
   xcl: M.extcodelabel;
   xdl: M.extdatalabel;
   lcl: M.localcodelabel;
   ldl: M.localdatalabel;
   (* other *)
   mem: M.memory;
   (* extra symbols (note: must not hold values derived from machine state) *)
   byindexsym: int * string;
   cachegolemsym: int * string;
}

(*
 * Make a (symbolic) operand set to use to populate an instruction list.
 * Returns the list of names/types involved and the set record.
 *
 * Exclude disabled processor state from the binding list.
 *)
let mkoperandset insnnum =
   let fullnames = ref [] in
   let mkname' enabled fieldname ty =
      let fullname = (insnnum, fieldname) in
      if enabled then
         fullnames := (M.CONTROLVAR fullname, ty) :: !fullnames
      ;
      fullname
   in
   let mkname enabled fieldname ty =
      SYM (mkname' enabled fieldname ty)
   in
   (* registers *)
   let rd = (mkname true "rd" M.REGISTER) in
   let cd = (mkname !E.use_insn_c0 "cd" M.CREGISTER) in
   let fd = (mkname !E.use_insn_fpu "fd" M.FREGISTER) in
   let fcd = (mkname !E.use_insn_fpu "fcd" M.FCREGISTER) in
   let rs = (mkname true "rs" M.REGISTER) in
   let cs = (mkname !E.use_insn_c0 "cs" M.CREGISTER) in
   let fs = (mkname !E.use_insn_fpu "fs" M.FREGISTER) in
   let fcs = (mkname !E.use_insn_fpu "fcs" M.FCREGISTER) in
   let rt = (mkname true "rt" M.REGISTER) in
   (* other machine state *)
   let whichcache = mkname !E.use_cache "whichcache" M.WHICHCACHE in
   (* immediates *)
   let simm16 = mkname true "simm16" M.SINT16 in
   let uimm16 = mkname true "uimm16" M.UINT16 in
   let uimm5 = mkname true "uimm5" M.UINT5 in
   (* labels *)
   (* XXX xcl/xdl won't work as is *)
   let xcl = (mkname !E.use_extcodelabels "extcode" M.PROGRAMPOINT) in
   let xdl = (mkname !E.use_extdatalabels "extdata" M.UINT32) in
   let lcl = (mkname !E.use_localcodelabels "label" M.PROGRAMPOINT) in
   let ldl = (mkname !E.use_localdatalabels "localdata" M.UINT32) in
   (* extra symbols *)
   let byindexsym = mkname' !E.use_cache "byindex" M.BOOL in
   let cachegolemsym = mkname' !E.use_cache "cachegolem" M.GOLEM in
   let set = {
      rd; cd; fd; fcd;
      rs; cs; fs; fcs;
      rt;
      whichcache;
      simm16; uimm16; uimm5;
      xcl; xdl;
      lcl; ldl;
      mem = M.MEMORY (rt, simm16);
      byindexsym;
      cachegolemsym;
   } in

   (* Also make variables for the values taken from registers. *)
   (* XXX this is kinda ugly, should have a stronger framework *)
(*
   let _rdval = (mkname true "rdval" M.UINT32) in
   let _rsval = (mkname true "rsval" M.UINT32) in
   let _rtval = (mkname true "rtval" M.UINT32) in
   let _fcsval = (mkname !E.use_insn_fpu "fcsval" M.UINT32) in
   let _csval = (mkname !E.use_insn_c0 "csval" M.UINT32) in
   let _fdval = (mkname !E.use_insn_fpu "fdval" M.UINT32) in
   let _fsval = (mkname !E.use_insn_fpu "fsval" M.UINT32) in
*)
   (List.rev !fullnames, set)

let insns opset =
   let full = [
      (* arithmetic *)
      M.ADDU (opset.rd, opset.rs, opset.rt);
      M.ADDIU (opset.rd, opset.rs, opset.simm16);
      M.DIV (opset.rs, opset.rt);
      M.DIVU (opset.rs, opset.rt);
      M.LUI (opset.rd, opset.uimm16);
      M.MULT (opset.rs, opset.rt);
      M.MULTU (opset.rs, opset.rt);
      M.SUBU (opset.rd, opset.rs, opset.rt);
      M.LUI_extcode_hi (opset.rd, opset.xcl);
      M.ADDIU_extcode_lo (opset.rd, opset.rs, opset.xcl);
      M.LUI_extdata_hi (opset.rd, opset.xdl);
      M.ADDIU_extdata_lo (opset.rd, opset.rs, opset.xdl);
      M.LUI_localcode_hi (opset.rd, opset.lcl);
      M.ADDIU_localcode_lo (opset.rd, opset.rs, opset.lcl);
      (* comparisons *)
      M.SLT (opset.rd, opset.rs, opset.rt);
      M.SLTI (opset.rd, opset.rs, opset.simm16);
      M.SLTU (opset.rd, opset.rs, opset.rt);
      M.SLTIU (opset.rd, opset.rs, opset.simm16);
      (* bitwise logic *)
      M.AND (opset.rd, opset.rs, opset.rt);
      M.ANDI (opset.rd, opset.rs, opset.uimm16);
      M.NOR (opset.rd, opset.rs, opset.rt);
      M.OR (opset.rd, opset.rs, opset.rt);
      M.ORI (opset.rd, opset.rs, opset.uimm16);
      M.XOR (opset.rd, opset.rs, opset.rt);
      M.XORI (opset.rd, opset.rs, opset.uimm16);
      (* shifts *)
      M.SLL (opset.rd, opset.rs, opset.uimm5);
      M.SLLV (opset.rd, opset.rs, opset.rt);
      M.SRA (opset.rd, opset.rs, opset.uimm5);
      M.SRAV (opset.rd, opset.rs, opset.rt);
      M.SRL (opset.rd, opset.rs, opset.uimm5);
      M.SRLV (opset.rd, opset.rs, opset.rt);
      (* transfers *)
      M.B opset.lcl;
      M.BEQ (opset.rs, opset.rt, opset.lcl);
      M.BNE (opset.rs, opset.rt, opset.lcl);
      M.J opset.xcl;
      M.JAL opset.xcl;
      M.JALR opset.rs;
      M.JR opset.rs;
      (* moves *)
      M.CFC1 (opset.rd, opset.fcs);
      M.CTC1 (opset.rs, opset.fcd);
      M.MFC0 (opset.rd, opset.cs);
      M.MFC1 (opset.rd, opset.fs);
      M.MFHI opset.rd;
      M.MFLO opset.rd;
      M.MTC0 (opset.rs, opset.cd);
      M.MTC1 (opset.rs, opset.fd);
      M.MTHI opset.rs;
      M.MTLO opset.rs;
      M.MOVN (opset.rd, opset.rs, opset.rt);
      M.MOVZ (opset.rd, opset.rs, opset.rt);
      (* loads *)
      M.LB (opset.rd, opset.mem);
      M.LBU (opset.rd, opset.mem);
      M.LDC1 (opset.fd, opset.mem);
      M.LH (opset.rd, opset.mem);
      M.LHU (opset.rd, opset.mem);
      M.LW (opset.rd, opset.mem);
      M.LWC1 (opset.fd, opset.mem);
      (* stores *)
      M.SB (opset.rs, opset.mem);
      M.SDC1 (opset.fs, opset.mem);
      M.SH (opset.rs, opset.mem);
      M.SW (opset.rs, opset.mem);
      M.SWC1 (opset.fs, opset.mem);
      (* system control *)
      M.CACHE_INDEX_WBINV (opset.whichcache, opset.mem, opset.cachegolemsym);
      M.CACHE_VADDR_WB (opset.whichcache, opset.mem, opset.cachegolemsym);
      M.CACHE_VADDR_WBINV (opset.whichcache, opset.mem, opset.cachegolemsym);
      M.CACHE_VADDR_INV (opset.whichcache, opset.mem, opset.cachegolemsym);
      M.TLBP;
      M.TLBR;
      M.TLBWI;
      M.TLBWR;
   ]
   in
   List.filter Enabled.insn_enabled full

(* would be nice to not need to duplicate the list *)
let insncode insn =
   match insn with
   (* arithmetic *)
   | M.ADDU _ ->                     0
   | M.ADDIU _ ->                    1
   | M.DIV _ ->                      2
   | M.DIVU _ ->                     3
   | M.LUI _ ->                      4
   | M.MULT _ ->                     5
   | M.MULTU _ ->                    6
   | M.SUBU _ ->                     7
   | M.LUI_extcode_hi _ ->           8
   | M.ADDIU_extcode_lo _ ->         9
   | M.LUI_extdata_hi _ ->           10
   | M.ADDIU_extdata_lo _ ->         11
   | M.LUI_localcode_hi _ ->         12
   | M.ADDIU_localcode_lo _ ->       13
   (* comparisons *)
   | M.SLT _ ->                      14
   | M.SLTI _ ->                     15
   | M.SLTU _ ->                     16
   | M.SLTIU _ ->                    17
   (* bitwise logic *)
   | M.AND _ ->                      18
   | M.ANDI _ ->                     19
   | M.NOR _ ->                      20
   | M.OR _ ->                       21
   | M.ORI _ ->                      22
   | M.XOR _ ->                      23
   | M.XORI _ ->                     24
   (* shifts *)
   | M.SLL _ ->                      25
   | M.SLLV _ ->                     26
   | M.SRA _ ->                      27
   | M.SRAV _ ->                     28
   | M.SRL _ ->                      29
   | M.SRLV _ ->                     30
   (* transfers *)
   | M.B _ ->                        31
   | M.BEQ _ ->                      32
   | M.BNE _ ->                      33
   | M.J _ ->                        34
   | M.JAL _ ->                      35
   | M.JALR _ ->                     36
   | M.JR _ ->                       37
   (* moves *)
   | M.CFC1 _ ->                     38
   | M.CTC1 _ ->                     39
   | M.MFC0 _ ->                     40
   | M.MFC1 _ ->                     41
   | M.MFHI _ ->                     42
   | M.MFLO _ ->                     43
   | M.MTC0 _ ->                     44
   | M.MTC1 _ ->                     45
   | M.MTHI _ ->                     46
   | M.MTLO _ ->                     47
   | M.MOVN _ ->                     48
   | M.MOVZ _ ->                     49
   (* loads *)
   | M.LB _ ->                       50
   | M.LBU _ ->                      51
   | M.LDC1 _ ->                     52
   | M.LH _ ->                       53
   | M.LHU _ ->                      54
   | M.LW _ ->                       55
   | M.LWC1 _ ->                     56
   (* stores *)
   | M.SB _ ->                       57
   | M.SDC1 _ ->                     58
   | M.SH _ ->                       59
   | M.SW _ ->                       60
   | M.SWC1 _ ->                     61
   (* system control *)
   | M.CACHE_INDEX_WBINV _ ->        62
   | M.CACHE_VADDR_WB _ ->           63
   | M.CACHE_VADDR_WBINV _ ->        64
   | M.CACHE_VADDR_INV _ ->          65
   | M.TLBP ->                       66
   | M.TLBR ->                       67
   | M.TLBWI ->                      68
   | M.TLBWR ->                      69

(* this is mostly paste also *)
let decode opset code =
   match code with
   (* arithmetic *)
   | 0 -> M.ADDU (opset.rd, opset.rs, opset.rt)
   | 1 -> M.ADDIU (opset.rd, opset.rs, opset.simm16)
   | 2 -> M.DIV (opset.rs, opset.rt)
   | 3 -> M.DIVU (opset.rs, opset.rt)
   | 4 -> M.LUI (opset.rd, opset.uimm16)
   | 5 -> M.MULT (opset.rs, opset.rt)
   | 6 -> M.MULTU (opset.rs, opset.rt)
   | 7 -> M.SUBU (opset.rd, opset.rs, opset.rt)
   | 8 -> M.LUI_extcode_hi (opset.rd, opset.xcl)
   | 9 -> M.ADDIU_extcode_lo (opset.rd, opset.rs, opset.xcl)
   | 10 -> M.LUI_extdata_hi (opset.rd, opset.xdl)
   | 11 -> M.ADDIU_extdata_lo (opset.rd, opset.rs, opset.xdl)
   | 12 -> M.LUI_localcode_hi (opset.rd, opset.lcl)
   | 13 -> M.ADDIU_localcode_lo (opset.rd, opset.rs, opset.lcl)
   (* comparisons *)
   | 14 -> M.SLT (opset.rd, opset.rs, opset.rt)
   | 15 -> M.SLTI (opset.rd, opset.rs, opset.simm16)
   | 16 -> M.SLTU (opset.rd, opset.rs, opset.rt)
   | 17 -> M.SLTIU (opset.rd, opset.rs, opset.simm16)
   (* bitwise logic *)
   | 18 -> M.AND (opset.rd, opset.rs, opset.rt)
   | 19 -> M.ANDI (opset.rd, opset.rs, opset.uimm16)
   | 20 -> M.NOR (opset.rd, opset.rs, opset.rt)
   | 21 -> M.OR (opset.rd, opset.rs, opset.rt)
   | 22 -> M.ORI (opset.rd, opset.rs, opset.uimm16)
   | 23 -> M.XOR (opset.rd, opset.rs, opset.rt)
   | 24 -> M.XORI (opset.rd, opset.rs, opset.uimm16)
   (* shifts *)
   | 25 -> M.SLL (opset.rd, opset.rs, opset.uimm5)
   | 26 -> M.SLLV (opset.rd, opset.rs, opset.rt)
   | 27 -> M.SRA (opset.rd, opset.rs, opset.uimm5)
   | 28 -> M.SRAV (opset.rd, opset.rs, opset.rt)
   | 29 -> M.SRL (opset.rd, opset.rs, opset.uimm5)
   | 30 -> M.SRLV (opset.rd, opset.rs, opset.rt)
   (* transfers *)
   | 31 -> M.B opset.lcl
   | 32 -> M.BEQ (opset.rs, opset.rt, opset.lcl)
   | 33 -> M.BNE (opset.rs, opset.rt, opset.lcl)
   | 34 -> M.J opset.xcl
   | 35 -> M.JAL opset.xcl
   | 36 -> M.JALR opset.rs
   | 37 -> M.JR opset.rs
   (* moves *)
   | 38 -> M.CFC1 (opset.rd, opset.fcs)
   | 39 -> M.CTC1 (opset.rs, opset.fcd)
   | 40 -> M.MFC0 (opset.rd, opset.cs)
   | 41 -> M.MFC1 (opset.rd, opset.fs)
   | 42 -> M.MFHI opset.rd
   | 43 -> M.MFLO opset.rd
   | 44 -> M.MTC0 (opset.rs, opset.cd)
   | 45 -> M.MTC1 (opset.rs, opset.fd)
   | 46 -> M.MTHI opset.rs
   | 47 -> M.MTLO opset.rs
   | 48 -> M.MOVN (opset.rd, opset.rs, opset.rt)
   | 49 -> M.MOVZ (opset.rd, opset.rs, opset.rt)
   (* loads *)
   | 50 -> M.LB (opset.rd, opset.mem)
   | 51 -> M.LBU (opset.rd, opset.mem)
   | 52 -> M.LDC1 (opset.fd, opset.mem)
   | 53 -> M.LH (opset.rd, opset.mem)
   | 54 -> M.LHU (opset.rd, opset.mem)
   | 55 -> M.LW (opset.rd, opset.mem)
   | 56 -> M.LWC1 (opset.fd, opset.mem)
   (* stores *)
   | 57 -> M.SB (opset.rs, opset.mem)
   | 58 -> M.SDC1 (opset.fs, opset.mem)
   | 59 -> M.SH (opset.rs, opset.mem)
   | 60 -> M.SW (opset.rs, opset.mem)
   | 61 -> M.SWC1 (opset.fs, opset.mem)
   (* system control *)
   | 62 -> M.CACHE_INDEX_WBINV (opset.whichcache, opset.mem,
                                 opset.cachegolemsym)
   | 63 -> M.CACHE_VADDR_WB (opset.whichcache, opset.mem,
                                 opset.cachegolemsym)
   | 64 -> M.CACHE_VADDR_WBINV (opset.whichcache, opset.mem,
                                 opset.cachegolemsym)
   | 65 -> M.CACHE_VADDR_INV (opset.whichcache, opset.mem,
                                 opset.cachegolemsym)
   | 66 -> M.TLBP
   | 67 -> M.TLBR
   | 68 -> M.TLBWI
   | 69 -> M.TLBWR
   | _ ->
        let codestr = string_of_int code in
        Util.crash ("insns: decode: invalid instruction code " ^ codestr)

(* would be nice to not need to duplicate the list AGAIN *)
let get_code_and_fields pos opcode =
   match opcode with
   (* arithmetic *)
   | "addu" -> (0, ["rd"; "rs"; "rt"])
   | "addiu" -> (1, ["rd"; "rs"; "simm16"])
   | "div" -> (2, ["rs"; "rt"])
   | "divu" -> (3, ["rs"; "rt"])
   | "lui" -> (4, ["rd"; "uimm16"])
   | "mult" -> (5, ["rs"; "rt"])
   | "multu" -> (6, ["rs"; "rt"])
   | "subu" -> (7, ["rd"; "rs"; "rt"])
   | "lui_extcode_hi" -> (8, ["rd"; "xcl"])
   | "addiu_extcode_lo" -> (9, ["rd"; "rs"; "xcl"])
   | "lui_extdata_hi" -> (10, ["rd"; "xdl"])
   | "addiu_extdata_lo" -> (11, ["rd"; "rs"; "xdl"])
   | "lui_localcode_hi" -> (12, ["rd"; "lcl"])
   | "addiu_localcode_lo" -> (13, ["rd"; "rs"; "lcl"])
   (* comparisons *)
   | "slt" -> (14, ["rd"; "rs"; "rt"])
   | "slti" -> (15, ["rd"; "rs"; "simm16"])
   | "sltu" -> (16, ["rd"; "rs"; "rt"])
   | "sltiu" -> (17, ["rd"; "rs"; "simm16"])
   (* bitwise logic *)
   | "and" -> (18, ["rd"; "rs"; "rt"])
   | "andi" -> (19, ["rd"; "rs"; "uimm16"])
   | "nor" -> (20, ["rd"; "rs"; "rt"])
   | "or" -> (21, ["rd"; "rs"; "rt"])
   | "ori" -> (22, ["rd"; "rs"; "uimm16"])
   | "xor" -> (23, ["rd"; "rs"; "rt"])
   | "xori" -> (24, ["rd"; "rs"; "uimm16"])
   (* shifts *)
   | "sll" -> (25, ["rd"; "rs"; "uimm5"])
   | "sllv" -> (26, ["rd"; "rs"; "rt"])
   | "sra" -> (27, ["rd"; "rs"; "uimm5"])
   | "srav" -> (28, ["rd"; "rs"; "rt"])
   | "srl" -> (29, ["rd"; "rs"; "uimm5"])
   | "srlv" -> (30, ["rd"; "rs"; "rt"])
   (* transfers *)
   | "b" -> (31, ["lcl"])
   | "beq" -> (32, ["rs"; "rt"; "lcl"])
   | "bne" -> (33, ["rs"; "rt"; "lcl"])
   | "j" -> (34, ["xcl"])
   | "jal" -> (35, ["xcl"])
   | "jalr" -> (36, ["rs"])
   | "jr" -> (37, ["rs"])
   (* moves *)
   | "cfc1" -> (38, ["rd"; "fcs"])
   | "ctc1" -> (39, ["rs"; "fcd"])
   | "mfc0" -> (40, ["rd"; "cs"])
   | "mfc1" -> (41, ["rd"; "fs"])
   | "mfhi" -> (42, ["rd"])
   | "mflo" -> (43, ["rd"])
   | "mtc0" -> (44, ["rs"; "cd"])
   | "mtc1" -> (45, ["rs"; "fd"])
   | "mthi" -> (46, ["rs"])
   | "mtlo" -> (47, ["rs"])
   | "movn" -> (48, ["rd"; "rs"; "rt"])
   | "movz" -> (49, ["rd"; "rs"; "rt"])
   (* loads *)
   | "lb" -> (50, ["rd"; (*"mem"*) "memrs"; "memsimm16"])
   | "lbu" -> (51, ["rd"; (*"mem"*) "memrs"; "memsimm16"])
   | "ldc1" -> (52, ["fd"; (*"mem"*) "memrs"; "memsimm16"])
   | "lh" -> (53, ["rd"; (*"mem"*) "memrs"; "memsimm16"])
   | "lhu" -> (54, ["rd"; (*"mem"*) "memrs"; "memsimm16"])
   | "lw" -> (55, ["rd"; (*"mem"*) "memrs"; "memsimm16"])
   | "lwc1" -> (56, ["fd"; (*"mem"*) "memrs"; "memsimm16"])
   (* stores *)
   | "sb" -> (57, ["rs"; (*"mem"*) "memrs"; "memsimm16"])
   | "sdc1" -> (58, ["fs"; (*"mem"*) "memrs"; "memsimm16"])
   | "sh" -> (59, ["rs"; (*"mem"*) "memrs"; "memsimm16"])
   | "sw" -> (60, ["rs"; (*"mem"*) "memrs"; "memsimm16"])
   | "swc1" -> (61, ["fs"; (*"mem"*) "memrs"; "memsimm16"])
   (* system control *)
   | "cache" ->
        (* too many complications to bother with this *)
        Util.crash (Pos.string_of_pos pos ^ ": You may not sketch with " ^
                  "the cache instruction")
   | "tlbp" -> (66, [])
   | "tlbr" -> (67, [])
   | "tlbwi" -> (68, [])
   | "tlbwr" -> (69, [])
   | "_" -> (-1, [
			"rd"; "cd"; "fd"; "fcd";
			"rs"; "cs"; "fs"; "fcs";
			"rt";
			(*"whichcache";*)
			"simm16"; "uimm16"; "uimm5";
			(*"xcl"; "xdl"; "lcl"; "ldl"; *)
			"memrs"; "memsimm16";
		 ])
   | _ ->
        Util.crash (Pos.string_of_pos pos ^ ": Invalid opcode " ^ opcode)

let get_fieldname_sym field opset =
   let unpack x =
      match x with
      | CONC _ -> Util.crash ("insns: get_fieldname_sym: " ^ field ^ ": conc")
      | SYM x' -> x'
   in
   match field with
   | "rd" -> (unpack opset.rd, M.REGISTER)
   | "cd" -> (unpack opset.cd, M.CREGISTER)
   | "fd" -> (unpack opset.fd, M.FREGISTER)
   | "fcd" -> (unpack opset.fcd, M.FCREGISTER)
   | "rs" -> (unpack opset.rs, M.REGISTER)
   | "cs" -> (unpack opset.cs, M.CREGISTER)
   | "fs" -> (unpack opset.fs, M.FREGISTER)
   | "fcs" -> (unpack opset.fcs, M.FCREGISTER)
   | "rt" -> (unpack opset.rt, M.REGISTER)
   | "whichcache" -> (unpack opset.whichcache, M.WHICHCACHE)
   | "simm16" -> (unpack opset.simm16, M.SINT16)
   | "uimm16" -> (unpack opset.uimm16, M.UINT16)
   | "uimm5" -> (unpack opset.uimm5, M.UINT5)
   | "xcl" -> (unpack opset.xcl, M.BOOL) (* XXX bool is obviously wrong *)
   | "xdl" -> (unpack opset.xdl, M.BOOL) (* XXX bool is obviously wrong *)
   | "lcl" -> (unpack opset.lcl, M.BOOL) (* XXX bool is obviously wrong *)
   | "ldl" -> (unpack opset.ldl, M.BOOL) (* XXX bool is obviously wrong *)
   | "memrs" -> begin
        match opset.mem with
        | MEMORY (rs, _) -> (unpack rs, M.REGISTER)
     end
   | "memsimm16" -> begin
        match opset.mem with
        | MEMORY (_, simm16) -> (unpack simm16, M.SINT16)
     end
   | _ -> Util.crash ("insns: get_fieldname_sym: invalid field " ^ field)


let insn_bitwidth = 7

(* crosscheck *)
let _ =
   let (_, opset) = mkoperandset 0 in
   let one insn =
      let code = insncode insn in
      let insn' = decode opset code in
      if insn <> insn' then
         Util.crash ("insns: crosscheck: bad at " ^ string_of_int code)
   in
   List.iter one (insns opset)
