(*
 * Copyright (c) 2018-2019
 *	The President and Fellows of Harvard College.
 *
 * Written by David A. Holland.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *)

open Mipsstate
open Mips

(**************************************************************)
(* processor state toggles *)

(*
 * controls availability of:
 *    c0_index c0_random c0_entrylo c0_entryhi
 *    tlb contents
 *)
let use_c0_tlb = ref false

(*
 * controls availability of:
 *    c0_prid
 *)
let use_c0_configregs = ref false

(*
 * controls availability of:
 *    c0_badvaddr c0_epc
 *)
let use_c0_trapregs = ref false

(*
 * controls availability of:
 *    c0_count c0_compare
 *)
let use_c0_timerregs = ref false

(*
 * controls availability of:
 *    c0_status
 *)
let use_c0_status = ref false

(*
 * controls availability of:
 *    c0_context
 *)
let use_c0_context = ref false

(*
 * controls availability of:
 *    fpu control regs
 *    fpu regs
 *)
let use_fpu = ref false

(*
 * controls availability of:
 *    hi lo
 *)
let use_hilo = ref false

(*
 * controls availability of:
 *    cache state
 *)
let use_cache = ref true

(*
 * if enabled allows all general registers; otherwise restricts to
 *    z0, v0, v1, t0, t1 (0, 2, 3, 8, 9)
 *)
let use_allregs = ref false


(**************************************************************)
(* resource toggles *)

let use_extcodelabels = ref false
let use_extdatalabels = ref false
let use_localcodelabels = ref false
let use_localdatalabels = ref false
let use_memory = ref false


(**************************************************************)
(* instruction group toggles *)

(*
 * always enabled (for now)
 *    addu addiu lui subu slt slti sltu sltiu
 *    and andi nor or ori xor xori
 *    sll sllv sra srav srl srlv
 *)

(* div divu mult multu mfhi mflo mthi mtlo *)
let use_insn_muldiv = ref false

(* LUI_extcode_hi, ADDIU_extcode_lo *)
let use_insn_extcode = ref false

(* LUI_extdata_hi, ADDIU_extdata_lo *)
let use_insn_extdata = ref false

(* LUI_localcode_hi, ADDIU_localcode_lo *)
let use_insn_localcode = ref false

(* b beq bne *)
let use_insn_branches = ref false

(* j jal jalr jr *)
let use_insn_jumps = ref false

(* mfc0 mtc0 *)
let use_insn_c0 = ref false

(* cfc1 ctc1 mfc1 mtc1 *)
let use_insn_fpu = ref false

(* ldc1 lwc1 sdc1 swc1 *)
let use_insn_fpu_memory = ref false

(* lb lbu lh lhu lw sb sh sw *)
let use_insn_memory = ref false

(* cache *)
let use_insn_cache = ref true

(* tlbp tlbr tlbwi tlbwr *)
let use_insn_tlb = ref false


(**************************************************************)
(* test by state *)

let creg_enabled cr =
   match cr with
   | C0_STATUS -> !use_c0_status
   | C0_INDEX -> !use_c0_tlb
   | C0_RANDOM -> !use_c0_tlb
   | C0_ENTRYLO -> !use_c0_tlb
   | C0_ENTRYHI -> !use_c0_tlb
   | C0_CONTEXT -> !use_c0_context
   | C0_BADVADDR -> !use_c0_trapregs
   | C0_EPC -> !use_c0_trapregs
   | C0_COUNT -> !use_c0_timerregs
   | C0_COMPARE -> !use_c0_timerregs
   | C0_PRID -> !use_c0_configregs
   

(**************************************************************)
(* test by instruction *)

let insn_enabled insn =
   match insn with
     (* arithmetic *)
   | ADDU _ -> true
   | ADDIU _ -> true
   | DIV _ -> !use_insn_muldiv
   | DIVU _ -> !use_insn_muldiv
   | LUI _ -> true
   | MULT _ -> !use_insn_muldiv
   | MULTU _ -> !use_insn_muldiv
   | SUBU _ -> true
   | LUI_extcode_hi _ -> !use_insn_extcode
   | ADDIU_extcode_lo _ -> !use_insn_extcode
   | LUI_extdata_hi _ -> !use_insn_extdata
   | ADDIU_extdata_lo _ -> !use_insn_extdata
   | LUI_localcode_hi _ -> !use_insn_localcode
   | ADDIU_localcode_lo _ -> !use_insn_localcode
     (* comparisons *)
   | SLT _ -> true
   | SLTI _ -> true
   | SLTU _ -> true
   | SLTIU _ -> true
     (* bitwise logic *)
   | AND _ -> true
   | ANDI _ -> true
   | NOR _ -> true
   | OR _ -> true
   | ORI _ -> true
   | XOR _ -> true
   | XORI _ -> true
     (* shifts *)
   | SLL _ -> true
   | SLLV _ -> true
   | SRA _ -> true
   | SRAV _ -> true
   | SRL _ -> true
   | SRLV _ -> true
     (* transfers *)
   | B _ -> !use_insn_branches
   | BEQ _ -> !use_insn_branches
   | BNE _ -> !use_insn_branches
   | J _ -> !use_insn_jumps
   | JAL _ -> !use_insn_jumps
   | JALR _ -> !use_insn_jumps
   | JR _ -> !use_insn_jumps
     (* moves *)
   | CFC1 _ -> !use_insn_fpu
   | CTC1 _ -> !use_insn_fpu
   | MFC0 _ -> !use_insn_c0
   | MFC1 _ -> !use_insn_fpu
   | MFHI _ -> !use_insn_muldiv
   | MFLO _ -> !use_insn_muldiv
   | MTC0 _ -> !use_insn_c0
   | MTC1 _ -> !use_insn_fpu
   | MTHI _ -> !use_insn_muldiv
   | MTLO _ -> !use_insn_muldiv
   | MOVN _ -> true
   | MOVZ _ -> true
     (* loads *)
   | LB _ -> !use_insn_memory
   | LBU _ -> !use_insn_memory
   | LDC1 _ -> !use_insn_fpu_memory
   | LH _ -> !use_insn_memory
   | LHU _ -> !use_insn_memory
   | LW _ -> !use_insn_memory
   | LWC1 _ -> !use_insn_fpu_memory
     (* stores *)
   | SB _ -> !use_insn_memory
   | SDC1 _ -> !use_insn_fpu_memory
   | SH _ -> !use_insn_memory
   | SW _ -> !use_insn_memory
   | SWC1 _ -> !use_insn_fpu_memory
     (* system control *)
   | CACHE_INDEX_WBINV _ -> !use_insn_cache
   | CACHE_VADDR_WB _ -> !use_insn_cache
   | CACHE_VADDR_WBINV _ -> !use_insn_cache
   | CACHE_VADDR_INV _ -> !use_insn_cache
   | TLBP -> !use_insn_tlb
   | TLBR -> !use_insn_tlb
   | TLBWI -> !use_insn_tlb
   | TLBWR -> !use_insn_tlb


(**************************************************************)
(* autoset *)

let autoset () =
   let use_c0_state =
      !use_c0_tlb && !use_c0_configregs && !use_c0_trapregs &&
      !use_c0_timerregs && !use_c0_status && !use_c0_context
   in
   if use_c0_state = false then begin
      use_insn_c0 := false;
      use_insn_tlb := false
   end else ();

   if !use_c0_tlb = false then
      use_insn_tlb := false
   else ();

   if !use_fpu = false then begin
      use_insn_fpu := false;
      use_insn_fpu_memory := false
   end else ();

   if !use_hilo = false then
      use_insn_muldiv := false
   else ();

   if !use_extcodelabels = false then begin
      use_insn_extcode := false;
      use_insn_jumps := false
   end else ();

   if !use_extdatalabels = false then
      use_insn_extdata := false
   else ();

   if !use_localcodelabels = false then begin
      use_insn_localcode := false;
      use_insn_branches := false
   end else ();

   if !use_memory = false then begin
      use_insn_fpu_memory := false;
      use_insn_memory := false
   end else ();

   if !use_cache = false then begin
      use_insn_cache := false
   end else ()
