/* $NetBSD$ */

/*-
 * Copyright (c) 2012 Jared D. McNeill <jmcneill@invisible.ca>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD$");

#include <sys/param.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/module.h>
#include <sys/kmem.h>

#include <dev/acpi/acpivar.h>

#include "acpi_call_io.h"

extern struct cfdriver acpi_call_cd;

static dev_type_open(acpi_call_open);
static dev_type_close(acpi_call_close);
static dev_type_ioctl(acpi_call_ioctl);

static const struct cdevsw acpi_call_cdevsw = {
	.d_open = acpi_call_open,
	.d_close = acpi_call_close,
	.d_ioctl = acpi_call_ioctl,
	.d_read = noread,
	.d_write = nowrite,
	.d_stop = nostop,
	.d_tty = notty,
	.d_poll = nopoll,
	.d_mmap = nommap,
	.d_kqfilter = nokqfilter,
	.d_flag = D_OTHER,
};

struct acpi_call_softc {
	device_t		sc_dev;
};

static int	acpi_call_match(device_t, cfdata_t, void *);
static void	acpi_call_attach(device_t, device_t, void *);
static int	acpi_call_detach(device_t, int);

CFATTACH_DECL_NEW(acpi_call, sizeof(struct acpi_call_softc),
	acpi_call_match,
	acpi_call_attach,
	acpi_call_detach,
	NULL);

static int
acpi_call_match(device_t parent, cfdata_t cfdata, void *aux)
{
	extern struct acpi_softc *acpi_softc;

	return acpi_softc != NULL;
}

static void
acpi_call_attach(device_t parent, device_t self, void *aux)
{
	struct acpi_call_softc *sc = device_private(self);

	KASSERT(acpi_softc != NULL);

	device_printf(self, "attached\n");

	sc->sc_dev = self;
}

static int
acpi_call_detach(device_t self, int flags)
{
	return 0;
}

static int
acpi_call_open(dev_t dev, int flags, int fmt, lwp_t *l)
{
	return 0;
}

static int
acpi_call_close(dev_t dev, int flags, int fmt, lwp_t *l)
{
	return 0;
}

static int
acpi_call_invoke(const char *method, prop_array_t params)
{
	ACPI_OBJECT_LIST arg_list;
	ACPI_OBJECT *arg = NULL;
	ACPI_STATUS rv;
	prop_object_iterator_t iter;
	prop_object_t obj;
	unsigned int n;

	arg_list.Count = params ? prop_array_count(params) : 0;
	if (arg_list.Count > 0) {
		arg = kmem_alloc(arg_list.Count * sizeof(ACPI_OBJECT), KM_SLEEP);
		if (arg == NULL)
			return ENOMEM;
		iter = prop_array_iterator(params);
		n = 0;
		while ((obj = prop_object_iterator_next(iter)) != NULL) {
			switch (prop_object_type(obj)) {
			case PROP_TYPE_NUMBER:
				arg[n].Type = ACPI_TYPE_INTEGER;
				arg[n].Integer.Value = prop_number_integer_value((prop_number_t)obj);
				break;
			case PROP_TYPE_STRING:
				arg[n].Type = ACPI_TYPE_STRING;
				arg[n].String.Pointer = __UNCONST(prop_string_cstring_nocopy((prop_string_t)obj));
				break;
			case PROP_TYPE_DATA:
				arg[n].Type = ACPI_TYPE_BUFFER;
				arg[n].Buffer.Pointer = __UNCONST(prop_data_data_nocopy((prop_data_t)obj));
				arg[n].Buffer.Length = prop_data_size((prop_data_t)obj);
				break;
			default:
				prop_object_iterator_release(iter);
				return EINVAL;
			}
			++n;
		}
		prop_object_iterator_release(iter);
	}
	arg_list.Pointer = arg;

	rv = AcpiEvaluateObject(NULL, method, &arg_list, NULL);

	if (arg != NULL)
		kmem_free(arg, arg_list.Count * sizeof(ACPI_OBJECT));

	switch (rv) {
	case AE_OK:
		return 0;
	case AE_NOT_FOUND:
		return ENOENT;
	default:
		return EIO;
	}
}

static int
acpi_call_ioctl1(struct plistref *pref, u_long cmd)
{
	prop_dictionary_t dict;
	prop_array_t params;
	const char *method;
	int error;

	error = prop_dictionary_copyin_ioctl(pref, cmd, &dict);
	if (error)
		return error;

	if (!prop_dictionary_get_cstring_nocopy(dict, "method", &method)) {
		error = EINVAL;
		goto done;
	}
	params = prop_dictionary_get(dict, "params");

	error = acpi_call_invoke(method, params);

done:
	prop_object_release(dict);
	return error;
}

static int
acpi_call_ioctl(dev_t dev, u_long cmd, void *data, int flag, lwp_t *l)
{
	printf("cmd =       %lx\n", cmd);
	printf("ACPI_CALL = %lx\n", ACPI_CALL);
	switch (cmd) {
	case ACPI_CALL:
		return acpi_call_ioctl1(data, cmd);
	default:
		return EOPNOTSUPP;
	}
}

MODULE(MODULE_CLASS_MISC, acpi_call, NULL);

static const struct cfiattrdata * const acpi_call_attrs[] = { NULL };

CFDRIVER_DECL(acpi_call, DV_DULL, acpi_call_attrs);
extern struct cfattach acpi_call_ca;
static int acpi_call_loc[] = { -1, -1 };

static struct cfdata acpi_call_cfdata[] = {
	{
		.cf_name = "acpi_call",
		.cf_atname = "acpi_call",
		.cf_unit = 0,
		.cf_fstate = FSTATE_STAR,
		.cf_loc = acpi_call_loc,
		.cf_flags = 0,
		.cf_pspec = NULL,
	},
	{ NULL, NULL, 0, 0, NULL, 0, NULL },
};

static int
acpi_call_init(void)
{
	devmajor_t cmajor = NODEVMAJOR, bmajor = NODEVMAJOR;
	int error;

	if ((error = config_cfdriver_attach(&acpi_call_cd)) != 0)
		return error;
	if ((error = config_cfattach_attach(acpi_call_cd.cd_name, &acpi_call_ca)) != 0) {
		config_cfdriver_detach(&acpi_call_cd);
		return error;
	}
	if ((error = config_cfdata_attach(acpi_call_cfdata, 1)) != 0) {
		config_cfattach_detach(acpi_call_cd.cd_name, &acpi_call_ca);
		config_cfdriver_detach(&acpi_call_cd);
		return error;
	}
	if ((error = devsw_attach(acpi_call_cd.cd_name, NULL, &bmajor, &acpi_call_cdevsw, &cmajor)) != 0) {
		config_cfdata_detach(acpi_call_cfdata);
		config_cfattach_detach(acpi_call_cd.cd_name, &acpi_call_ca);
		config_cfdriver_detach(&acpi_call_cd);
		return error;
	}
	config_attach_pseudo(acpi_call_cfdata);

	return 0;
}

static int
acpi_call_fini(void)
{
	config_cfdata_detach(acpi_call_cfdata);
	config_cfattach_detach(acpi_call_cd.cd_name, &acpi_call_ca);
	config_cfdriver_detach(&acpi_call_cd);
	devsw_detach(NULL, &acpi_call_cdevsw);

	return 0;
}

static int
acpi_call_modcmd(modcmd_t cmd, void *arg)
{
	switch (cmd) {
	case MODULE_CMD_INIT:
		return acpi_call_init();	
	case MODULE_CMD_FINI:
		return acpi_call_fini();
	case MODULE_CMD_AUTOUNLOAD:
		return EBUSY;
	default:
		return ENOTTY;
	}
}
