/* $NetBSD$ */

/*-
 * Copyright (c) 2012 Jared D. McNeill <jmcneill@invisible.ca>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <prop/proplib.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "acpi_call_io.h"

static void
usage(const char *pn)
{
	fprintf(stderr, "usage: %s method [<s|i|d>=arg ...]\n", pn);
	exit(EXIT_FAILURE);
}

int
open_device(void)
{
	const char *dev_path;

	dev_path = getenv("ACPI_CALL_DEV");
	if (dev_path == NULL)
		dev_path = _PATH_ACPI_CALL;

	return open(dev_path, O_RDONLY);
}

int
main(int argc, char *argv[])
{
	prop_dictionary_t dict;
	prop_array_t arr;
	prop_string_t str;
	prop_number_t num;
	prop_data_t dat;
	const char *method;
	int n, error, fd;

	if (argc < 2)
		usage(argv[0]);
		/* NOTREACHED */

	method = argv[1];

	fd = open_device();
	if (fd == -1) {
		fprintf(stderr, "couldn't open acpi_call device: %s\n",
		    strerror(errno));
		return EXIT_FAILURE;
	}

	dict = prop_dictionary_create();
	arr = prop_array_create();

	prop_dictionary_set_cstring(dict, "method", method);
	for (n = 2; n < argc; n++) {
		if (strncmp(argv[n], "s=", 2) == 0) {
			str = prop_string_create_cstring(argv[n] + 2);
			prop_array_add(arr, str);
		} else if (strncmp(argv[n], "i=", 2) == 0) {
			num = prop_number_create_integer(strtol(argv[n] + 2, NULL, 0));
			prop_array_add(arr, num);
		} else if (strncmp(argv[n], "d=", 2) == 0) {
			uint8_t *buf;
			size_t len;
			uint32_t _num;
			int i;

			len = strlen(argv[n] + 2) / 2;
 			buf = calloc(1, len);
			for (i = 0; i < len; i++) {
				if (sscanf(argv[n] + 2 + (i * 2), "%02x", &_num) != 1) {
					fprintf(stderr, "bad data parameter\n");
					usage(argv[0]);
					/* NOTREACHED */
				}
				buf[i] = (uint8_t)_num;
			}
			dat = prop_data_create_data(buf, len);
			free(buf);
			prop_array_add(arr, dat);
		} else {
			usage(argv[0]);
			/* NOTREACHED */
		}
	}
	prop_dictionary_set(dict, "params", arr);

	error = prop_dictionary_send_ioctl(dict, fd, ACPI_CALL);
	if (error)
		printf("%s() returned %d (%s)\n", method, error, strerror(error));

	close(fd);

	return error;
}
