/*-
 * Copyright (c) 2012 Alistair Crooks <agc@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <sys/types.h>
#include <sys/param.h>

#include <inttypes.h>
#include <lzma.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "xzmem.h"

#define KB(x)			((x) * 1024)
#define MB(x)			((x) * 1024 * 1024)

#define DEFAULT_XZ_COMPRESSION	6
#define LARGEST_XZ_COMPRESSION	9

static const unsigned	sizes[10] = {
	/* 0 */	KB(256),
	/* 1 */	MB(1),
	/* 2 */	MB(2),
	/* 3 */	MB(4),
	/* 4 */	MB(4),
	/* 5 */	MB(8),
	/* 6 */	MB(8),
	/* 7 */	MB(16),
	/* 8 */	MB(32),
	/* 9 */	MB(64),
};

#define PRESET_DICT_SIZE	MB(1)

/* possibly map lzma errors into zlib error values */
static int
maperror(const int xzerror)
{
	if (getenv("ZLIB_ERROR_EMULATION") == NULL) {
		return xzerror;
	}
	switch(xzerror) {
	case LZMA_OK:
		return Z_OK;
	case LZMA_MEM_ERROR:
		return Z_MEM_ERROR;
	case LZMA_BUF_ERROR:
		return Z_BUF_ERROR;
	default:
		return Z_ERRNO;
	}
}

/* allocate space based on compression level */
static unsigned
allocate(uint8_t **array, int level)
{
	unsigned	 dictsize;

	if (level < 0 || level > 9) {
		level = DEFAULT_XZ_COMPRESSION;
	}
	dictsize = sizes[level];
	if ((*array = calloc(1, dictsize)) == NULL) {
		return 0;
	}
	return dictsize;
}

/* this function does all the compression */
static int
docompress(uint8_t *dest, size_t *destlen, const uint8_t *src, size_t srclen, int level)
{
	lzma_options_lzma	 opt;
	lzma_filter		 filters[2];
	lzma_ret		 ret;
	unsigned		 preset_dict_size;
	uint8_t			*preset_dict_buffer;
	size_t			 outcc;
	FILE			*fp;

	if ((fp = fmemopen(dest, *destlen, "w")) == NULL) {
		return maperror(LZMA_BUF_ERROR);
	}
	lzma_lzma_preset(&opt, LZMA_PRESET_DEFAULT);
	preset_dict_buffer = NULL;
	if ((preset_dict_size = allocate(&preset_dict_buffer, level)) == 0) {
		return maperror(LZMA_MEM_ERROR);
	}
	opt.dict_size = preset_dict_size;
	opt.preset_dict = preset_dict_buffer;
	opt.preset_dict_size = preset_dict_size;
	filters[0].id = LZMA_FILTER_LZMA2;
	filters[0].options = &opt;
	filters[1].id = LZMA_VLI_UNKNOWN;
	outcc = 0;
	ret = lzma_raw_buffer_encode(filters, NULL, src, srclen, dest, &outcc, *destlen);
	fclose(fp);
	*destlen = outcc;
	free(preset_dict_buffer);
	return maperror(ret);
}

/* compression via lzma - same signature as libz */
int
xz_compress(uint8_t *dest, size_t *destlen, const uint8_t *src, size_t srclen)
{
	return docompress(dest, destlen, src, srclen, DEFAULT_XZ_COMPRESSION);
}

/* compression via lzma - same signature as libz */
int
xz_compress2(uint8_t *dest, size_t *destlen, const uint8_t *src, size_t srclen, int level)
{
	return docompress(dest, destlen, src, srclen, level);
}

/* decompression via lzma - same signature as libz */
int
xz_decompress(uint8_t *dest, size_t *destlen, const uint8_t *src, size_t srclen)
{
	lzma_options_lzma	 opt;
	lzma_filter		 filters[2];
	lzma_ret		 ret;
	unsigned		 preset_dict_size;
	uint8_t			*preset_dict_buffer;
	size_t			 outcc;
	size_t			 incc;
	FILE			*fp;

	if ((fp = fmemopen(dest, *destlen, "w")) == NULL) {
		return -1;
	}
	lzma_lzma_preset(&opt, LZMA_PRESET_DEFAULT);
	preset_dict_buffer = NULL;
	/* we have to be pessimistic here */
	if ((preset_dict_size = allocate(&preset_dict_buffer, LARGEST_XZ_COMPRESSION)) == 0) {
		return maperror(LZMA_MEM_ERROR);
	}
	opt.dict_size = preset_dict_size;
	opt.preset_dict = preset_dict_buffer;
	opt.preset_dict_size = preset_dict_size;
	filters[0].id = LZMA_FILTER_LZMA2;
	filters[0].options = &opt;
	filters[1].id = LZMA_VLI_UNKNOWN;
	outcc = 0;
	incc = 0;
	ret = lzma_raw_buffer_decode(filters, NULL, src, &incc, srclen, dest, &outcc, *destlen);
	fclose(fp);
	*destlen = outcc;
	free(preset_dict_buffer);
	return maperror(ret);
}

/* estimate worst-case space needed for compression - same signature as libz */
size_t
xz_compressBound(size_t insize)
{
	return (insize * 104) / 100;
}
