/*-
 * Copyright (c) 2011 Alistair Crooks <agc@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <sys/types.h>
#include <sys/param.h>

#include <inttypes.h>
#include <sha1.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "hashtree.h"

/* read a key from a file */
static ssize_t
readkey(const char *f, uint8_t *key, size_t keysize)
{
	ssize_t	 cc;
	FILE	*fp;

	if ((fp = fopen(f, "r")) == NULL) {
		(void) fprintf(stderr, "hmac: can't open keyfile '%s'\n", f);
		return -1;
	}
	if ((cc = read(fileno(fp), key, keysize)) < 0) {
		(void) fprintf(stderr, "hmac: can't read keyfile '%s'\n", f);
		cc = -1;
	}
	(void) fclose(fp);
	return cc;
}

#define MB(x)	(1024 * 1024 * (x))

/* read from stdin (up to a max size) */
static ssize_t
dostdin(hashtree_t *hashtree, const char *alg, uint8_t *key, int sz)
{
	uint8_t	*buf;
	uint8_t	*out;
	ssize_t	 cc;
	size_t	 sumbytes;
	int	 i;

	if ((buf = calloc(1, (unsigned)sz)) == NULL) {
		(void) fprintf(stderr, "can't allocate %d bytes\n", sz);
		return -1;
	}
	for (cc = 0 ; (i = read(STDIN_FILENO, &buf[cc], (unsigned)(sz - cc))) > 0 && cc < sz ; cc += i) {
	}
	sumbytes = HASHTREE_Sumsize(hashtree, cc);
	out = calloc(1, sumbytes);
	HASHTREE_Data(hashtree, buf, cc, out);
	HASHTREE_Print(hashtree, stdout, NULL, cc, 0, out);
	free(out);
	free(buf);
	return cc;
}

int
main(int argc, char **argv)
{
	hashtree_t	 hashtree;
	const char	*alg;
	uint64_t	 blocksize;
	uint64_t	 blocks;
	uint8_t		 key[128];
	ssize_t		 size;
	uint8_t		*out;
	int32_t		 keylen;
	char		*keyfile;
	char		*s;
	int		 sz;
	int		 ex;
	int		 i;

	(void) memset(&hashtree, 0x0, sizeof(hashtree));
	alg = "sha1";
	keyfile = NULL;
	(void) memset(key, 0x0, sizeof(key));
	sz = MB(4);
	blocksize = 0;
	blocks = 0;
	keylen = 0;
	while ((i = getopt(argc, argv, "a:b:f:k:n:s:")) != -1) {
		switch(i) {
		case 'a':
			alg = optarg;
			break;
		case 'b':
			blocksize = strtoull(optarg, NULL, 10);
			break;
		case 'f':
			keyfile = optarg;
			break;
		case 'k':
			(void) snprintf((char *)key, sizeof(key), "%s", optarg);
			break;
		case 'n':
			blocks = strtoull(optarg, NULL, 10);
			break;
		case 's':
			sz = atoi(optarg);
			break;
		default:
			break;
		}
	}
	HASHTREE_Init(&hashtree, alg, blocksize, blocks);
	if (HASHTREE_Keyneeded(&hashtree)) {
		if (keyfile) {
			if ((keylen = readkey(keyfile, key, sizeof(key))) < 0) {
				exit(EXIT_FAILURE);
			}
		} else if (key[0] == 0x0) {
			s = getpass("hmac key: ");
			keylen = snprintf((char *)key, sizeof(key), "%s", s);
		}
		HASHTREE_Setkey(&hashtree, key, (uint32_t)keylen);
	} else {
		(void) memset(key, 0x0, sizeof(key));
		keylen = 0;
	}
	ex = EXIT_SUCCESS;
	if (optind == argc) {
		if (!dostdin(&hashtree, alg, key, sz)) {
			ex = EXIT_FAILURE;
		}
	} else {
		for (i = optind ; i < argc ; i++) {
			if ((size = HASHTREE_File(&hashtree, argv[i], &out)) >= 0) {
				HASHTREE_Print(&hashtree, stdout, argv[i], size, 0, out);
				free(out);
			} else {
				ex = EXIT_FAILURE;
			}
		}
	}
	exit(ex);
}
