#! /bin/sh

# $NetBSD$

# Copyright (c) 2011 Alistair Crooks <agc@NetBSD.org>
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#


# Usage:
#
# getpkgsrc [-d=repo] [-p] [-v] category/package...
#	-d=repo is anoncvs@anoncvs.netbsd.org:/cvsroot
#	-p means preserve checked out tree
#	-v means to act verbsoely
#	packages should be specified as <category>/<package-name>
#
# this utility requires a bootstrapped pkgsrc, and a cvs utility

# utility function to print a message and die
die() {
	echo $* >&2
	exit 1
}

# these are the packages we need
musthaves="sysutils/install-sh"

# function which checks out a single
checkout() {
	# get the package wanted
	cvs -d $repo co -P pkgsrc/$1

	# get any prereqs
	bl3=""
	if [ -f pkgsrc/$1/buildlink3.mk ]; then
		bl3=pkgsrc/$1/buildlink3.mk
	fi
	prereqs="${musthaves} $(awk '/^.include.*\/buildlink3.mk/ { sub("\"\\.\\./\\.\\./", "", $2); sub("/buildlink3.mk\"", "", $2); print $2 }' pkgsrc/$1/Makefile ${bl3})"
	tools="$(awk '/^[ 	]*USE_LIBTOOL=/ { print "devel/libtool-base" }' pkgsrc/$1/Makefile)"
	# XXX - other tools here

	# check the prereqs out
	for p in ${prereqs} ${tools}; do
		if [ ! -d pkgsrc/${p} ]; then
			checkout ${p}
		fi
	done
}

# check arguments
case $# in
0)	die "No package names given" ;;
esac

# local variables
removetree=true
repo=anoncvs@anoncvs.netbsd.org:/cvsroot

# command line arguments
while [ $# -gt 0 ]; do
	case "$1" in
	-d=*)	repo=$(echo "$1" | sed -e 's|-d=||') ;;
	-p)	removetree=false ;;
	-v)	set -x ;;
	*)	break ;;
	esac
	shift
done

case "$(uname -s)" in
NetBSD)	make=make ;;
*)	make=bmake ;;
esac

# check we can write in $cwd
date > .getpkgsrc || die "can't create files in $(pwd)"
rm .getpkgsrc

# get the mk tree
cvs -d $repo co -P pkgsrc/mk

# get package and dependency tree too
for d in $*; do
	checkout ${d}
done

# also look for tools
tooldeps=""
for d in $*; do
	tools=$(cd pkgsrc/${d} && ${make} show-var VARNAME=TOOLS_CREATE | awk '{ sub("\\[.*", ""); print }')
	for t in ${tools}; do
		tooldep=$(cd pkgsrc/$d && ${make} show-var VARNAME=TOOLS_DEPENDS.${t})
		case "${tooldep}" in
		"")	;;
		*)	tdir=$(echo "${tooldep}" | awk '{ sub(".*:\\.\\./\\.\\./", ""); print }')
			checkout ${tdir}
			;;
		esac
	done
done

# make it
made=""
notmade=""
for d in $*; do
	(cd pkgsrc/$d && ${make} install)
	pkg=$(basename $d)
	if $(pkg_info -qe $pkg); then
		made="${made} $pkg"
		echo "package ${pkg} built and installed"
	else
		notmade="${notmade} $pkg"
	fi
done

# clean up
case "${notmade}" in
"")	if $removetree; then
		rm -rf pkgsrc
	fi
	exit 0 ;;
*)	die "Problems in packaging ${notmade}" ;;
esac

