/* hashtest file from Simon Burge:

	ftp://ftp.netbsd.org/pub/NetBSD/misc/simonb/hashtest.tar.gz

See

	http://mail-index.netbsd.org/tech-perform/2001/11/29/0002.html

*/

#include <sys/types.h>

#include <arpa/inet.h>

#include <stdio.h>
#include <string.h>

#include "hashes.h"

#ifndef USE_ARG
#define USE_ARG(x)    /*LINTED*/(void)&x
#endif

static unsigned int
dumbhash(const unsigned char *str, size_t len)
{
	unsigned int hash;

	hash = 0;
	while (len-- > 0)
		hash += *str++;
	return (hash);
}

static unsigned int
dumbmulhash(const unsigned char *str, size_t len)
{
	unsigned int hash;
	size_t i;

	hash = 0;
	for (i = 1; i <= len; i++)
		hash += *str++ * i;
	return (hash);
}

/*
** A simple and fast generic string hasher based on Peter K. Pearson's
** article in CACM 33-6, pp. 677.
*/

static unsigned char T[] = {
    1, 87, 49, 12, 176, 178, 102, 166, 121, 193, 6, 84, 249, 230, 44, 163,
    14, 197, 213, 181, 161, 85, 218, 80, 64, 239, 24, 226, 236, 142, 38, 200,
    110, 177, 104, 103, 141, 253, 255, 50, 77, 101, 81, 18, 45, 96, 31, 222,
    25, 107, 190, 70, 86, 237, 240, 34, 72, 242, 20, 214, 244, 227, 149, 235,
    97, 234, 57, 22, 60, 250, 82, 175, 208, 5, 127, 199, 111, 62, 135, 248,
    174, 169, 211, 58, 66, 154, 106, 195, 245, 171, 17, 187, 182, 179, 0, 243,
    132, 56, 148, 75, 128, 133, 158, 100, 130, 126, 91, 13, 153, 246, 216, 219,
    119, 68, 223, 78, 83, 88, 201, 99, 122, 11, 92, 32, 136, 114, 52, 10,
    138, 30, 48, 183, 156, 35, 61, 26, 143, 74, 251, 94, 129, 162, 63, 152,
    170, 7, 115, 167, 241, 206, 3, 150, 55, 59, 151, 220, 90, 53, 23, 131,
    125, 173, 15, 238, 79, 95, 89, 16, 105, 137, 225, 224, 217, 160, 37, 123,
    118, 73, 2, 157, 46, 116, 9, 145, 134, 228, 207, 212, 202, 215, 69, 229,
    27, 188, 67, 124, 168, 252, 42, 4, 29, 108, 21, 247, 19, 205, 39, 203,
    233, 40, 186, 147, 198, 192, 155, 33, 164, 191, 98, 204, 165, 180, 117, 76,
    140, 36, 210, 172, 41, 54, 159, 8, 185, 232, 113, 196, 231, 47, 146, 120,
    51, 65, 28, 144, 254, 221, 93, 189, 194, 139, 112, 43, 71, 109, 184, 209,
};

static unsigned int
lennart(const unsigned char *str, size_t len)
{
	unsigned int h1, h2;

	for (h1 = 0, h2 = str[0]; --len > 0 ; str++) {
		h1 = T[h1 ^ str[0]];
		h2 = T[h2 ^ str[1]];
	}

	return (h1 << 8) | h2;
}


static unsigned int
crchash(const unsigned char *str, size_t len)
{
	unsigned int hash, highorder;

	hash = 0;
	while (len-- > 0) {
		highorder = hash & 0xf8000000;
		hash = hash << 5;
		hash = hash ^ (highorder >> 27);
		hash = hash ^ *str++;
	}
	return (hash);
}

static unsigned int
perlhash(const unsigned char *str, size_t len)
{
	unsigned int hash;

	hash = 0;
	while (len-- > 0) {
		hash = hash * 33 + *str++;
	}
	hash = hash + (hash >> 5);

	return (hash);
}

static unsigned int
perlxorhash(const unsigned char *str, size_t len)
{
	unsigned int hash;

	hash = 0;
	while (len-- > 0) {
		hash = hash * 33 ^ *str++;
	}
	hash = hash + (hash >> 5);

	return (hash);
}

static unsigned int
pythonhash(const unsigned char *str, size_t len)
{
	unsigned int hash;

	hash = 0;
	while (len--) {
		hash += *str++;
		hash += (hash << 10);
		hash ^= (hash >> 6);
	}
	hash += (hash << 3);
	hash ^= (hash >> 11);
	hash = (hash + (hash << 15));

	return (hash);
}

static unsigned int
mousehash(const unsigned char *str, size_t len)
{
	unsigned hash;

	hash = 0xffffffff;
	while (len-- > 0) {
		hash ^= *str++;
		hash = (hash >> 1) ^ ((hash & 1) ? 0xedb88320 : 0);
	}

	return (hash);
}

static unsigned int
bernstein(const unsigned char *str, size_t len)
{
	unsigned int hash;

	hash = 5381;
	while (len-- > 0) {
		hash = hash * 33 + *str++;
	}
	hash = hash + (hash >> 5);

	return (hash);
}

static unsigned int honeymanCrcTable[128] = {
	0, 1207959552, 603979776, 1811939328, 301989888, 1509949440, 905969664,
	2113929216, 150994944, 1090519040, 754974720, 1694498816, 452984832,
	1392508928, 1056964608, 1996488704, 75497472, 1283457024, 545259520,
	1753219072, 377487360, 1585446912, 847249408, 2055208960, 226492416,
	1166016512, 696254464, 1635778560, 528482304, 1468006400, 998244352,
	1937768448, 37748736, 1245708288, 641728512, 1849688064, 272629760,
	1480589312, 876609536, 2084569088, 188743680, 1128267776, 792723456,
	1732247552, 423624704, 1363148800, 1027604480, 1967128576, 113246208,
	1321205760, 583008256, 1790967808, 348127232, 1556086784, 817889280,
	2025848832, 264241152, 1203765248, 734003200, 1673527296, 499122176,
	1438646272, 968884224, 1908408320, 18874368, 1226833920, 622854144,
	1830813696, 320864256, 1528823808, 924844032, 2132803584, 136314880,
	1075838976, 740294656, 1679818752, 438304768, 1377828864, 1042284544,
	1981808640, 94371840, 1302331392, 564133888, 1772093440, 396361728,
	1604321280, 866123776, 2074083328, 211812352, 1151336448, 681574400,
	1621098496, 513802240, 1453326336, 983564288, 1923088384, 56623104,
	1264582656, 660602880, 1868562432, 291504128, 1499463680, 895483904,
	2103443456, 174063616, 1113587712, 778043392, 1717567488, 408944640,
	1348468736, 1012924416, 1952448512, 132120576, 1340080128, 601882624,
	1809842176, 367001600, 1574961152, 836763648, 2044723200, 249561088,
	1189085184, 719323136, 1658847232, 484442112, 1423966208, 954204160,
	1893728256,
};

static unsigned int
honeyman(const unsigned char *str, size_t len)
{
	unsigned int hash;

	hash = 0;

	while (len-- > 0)
		hash = (hash >> 7) ^ honeymanCrcTable[(hash ^ (*str++)) & 0x7f];

	return (hash);
}

/* Defines the so called `hashpjw' function by P.J. Weinberger
   [see Aho/Sethi/Ullman, COMPILERS: Principles, Techniques and Tools,
   1986, 1987 Bell Telephone Laboratories, Inc.]  */

/* We assume to have `unsigned long int' value with at least 32 bits.  */
#define	HASHWORDBITS	32

static unsigned int
pjwhash(const unsigned char *str, size_t len)
{
	unsigned int hash, g;

	/* Compute the hash value for the given string.  */
	hash = 0;
	while (len-- > 0) {
		hash <<= 4;
		hash += *str++;
		g = hash & (0xf << (HASHWORDBITS - 4));
		if (g != 0) {
			hash ^= g >> (HASHWORDBITS - 8);
			hash ^= g;
		}
	}

	return (hash);
}

#define bobmix(a,b,c) \
{ \
  a -= b; a -= c; a ^= (c>>13); \
  b -= c; b -= a; b ^= (a<<8); \
  c -= a; c -= b; c ^= (b>>13); \
  a -= b; a -= c; a ^= (c>>12);  \
  b -= c; b -= a; b ^= (a<<16); \
  c -= a; c -= b; c ^= (b>>5); \
  a -= b; a -= c; a ^= (c>>3);  \
  b -= c; b -= a; b ^= (a<<10); \
  c -= a; c -= b; c ^= (b>>15); \
}

static unsigned int
bobhash(const unsigned char *k, size_t length)
{
  unsigned int a, b, c, len;

  /* Set up the internal state */
  len = length;
  a = b = 0x9e3779b9;  /* the golden ratio; an arbitrary value */
  c = 0;

  /*---------------------------------------- handle most of the key */
  while (len >= 12)
    {
      a += (k[0] +(k[1]<<8) +(k[2]<<16) +(k[3]<<24));
      b += (k[4] +(k[5]<<8) +(k[6]<<16) +(k[7]<<24));
      c += (k[8] +(k[9]<<8) +(k[10]<<16)+(k[11]<<24));
      bobmix(a,b,c);
      k += 12; len -= 12;
    }

  /*------------------------------------- handle the last 11 bytes */
  c += length;
  switch(len)              /* all the case statements fall through */
    {
    case 11: c+=(k[10]<<24);
    	/*FALLTHROUGH*/
    case 10: c+=(k[9]<<16);
    	/*FALLTHROUGH*/
    case 9 : c+=(k[8]<<8);
      /* the first byte of c is reserved for the length */
    	/*FALLTHROUGH*/
    case 8 : b+=(k[7]<<24);
    	/*FALLTHROUGH*/
    case 7 : b+=(k[6]<<16);
    	/*FALLTHROUGH*/
    case 6 : b+=(k[5]<<8);
    	/*FALLTHROUGH*/
    case 5 : b+=k[4];
    	/*FALLTHROUGH*/
    case 4 : a+=(k[3]<<24);
    	/*FALLTHROUGH*/
    case 3 : a+=(k[2]<<16);
    	/*FALLTHROUGH*/
    case 2 : a+=(k[1]<<8);
    	/*FALLTHROUGH*/
    case 1 : a+=k[0];
    	/*FALLTHROUGH*/
      /* case 0: nothing left to add */
    }
  bobmix(a,b,c);
  /*-------------------------------------------- report the result */
  return c;
}

static unsigned int
torekhash(const unsigned char *str, size_t len)
{
	unsigned int hash;
	size_t loop;

#define HASH4a   hash = (hash << 5) - hash + *str++;
#define HASH4b   hash = (hash << 5) + hash + *str++;
#define HASH4 HASH4b

	hash = 0;
	if (len > 0) {
		loop = (len + 8 - 1) >> 3;

		switch (len & (8 - 1)) {
		case 0:
			do {
				HASH4;
				/* FALLTHROUGH */
		case 7:
				HASH4;
				/* FALLTHROUGH */
		case 6:
				HASH4;
				/* FALLTHROUGH */
		case 5:
				HASH4;
				/* FALLTHROUGH */
		case 4:
				HASH4;
				/* FALLTHROUGH */
		case 3:
				HASH4;
				/* FALLTHROUGH */
		case 2:
				HASH4;
				/* FALLTHROUGH */
		case 1:
				HASH4;
			} while (--loop);
		}
	}

	return (hash);
}

static unsigned int
byacchash(const unsigned char *str, size_t len)
{
	unsigned int hash;

	hash = 0;

	while (len-- > 0)
		hash = 31 * hash + *str++;

	return (hash);
}

static unsigned int
tclhash(const unsigned char *str, size_t len)
{
	unsigned int hash;

	hash = 0;

	while (len-- > 0)
		hash = (hash << 3) + hash + *str++;

	return (hash);
}

static unsigned int
gawkhash(const unsigned char *s, size_t len)
{
	unsigned int h = 0;

	/*
	 * This is INCREDIBLY ugly, but fast.  We break the string up into
	 * 8 byte units.  On the first time through the loop we get the
	 * "leftover bytes" (strlen % 8).  On every other iteration, we
	 * perform 8 HASHC's so we handle all 8 bytes.  Essentially, this
	 * saves us 7 cmp & branch instructions.  If this routine is
	 * heavily used enough, it's worth the ugly coding.
	 *
	 * OZ's original sdbm hash, copied from Margo Seltzers db package.
	 */

	/*
	 * Even more speed:
	 * #define HASHC   h = *s++ + 65599 * h
	 * Because 65599 = pow(2, 6) + pow(2, 16) - 1 we multiply by shifts
	 */
#define HASHC   htmp = (h << 6); h = *s++ + htmp + (htmp << 10) - h

	unsigned int htmp;

	h = 0;

	/* "Duff's Device" for those who can handle it */
	if (len > 0) {
		size_t loop = (len + 8 - 1) >> 3;

		switch (len & (8 - 1)) {
		case 0:
			do {    /* All fall throughs */
				HASHC; /*FALLTHROUGH*/
		case 7:         HASHC; /*FALLTHROUGH*/
		case 6:         HASHC; /*FALLTHROUGH*/
		case 5:         HASHC; /*FALLTHROUGH*/
		case 4:         HASHC; /*FALLTHROUGH*/
		case 3:         HASHC; /*FALLTHROUGH*/
		case 2:         HASHC; /*FALLTHROUGH*/
		case 1:         HASHC; /*FALLTHROUGH*/
			} while (--loop);
		}
	}

	return h;
}

static unsigned int
gcc3_hash (const unsigned char *str, size_t len)
{
	unsigned int n = len;
	unsigned int r = 0;
#define HASHSTEP(r, c) ((r) * 67 + ((c) - 113));

	while (n--)
		r = HASHSTEP (r, *str++);

	return r + len;
#undef HASHSTEP
}

static unsigned int
gcc3_hash2 (const unsigned char *str, size_t len)
{
	unsigned int r = 0;
#define HASHSTEP(r, c) ((r) * 67 + ((c) - 113));

	while (len--)
		r = HASHSTEP (r, *str++);

	return r;
#undef HASHSTEP
}

static unsigned int
nemhash(const unsigned char *str, size_t len)
{
	unsigned int hash;

	hash = 0;
	while (len--)
		hash += (hash << 8) + *str++;

	return (hash + (hash << 8));
}

/*
 * we opencode the hash values from the perlhash command here, since
 * we're defining a switch table for hash functions, and to do a
 * stepwise string comparison here would be missing the point a bit.
 * perlhash was chosen for its speed and range, see discussion on the
 * mailing list referenced above.
 */
int
hashfunc(const char *in, const size_t insize, const char *op, void *vp, size_t outsize)
{
	unsigned	res;

	USE_ARG(outsize);
	switch(perlhash((const uint8_t *)op, strlen(op))) {
	case 0x816b47b6:
		res = dumbhash((const uint8_t *)in, insize);
		break;
	case 0xe163b02d:
		res = dumbmulhash((const uint8_t *)in, insize);
		break;
	case 0xaf865070:
		res = lennart((const uint8_t *)in, insize);
		break;
	case 0xdc9b2197:
		res = crchash((const uint8_t *)in, insize);
		break;
	case 0xd3babdd6:
		res = perlhash((const uint8_t *)in, insize);
		break;
	case 0xa150cff6:
		res = perlxorhash((const uint8_t *)in, insize);
		break;
	case 0x336ac3ec:
		res = pythonhash((const uint8_t *)in, insize);
		break;
	case 0x4de0f41a:
		res = mousehash((const uint8_t *)in, insize);
		break;
	case 0x9419892e:
		res = bernstein((const uint8_t *)in, insize);
		break;
	case 0xc9bfa906:
		res = honeyman((const uint8_t *)in, insize);
		break;
	case 0xb2cc8a47:
		res = pjwhash((const uint8_t *)in, insize);
		break;
	case 0x85eef3f9:
		res = bobhash((const uint8_t *)in, insize);
		break;
	case 0x775f94ee:
		res = torekhash((const uint8_t *)in, insize);
		break;
	case 0x5e989f90:
		res = byacchash((const uint8_t *)in, insize);
		break;
	case 0xd6b0c1a2:
		res = tclhash((const uint8_t *)in, insize);
		break;
	case 0x5963f3ab:
		res = gawkhash((const uint8_t *)in, insize);
		break;
	case 0x8226d5a5:
		res = gcc3_hash((const uint8_t *)in, insize);
		break;
	case 0x6520f2ce:
		res = gcc3_hash2((const uint8_t *)in, insize);
		break;
	case 0x746b6c9:
		res = nemhash((const uint8_t *)in, insize);
		break;
	default:
		(void) fprintf(stderr, "no hash function for '%s'\n", op);
		return 0;
	}
	res = htonl(res);
	(void) memcpy(vp, &res, sizeof(res));
	return sizeof(res);
}
