/*-
 * Copyright (c) 2010 Alistair Crooks <agc@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "codecs.h"

/* read all of stdin into memory */
static int
readmem(char **temp)
{
	unsigned	 newsize;
	unsigned	 size;
	size_t		 maxsize;
	char		 buf[BUFSIZ * 8];
	char		*loc;
	int	 	 n;

	maxsize = 4 * 1024 * 1024;
	*temp = NULL;
	for (size = 0; (n = read(STDIN_FILENO, buf, sizeof(buf))) > 0 ; size += n) {
		/* round up the allocation */
		newsize = size + ((n / BUFSIZ) + 1) * BUFSIZ;
		if (newsize > maxsize) {
			(void) fprintf(stderr, "bounds check\n");
			return (int)size;
		}
		loc = realloc(*temp, newsize);
		if (loc == NULL) {
			(void) fprintf(stderr, "short read\n");
			return size;
		}
		(void) memset(&loc[size], 0x0, newsize - size);
		*temp = loc;
		(void) memcpy(&(*temp)[size], buf, n);
	}
	return (int)size;
}

/* perform the action aliased by 'op' for a file */
static int
readfile(const char *f, char **in)
{
	struct stat	 st;
	size_t		 len;
	FILE		*fp;
	int		 cc;
	int		 done;

	if (f == NULL) {
		return readmem(in);
	}
	if ((fp = fopen(f, "r")) == NULL) {
		(void) fprintf(stderr, "can't open '%s'\n", f);
		return 0;
	}
	(void) fstat(fileno(fp), &st);
	len = (size_t)st.st_size;
	if ((*in = calloc(1, len + 1)) == NULL) {
		(void) fprintf(stderr, "can't malloc %zu bytes\n", len);
		(void) fclose(fp);
		return 0;
	}
	for (cc = 0 ; (done = read(fileno(fp), &(*in)[cc], len - cc)) > 0 && (size_t)cc < len ; cc += done) {
	}
	(void) fclose(fp);
	return cc;
}

/* translate the memory */
static int
xlatemem(codecs_t *codecs, const char *in, size_t len, const char *op, const int fd)
{
	size_t	 outsize;
	size_t	 got;
	size_t	 cc;
	char	*out;
	int	 wc;

	if ((got = codecs_alloc_transform(codecs, in, len, op, &out, &outsize)) > 0) {
		for (cc = 0 ; (wc = write(fd, &out[cc], got - cc)) > 0 && cc < got ; cc += wc) {
		}
	}
	free(out);
	return got;
}

/* do the action in op for a file - or memory */
static int
dofile(codecs_t *codecs, const char *f, const char *op, const int fd)
{
	char	*in;
	int	 len;
	int	 ret;

	in = NULL;
	if (!codecs_valid_op(codecs, op)) {
		(void) fprintf(stderr, "codecs: invalid operation '%s'\n", op);
		return 0;
	}
	if (codecs_input_needed(codecs, op)) {
		if ((len = readfile(f, &in)) == 0) {
			return 0;
		}
	} else {
		len = strtol(f, NULL, 10);
	}
	ret = xlatemem(codecs, in, len, op, fd);
	if (in != NULL) {
		free(in);
	}
	return ret > 0;
}

int
main(int argc, char **argv)
{
	codecs_t	 codecs;
	const char	*out;
	FILE		*fp;
	int	 	 fd;
	int	 	 ex;
	int	 	 i;

	(void) memset(&codecs, 0x0, sizeof(codecs));
	out = NULL;
	while ((i = getopt(argc, argv, "o:")) != -1) {
		switch(i) {
		case 'o':
			out = optarg;
			break;
		default:
			break;
		}
	}
	ex = EXIT_SUCCESS;
	if (out == NULL) {
		fd = STDOUT_FILENO;
		fp = NULL; /* sigh, only here to shut up the compiler */
	} else {
		if ((fp = fopen(out, "w")) == NULL) {
			(void) fprintf(stderr, "codecs: can't open '%s'\n", out);
			return -1;
		}
		fd = fileno(fp);
	}
	if (optind == argc - 1) {
		if (!dofile(&codecs, NULL, argv[optind], fd)) {
			ex = EXIT_FAILURE;
		}
	} else {
		for (i = optind + 1 ; i < argc ; i++) {
			if (!dofile(&codecs, argv[i], argv[optind], fd)) {
				ex = EXIT_FAILURE;
			}
		}
	}
	codecs_end(&codecs);
	exit(ex);
}
