#! /bin/sh

# $NetBSD$

# Copyright (c) 2012 Alistair Crooks <agc@NetBSD.org>
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

# from scratch shell script

bsdlibtool_version=20120721

machine_arch=$(uname -p)
vendor="unknown"
os=$(uname -s | tr A-Z a-z)
c_pic_flags="-fPIC -DPIC"
cpp_pic_flags="-DPIC"
sparc_cpp_pic_flags="-DPIC -DBIGPIC"
tag=""
libtype="shared"
mode=""
doit=true
verbose=false
current=0
age=0
revision=0

# args are mode, then cmd
runcmd() {
	${verbose} && echo "bsdlibtool: $1: $2"
	${doit} && eval $2
}

# parse the command line args
while [ $# -gt 0 ]; do
	case "$1" in
	--dry-run)
		doit=false
		;;
	--help)
		cat << EOF
Usage: bsdlibtool --mode=MODE commands

bsdlibtool --dry-run
	do not execute any commands
bsdlibtool --help
	print this message
	bsdlibtool --help
bsdlibtool --mode=compile
	compile both shared and unshared object files:
	bsdlibtool --tag=CC --mode=compile --verbose cc -g -O -c hello.c
bsdlibtool --mode=execute
	add a library directory to the library path and execute a command:
	bsdlibtool --mode=execute -dlopen /usr/local/lib/libhello.la greet
bsdlibtool --mode=finish
	complete libtool installation process
	bsdlibtool --mode=finish
bsdlibtool --mode=install
	install shared and unshared objects and symbolic links in the target directory
	bsdlibtool --mode=install --verbose install -c -m 0755 \\
		-o root -g wheel libhello.la /usr/local/lib
bsdlibtool --mode=link
	make shared and unshared library objects
	bsdlibtool --mode=link --verbose cc -g -O -o libhello.la \\
		foo.lo hello.lo -rpath /usr/local/lib -lm -version-info 3:0
bsdlibtool --tag=language
	use the language argument as an indication to bsdlibtool which form
	of linking to do
	bsdlibtool --tag=CC
bsdlibtool --target=cpu-vendor-os
	perform build based on definitions for the target machine arch,
	the hardware vendor, and the operating system
	bsdlibtool --target=sparc64-unknown-netbsd
bsdlibtool --verbose
	perform operations in a verbose manner
bsdlibtool --version
	print the version number and exit

EOF
		exit 0
		;;
	--mode=*)
		mode=${1#--mode=}
		;;
	--tag=*)
		tag=${1#--tag=}
		;;
	--target=*)
		machine_arch=${1#--target=}
		os=${machine_arch##*-}
		vendor=${machine_arch#*-}
		vendor=${vendor%-*}
		machine_arch=${machine_arch%%-*}
		;;
	--verbose)
		verbose=true
		;;
	--version)
		echo "bsdlibtool ${bsdlibtool_version}"
		exit 0
		;;
	*)
		break
		;;
	esac
	shift
done

# customise based on target machine os
case "${os}" in
netbsd)
	# customise based on target machine arch
	case "${machine_arch}" in
	sparc|sparc64)
		cpp_pic_flags=${sparc_cpp_pic_flags}
		;;
	esac
esac

# libtool mode
case "${mode}" in
compile)
	case "${libtype}" in
	shared)
		test -d .libs || mkdir .libs
		case "${tag}" in
		CC)
			for f in $*; do
				case "$f" in
				*.c)	ofile=${f%.c}
					break
					;;
				esac
			done
			picflags=${c_pic_flags}
			;;
		CXX)
			for f in $*; do
				case "$f" in
				*.CC)	ofile=${f%.CC}; break;;
				*.cc)	ofile=${f%.cc}; break;;
				*.cxx)	ofile=${f%.cxx}; break;;
				*.c++)	ofile=${f%.c++}; break;;
				esac
			done
			picflags=${cpp_pic_flags}
			;;
		"")
			echo "Unable to infer tag" >&2
			exit 1
			;;
		esac
		# we run into quoting problems in runcmd if we invoke that way
		${verbose} && echo "libtool: compile: $* ${picflags} -o .libs/${ofile}.o"
		${doit} && $* ${picflags} -o .libs/${ofile}.o
		ret=$?
		case "${ret}" in
		0)
			# we run into quoting problems in runcmd if we invoke that way
			${verbose} && echo "libtool: compile: $* -o ${ofile}.o"
			if ${doit}; then
				$* -o ${ofile}.o >/dev/null 2>&1
				cat > ${ofile}.lo << EOF
# ${ofile}.lo
# created by bsdlibtool ${bsdlibtool_version}
# so don't believe it when it says
# Generated by ltmain.sh (GNU libtool) 2.2.6b
pic_object='.libs/${ofile}.o'
non_pic_object='${ofile}.o'
EOF
			fi
			;;
		*)
			exit $ret
			;;
		esac
		;;
	esac
	;;
execute)
	# execute a program after setting library path
	gotdlopen=false
	libpath=""
	cmd=""
	for f in $*; do
		if ${gotdlopen}; then
			gotdlopen=false
			case "${libpath}" in
			"")	libpath="${f%/*}" ;;
			*) 	libpath="${libpath}:${f%/*}" ;;
			esac
			continue
		fi
		case "$f" in
		-dlopen)
			gotdlopen=true
			;;
		*)
			cmd="${cmd} ${f}"
			;;
		esac
	done
	env LD_LIBRARY_PATH=${libpath} ${cmd}
	;;
finish)
	echo "Libraries have been installed in:"
	echo "	$1"
	;;
install)
	# install library to destination
	instprefixdir=""
	gotit=false
	secondlastarg=$(expr $# - 1)
	lastarg=$#
	i=1
	for f in $*; do
		if [ $i -eq ${secondlastarg} ]; then
			instfile=${f}
		elif [ $i -eq ${lastarg} ]; then
			instdir=${f}
		else
			if ${gotit}; then
				gotit=false
				instprefixdir=${f}/
			fi
			case "${f}" in
			-inst-prefix-dir)	gotit=true ;;
			esac
			cmd="${cmd} ${f}"
		fi
		i=$(expr $i + 1)
	done
	case "${instfile}" in
	.la)
		test -d ${instdir} || instdir=${instdir%/.*}
		labase=${instfile%.la}
		current=$(grep '^current=[0-9]*' .libs/${labase}.lai)
		current=${current#current=}
		age=$(grep '^age=[0-9]*' .libs/${labase}.lai)
		age=${age#age=}
		revision=$(grep '^revision=[0-9]*' .libs/${labase}.lai)
		revision=${revision#revision=}
		lib3=${labase}.so.${current}.${age}.${revision}
		lib1=${labase}.so.${current}
		runcmd install "${cmd} .libs/${lib3} ${instprefixdir}${instdir}"
		runcmd install "(cd ${instprefixdir}${instdir} && ln -s -f ${lib3} ${lib1})"
		runcmd install "(cd ${instprefixdir}${instdir} && ln -s -f ${lib3} ${labase}.so)"
		runcmd install "${precmd} .libs/${labase}.lai ${instprefixdir}${instdir}/${labase}.la"
		runcmd install "${precmd} .libs/${labase}.a ${instprefixdir}${instdir}/${labase}.a"
		runcmd install "chmod 644 ${instprefixdir}${instdir}/${labase}.a"
		runcmd install "ranlib ${instprefixdir}${instdir}/${labase}.a"
		;;
	*)
		runcmd install "${cmd} .libs/${instfile} ${instprefixdir}${instdir}/"
		;;
	esac
	;;
link)
	# link into a library
	outname=""
	case "${libtype}" in
	shared)
		objs=""
		libsobjs=""
		linklibs=""
		rpathset=false
		release=""
		releasesep=""
		gotrelease=false
		gotminuso=false
		for f in $*; do
			if ${rpathset}; then
				rpathset=false
				rpath="${f}"
				continue
			fi
			if ${gotrelease}; then
				gotrelease=false
				release=${f}
				continue
			fi
			if ${gotminuso}; then
				gotminuso=false
				outname=${f}
			fi
			case "$f" in
			*.la)
				labase=${f%.la}
				;;
			*.lo)
				objs="${objs} ${f%.lo}.o"
				libsobjs="${libsobjs} .libs/${f%.lo}.o"
				;;
			-l*)
				linklibs="${linklibs} $f"
				;;
			-o)
				gotminuso=true
				;;
			-rpath)
				rpathset=true
				;;
			-version-info)
				gotrelease=true
				releasesep=":"
				;;
			-release)
				gotrelease=true
				releasesep="."
				;;
			esac
		done
		case "${release}" in
		"")	;;
		*)
			oldifs=${IFS}
			IFS="${releasesep}"
			set -- ${release}
			current=${1:-0}
			age=${2:-0}
			revision=${3:-0}
			IFS=${oldifs}
			;;
		esac
		case "${outname}" in
		*.la)
			lib3=${labase}.so.${current}.${age}.${revision}
			lib1=${labase}.so.${current}
			# create shared lib
			runcmd link "cc -Wl,-x -shared -Wl,-soname -Wl,${lib1} -o .libs/${lib3} -Wl,--whole-archive ${libsobjs} -Wl,--no-whole-archive ${linklibs}"
			runcmd link "(cd .libs && rm -f ${lib1} && ln -s ${lib3} ${lib1})"
			runcmd link "(cd .libs && rm -f ${labase}.so && ln -s ${lib3} ${labase}.so)"
			# create archive
			runcmd link "ar cru .libs/${labase}.a ${objs}"
			runcmd link "ranlib .libs/${labase}.a"
			# write .la informational file
			if ${doit}; then
				cat > ${labase}.la << EOF
# ${labase}.la
# created by bsdlibtool ${bsdlibtool_version}
# so don't believe it when it says
# Generated by ltmain.sh (GNU libtool) 2.2.6b
dlname='${labase}.so.${current}'
library_names='${lib3} ${lib1} ${labase}.so'
old_library='${labase}.a'
inherited_linker_flags=''
dependency_libs='${linklibs}'
weak_library_names=''
current=${current}
age=${age}
revision=${revision}
installed=no
shouldnotlink=no
dlopen=''
dlpreopen=''
libdir='${rpath}'
EOF
			fi
			# make symlinks
			runcmd link "(cd .libs && rm -f ${labase}.la && ln -s ../${labase}.la ${labase}.la)"
			if ${doit}; then
				sed -e 's/installed=no/installed=yes/' ${labase}.la > .libs/${labase}.lai
			fi
			;;
		*)
			cmd=""
			gotminuso=false
			for f in $*; do
				if ${gotminuso}; then
					case "${f}" in
					*.la)	cmd="${cmd} .libs/${f%.la}.so" ;;
					*)	cmd="${cmd} .libs/${f}" ;;
					esac
					gotminuso=false
					continue
				fi
				case "${f}" in
				*.la)	cmd="${cmd} .libs/${f%.la}.so" ;;
				-o)	cmd="${cmd} -o"; gotminuso=true ;;
				*)	cmd="${cmd} ${f}" ;;
				esac
			done
			runcmd link "${cmd}"
			if ${doit}; then
				cat > ${outname} << EOF
#! /bin/sh
# created by bsdlibtool ${bsdlibtool_version}
# so don't believe it when it says
# Generated by ltmain.sh (GNU libtool) 2.2.6b
case "\${LD_LIBRARY_PATH}" in
"")	ldlibpath="$(pwd)/.libs" ;;
*)	ldlibpath="$(pwd)/.libs:\${LD_LIBRARY_PATH}" ;;
esac
env LD_LIBRARY_PATH="\${ldlibpath}" .libs/${outname} \${1+"\$@"}
EOF
			fi
			runcmd link "chmod 755 ${outname}"
			;;
		esac
		;;
	esac
	;;
"")
	echo "No mode provided. No implicit guesses recognised" >&2
	exit 1
	;;
esac

exit 0
