/* $NetBSD: fsu_dir.c,v 1.2 2009/03/24 15:14:24 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/stat.h>
#include <sys/fcntl.h>

#include <assert.h>
#include <dirent.h>
#include <err.h>
#include <stdlib.h>
#include <string.h>

#include <rump/rump.h>
#include <rump/rump_syscalls.h>

#include <fsurs_utils.h>

static char *fsu_getdirname(void);

int
fsu_dirfd(FSU_DIR *dir)
{

	return dir->dd_dd;
}

FSU_DIR
*fsu_opendir(const char *path)
{
	FSU_DIR *dir;
	struct stat sb;
	int dd, rv;

	assert(path != NULL);
	
	rv = rump_sys_stat(path, &sb);
	if (rv != 0 || !S_ISDIR(sb.st_mode))
		return NULL;
	
	dd = rump_sys_open(path, O_RDONLY, 0644);
	if (dd == -1)
		return NULL;

	dir = malloc(sizeof(FSU_DIR));
	if (dir == NULL)
		return NULL;

	dir->dd_dd = dd;
	dir->dd_off = dir->dd_size = 0;
	return dir;
}

void
fsu_closedir(FSU_DIR *dir)
{

	assert(dir != NULL);

	rump_sys_close(dir->dd_dd);
	free(dir);
}

struct dirent
*fsu_readdir(FSU_DIR *dir)
{
	struct dirent *dent;

	assert(dir != NULL);
retry:
	if (dir->dd_size <= 0) {
		dir->dd_size = rump_sys_getdents(dir->dd_dd,
						 (char *)dir->dd_buf,
						 sizeof(dir->dd_buf));
		if (dir->dd_size <= 0)
			return NULL;

		dir->dd_dent = (struct dirent *)dir->dd_buf;
	}

	dent = dir->dd_dent;

	dir->dd_size -= _DIRENT_SIZE(dir->dd_dent);
	dir->dd_dent = _DIRENT_NEXT(dir->dd_dent);

	/* don't return dirents for removed files */
	if (dent->d_ino == 0)
		goto retry;

	return dent;
}

void
fsu_rewinddir(FSU_DIR *dir)
{
	
	assert(dir != NULL);

	rump_sys_lseek(dir->dd_dd, 0, SEEK_SET);
	dir->dd_off = 0;
	dir->dd_size = 0;
}
#include <stdio.h>
char *
fsu_getapath(const char *dir)
{
	size_t new_size, old_size;
	char *new_dir, *old_dir, *p;
	const char *tmp;

	tmp = p = NULL;
	old_dir = fsu_getcwd();

	if (rump_sys_chdir(dir) != 0) {
		new_dir = strdup(dir);
		p = strrchr(new_dir, '/');
		if (p != NULL) {
			*p = 0;
			tmp = dir + (p - new_dir) + 1;
			if (rump_sys_chdir(new_dir) != 0) {
				free(new_dir);
				free(old_dir);
				return NULL;
			}
		} else
			tmp = (dir[0] == '/') ? dir + 1 : dir;

		free(new_dir);
	}
	new_dir = fsu_getcwd();
	new_size = old_size = strlen(new_dir);

	if (tmp != NULL) {
		if (new_size == 1) {
			new_size = strlen(tmp) + 2;
			old_size = 0;
		} else
			new_size += strlen(tmp) + 2; /* '/' and '\0' */

		new_dir = realloc(new_dir, new_size);
		new_dir[old_size] = '/';
		new_dir[old_size + 1] = 0;
		strlcat(new_dir, tmp, new_size);
		new_dir[new_size - 1] = 0;
	}

	rump_sys_chdir(old_dir);
	free(old_dir);
	return new_dir;
}

#define ALLOC_SIZE (10)

/* Gets the absolute path of the current directory */
char
*fsu_getcwd(void)
{
	int i, rv;
	char **names, **tmpnames;
	char *dirname, *tmp;
	size_t len, clen;

	names = malloc(ALLOC_SIZE * sizeof(char *));
	if (names == NULL)
		return NULL;

	len = 1;

	for (i = 0;; ++i) {
		tmp = fsu_getdirname();
		if (tmp == NULL)
			goto err;
		else if (tmp[0] == '/' && tmp[1] == '\0') {
			free(tmp);
			break;
		}

		names[i] = tmp;
		len += strlen(tmp) + 1;

		rv = rump_sys_chdir("..");
		if (rv == -1)
			goto err;

		if ((i + 1) % ALLOC_SIZE == 0) {
			tmpnames = realloc(names,
			    (i + 1 + ALLOC_SIZE) * sizeof(char *));
			if (tmpnames == NULL)
				goto err;
			names = tmpnames;
		}
	}

	if (i == 0)
		return strdup("/");

	dirname = malloc(len);
	if (dirname == NULL)
		goto err;

	for (clen = 0, --i; i >= 0; --i) {
		dirname[clen] = '/';
		dirname[clen + 1] = '\0';
		clen = strlcat(dirname, names[i], len);
		free(names[i]);
	}
	free(names);

	rump_sys_chdir(dirname);
	return dirname;

err:
	for (--i; i >= 0; --i) {
		rump_sys_chdir(names[i]);
		free(names[i]);
	}
	free(names);
	return NULL;
}

static ino_t root_ino = 0;

/* Gets the name of the current directory */
static char
*fsu_getdirname()
{
	struct stat dot;
	struct dirent *dent;
	int fd, rv, size;
	uint8_t buf[8192];
	off_t off;

	off = 0;

	if (root_ino == 0) {
		rv = rump_sys_stat("/", &dot);
		if (rv == -1)
			return NULL;
		root_ino = dot.st_ino;
	}

	rv = rump_sys_stat(".", &dot);
	if (rv == -1)
		return NULL;

	if (dot.st_ino == root_ino)
		return strdup("/");

	fd = rump_sys_open("..", O_RDONLY, 0644);
	if (fd == -1)
		err(1, "open");	

	for (;;) {
		size = rump_sys_getdents(fd, (char *)buf, sizeof(buf));
		if (size == -1) {
			warn("..");
			return NULL;
		} else if (size == 0)
			return NULL;
		dent = (struct dirent *)buf;

		while (size > 0) {
			if (dent->d_fileno == dot.st_ino)
				return strdup(dent->d_name);
			size -= _DIRENT_SIZE(dent);
			dent = _DIRENT_NEXT(dent);
		}
	}
	rump_sys_close(fd);
	/* NOTREACHED */
}
