/*
 * Copyright (c) 1996, 1997, 1998, 1999, 2002 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Adam Hamsik.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * I want to say thank you to all people who helped me with this awesome project.
 */

#include <sys/types.h>
#include <sys/param.h>

#include <sys/buf.h>
#include <sys/conf.h>
#include <sys/dkio.h>
#include <sys/disk.h>
#include <sys/disklabel.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/ioccom.h>
#include <sys/kmem.h>
#include <sys/lkm.h>

#include "netbsd-dm.h"
#include "dm.h"

static dev_type_open(dmopen);
static dev_type_close(dmclose);
static dev_type_read(dmread);
static dev_type_write(dmwrite);
static dev_type_ioctl(dmioctl);
static dev_type_strategy(dmstrategy);
static dev_type_dump(dmdump);
static dev_type_size(dmsize);

/* init routine */
int dm_lkmentry(struct lkm_table *, int, int);

/* handle routine */
static int dm_handle(struct lkm_table *, int);

/* attach and detach routines */
int dm_attach(void);
int dm_destroy(void);

static int dm_cmd_to_fun(prop_dictionary_t);
static int disk_ioctl_switch(dev_t, u_long, void *);
static int dm_ioctl_switch(u_long);
static void dmminphys(struct buf*);
static void dmgetdisklabel(struct dm_dev*, dev_t); 
/* ***Variable-definitions*** */
const struct bdevsw dm_bdevsw = {
	dmopen, dmclose, dmstrategy, dmioctl, dmdump, dmsize,
		D_DISK
};

const struct cdevsw dm_cdevsw = {
	dmopen, dmclose, dmread, dmwrite, dmioctl,
		nostop, notty, nopoll, nommap, nokqfilter, D_DISK
};


MOD_DEV("dm", "dm", &dm_bdevsw, -1, &dm_cdevsw, -1);

/* number of opened devices reference count */
static int dm_ref_count;

/* Info about all devices */
struct dm_softc *dm_sc;

/* This array is used to translate cmd to function pointer. */
struct cmd_function cmd_fn[] = {
		{"version", dm_get_version_ioctl},
		{"targets", dm_list_versions_ioctl},
		{"create",  dm_dev_create_ioctl},
		{"info",    dm_dev_status_ioctl},
		{"mknodes", dm_dev_status_ioctl},		
		{"names",   dm_dev_list_ioctl},
		{"suspend", dm_dev_suspend_ioctl},
		{"remove",  dm_dev_remove_ioctl}, 
		{"rename",  dm_dev_rename_ioctl},
		{"resume",  dm_dev_resume_ioctl},
		{"clear",   dm_table_clear_ioctl},
		{"deps",    dm_table_deps_ioctl},
		{"reload",  dm_table_load_ioctl},	
		{"status",  dm_table_status_ioctl},
		{"table",   dm_table_status_ioctl},
		{NULL, NULL}	
};

/* attach routine */
int
dm_attach(void)
{

	dm_sc = (struct dm_softc *)kmem_alloc(sizeof(struct dm_softc), KM_NOSLEEP);

	if (dm_sc == NULL){
		aprint_error("Not enough memory for dm device.\n");
		return(ENOMEM);
	}

	dm_sc->sc_minor_num = 0;
	dm_sc->sc_ref_count = 0;
	
	dm_target_init();

	dm_dev_init();
	
	dm_pdev_init();
		
	return 0;
}

/* Destroy routine */
int
dm_destroy(void)
{

	(void)kmem_free(dm_sc,sizeof(struct dm_softc));

	return 0;
}

/* lkm entry for ld */
int
dm_lkmentry(struct lkm_table *lkmtp, int cmd, int ver)
{
	DISPATCH(lkmtp, cmd, ver, dm_handle, dm_handle, dm_handle);
}

/* handle function for all load/stat/unload actions */
static int
dm_handle(struct lkm_table *lkmtp, int cmd)
{

	int r;

	r=0;
  
	switch(cmd) {

	case  LKM_E_LOAD:
		aprint_error("Loading dm driver.\n");
		dm_attach();
		break;

	case  LKM_E_UNLOAD:
		if (dm_ref_count == 0) {
			aprint_error("Unloading dm driver\n");
			dm_destroy();
		} else
			r = EBUSY;
		break;
		
	case LKM_E_STAT:
		aprint_error("Opened dm devices: %d\n",dm_ref_count);
		break;
	}	  
	return r;
}

static int
dmopen(dev_t dev, int flags, int mode, struct lwp *l)
{

	struct dm_dev *dmv;

	if ((dmv = dm_dev_lookup_minor(minor(dev))) != NULL) {
		if (dmv->dm_dk == NULL)
			dmgetdisklabel(dmv,dev);

		dmv->ref_cnt++;
		}

	printf("open routine called %d\n",minor(dev));



	return 0;
}


static int
dmclose(dev_t dev, int flags, int mode, struct lwp *l)
{
	struct dm_dev *dmv;

	if ((dmv = dm_dev_lookup_minor(minor(dev))) != NULL)
	dmv->ref_cnt--;

	printf("CLOSE routine called\n");

	return 0;
}

/*
 * Called after ioctl call on mapper/control or dm device.
 */

static int
dmioctl(dev_t dev, const u_long cmd, void *data, int flag, struct lwp *l)
{
	int r;
	prop_dictionary_t dm_dict_in;

	
	r = 0;
	
	if (data == NULL)
		return(EINVAL);

	if (disk_ioctl_switch(dev,cmd,data) != 0) {
		struct plistref *pref = (struct plistref *) data;

		r = prop_dictionary_copyin_ioctl(pref, cmd, &dm_dict_in);
		if (r)
			return r;

		dm_check_version(dm_dict_in);
		
		/* call cmd selected function */
		r = dm_ioctl_switch(cmd);
		if (r < 0)
			goto out;

		r = dm_cmd_to_fun(dm_dict_in);
		if (r < 0)
			goto out;

		char *xml;
		xml = prop_dictionary_externalize(dm_dict_in);
		printf("%s\n",xml);
		
		r = prop_dictionary_copyout_ioctl(pref, cmd, dm_dict_in);
	}

out:
	return r;
}

static int
dm_cmd_to_fun(prop_dictionary_t dm_dict){
	int i,len,slen;
	int r;
	char *command;
	
	r = 0;
		
	(void)prop_dictionary_get_cstring(dm_dict,DM_IOCTL_COMMAND,&command);

	len = strlen(command);
		
	for(i=0;cmd_fn[i].cmd != NULL;i++){
		slen = strlen(cmd_fn[i].cmd);

		if (len != slen)
			continue;

		if ((strncmp(command,cmd_fn[i].cmd,slen)) == 0) {
			r = cmd_fn[i].fn(dm_dict);
			break;
		}
	}
	

	(void)kmem_free(command,len);

	return r;
}

/* Call apropriate ioctl handler function. */
static int
dm_ioctl_switch(u_long cmd)
{
	int r;
	
	r = 0;

	switch(cmd) {
		
	case NETBSD_DM_IOCTL:
		printf("NetBSD_DM_IOCTL called\n");
		break;
		
	default:
		 printf("unknown ioctl called\n");
		 return EPASSTHROUGH;
		 break; /* NOT REACHED */
	}

	 return r;
 }

 /*
  * Check for disk specific ioctls.
  */

 static int
 disk_ioctl_switch(dev_t dev, u_long cmd, void *data)
 {
	 struct dm_dev *dmv;

	 if ((dmv = dm_dev_lookup_minor(minor(dev))) == NULL)
		 return 1;

	 switch(cmd) {

	 case DIOCGWEDGEINFO:
	 {
		 struct dkwedge_info *dkw = (void *) data;

		 printf("DIOCGWEDGEINFO ioctl called\n");

		 strlcpy(dkw->dkw_devname, dmv->name, 16);
		 strlcpy(dkw->dkw_wname, dmv->name, DM_NAME_LEN);
		 strlcpy(dkw->dkw_parent, dmv->name, 16);

		 dkw->dkw_offset = 0;
		 dkw->dkw_size = dmsize(dev);
		 strcpy(dkw->dkw_ptype, DKW_PTYPE_FFS);

		 break;
	 }

	 case DIOCGDINFO:
		 *(struct disklabel *)data = *(dmv->dm_dk->dk_label);
		 break;

	 case DIOCGPART:	
	 case DIOCWDINFO:
	 case DIOCSDINFO:
	 case DIOCKLABEL:
	 case DIOCWLABEL:
	 case DIOCGDEFLABEL:

	 default:
		 printf("unknown disk_ioctl called\n");
		 return 1;
		 break; /* NOT REACHED */
	 }

	 return 0;

 }

 static void
dmstrategy(struct buf *bp)
{

	struct dm_dev *dmv;
	struct dm_table  *tbl;
	
	struct dm_table_entry *table_en;

	struct buf *nestbuf;

	uint64_t table_start;
	uint64_t table_end;
	
	uint64_t buf_start;
	uint64_t buf_len;

	uint64_t start;
	uint64_t end;

	uint64_t issued_len;
	
	buf_start = bp->b_blkno * DEV_BSIZE;
	buf_len = bp->b_bcount;

	tbl = NULL; /* XXX gcc uninitialized */

	table_end = 0;

	issued_len = 0;
	
	printf("dmstrategy routine called %d--%d\n",minor(bp->b_dev),bp->b_bcount);
	
	if ( (dmv = dm_dev_lookup_minor(minor(bp->b_dev))) == NULL) {
		bp->b_error = EIO;
		bp->b_resid = bp->b_bcount;
		biodone(bp);
		return;
	} 

	/* Select active table */
	tbl = &dmv->tables[dmv->cur_active_table];

	 /* Nested buffers count down to zero therefore I have
	    to set bp->b_resid to maximal value. */
	bp->b_resid = bp->b_bcount;
	
	/*
	 * Find out what tables I want to select.
	 */
	SLIST_FOREACH(table_en, tbl, next)
	{
		
		/* I need need number of bytes not blocks. */
		table_start = table_en->start * DEV_BSIZE;
		/* I have to sub 1 from table_en->length to prevent off by one error  */
		table_end = table_start + (table_en->length)* DEV_BSIZE;
		
		start = MAX(table_start, buf_start);

		end = MIN(table_end, buf_start + buf_len);

		printf("----------------------------------------\n");
		printf("table_start %010" PRIu64 ", table_end %010" PRIu64 "\n",table_start,
		    table_end);
		
		printf("buf_start %010" PRIu64 ", buf_len %010" PRIu64 "\n",buf_start,buf_len);
		printf("start-buf_start %010" PRIu64 ", end %010" PRIu64 "\n", start - buf_start,
		    end);
		printf("end-start %010" PRIu64 "\n", end - start);
		printf("\n----------------------------------------\n");

		if (start < end) {
			/* create nested buffer  */
			nestbuf = getiobuf(NULL, true);

			nestiobuf_setup(bp, nestbuf, start - buf_start, (end-start));

			issued_len += end-start;
			
			/* I need number of blocks. */
			nestbuf->b_blkno = (start - table_start) / DEV_BSIZE;

			table_en->target->strategy(table_en, nestbuf);
		}
	}	

	if (issued_len < buf_len)
		nestiobuf_done(bp, buf_len - issued_len, EINVAL);
	
	return;

}


static int
dmread(dev_t dev, struct uio *uio, int flag)
{
	return (physio(dmstrategy, NULL, dev, B_READ, dmminphys, uio));
	//return ENODEV;
}

static int
dmwrite(dev_t dev, struct uio *uio, int flag)
{
	return (physio(dmstrategy, NULL, dev, B_WRITE, dmminphys, uio));
	//return ENODEV;
}

static int
dmdump(dev_t dev, daddr_t blkno, void *va, size_t size)
{
	return ENODEV;
}

static int
dmsize(dev_t dev)
{
	struct dm_dev *dmv;
	struct dm_table  *tbl;
	struct dm_table_entry *table_en;

	uint64_t length;
	
	length = 0;
	
	printf("dmsize routine called %d\n",minor(dev));
	
	if ( (dmv = dm_dev_lookup_minor(minor(dev))) == NULL)
		return ENODEV;
	
	/* Select active table */
	tbl = &dmv->tables[dmv->cur_active_table];

	/*
	 * Find out what tables I want to select.
	 * if length => rawblkno then we should used that table.
	 */
	SLIST_FOREACH(table_en, tbl, next)
	    length += table_en->length;

	
	return length;
}

static void
dmminphys(struct buf *bp)
{
	bp->b_bcount = MIN(bp->b_bcount, MAXPHYS);
}
 /*
  * Load the label information on the named device
  * Actually fabricate a disklabel
  *
  * EVENTUALLY take information about different
  * data tracks from the TOC and put it in the disklabel
  */


static void
dmgetdisklabel(struct dm_dev *dmv, dev_t dev)
{
	struct disklabel *lp = dmv->dm_dk->dk_label;
	const char *errstring;

	memset(dmv->dm_dk->dk_cpulabel, 0, sizeof(struct cpu_disklabel));

	/*
	 * Call the generic disklabel extraction routine
	 */
	errstring = readdisklabel(dev, dmstrategy, lp, dmv->dm_dk->dk_cpulabel);

	/* if all went OK, we are passed a NULL error string */
	if (errstring == NULL)
		return;

	printf("\t%s\t %u sectors \n",dmv->name,dmv->dm_dk->dk_label->d_secsize);
}

