/* $NetBSD$ */

/*-
 * Copyright (c) 2006 Elad Efrat <elad@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Elad Efrat.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/queue.h>
#include <prop/proplib.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <util.h>

#include "haze.h"

/*
 * Print information about an update.
 * [SKELETON]
 */
void
update_report(prop_dictionary_t u)
{
	char *t1, *t2;
	char *p;

	t1 = compact_string(dict_getsc(u, "title"));
	t2 = compact_string(dict_getsc(u, "brief"));

	if (strcasecmp(dict_gets(cfg, "detail-level"), "high") == 0) {
		prop_array_t refs;
		char *buf = NULL;
		size_t n, i;

		refs = prop_dictionary_get(u, "references");
		n = prop_array_count(refs);
		if (n > 0) {
			asprintf(&buf, "  References:\n");
			for (i = 0; i < n; i++) {
				easprintf(&buf, "%s    %s: %s%s",
				    buf,
				    dict_gets(prop_array_get(refs, i), "type"),
				    dict_gets(prop_array_get(refs, i), "data"),
				    ((i + 1) < n) ? "\n" : "");
			}
		}

		easprintf(&p, "[%s] %s (%s, %s)\n  Title: %s\n  %s\n%s",
		    dict_gets(u, "date"),
		    dict_gets(u, "id"), dict_gets(u, "class"),
		    dict_gets(u, "priority"), t1, t2, buf);

		free(buf);
	} else if (strcasecmp(dict_gets(cfg, "detail-level"), "medium") == 0) {
		easprintf(&p, "[%s] %s (%s, %s)\n  Title: %s", dict_gets(u, "date"),
		    dict_gets(u, "id"), dict_gets(u, "class"),
		    dict_gets(u, "priority"), t1);
	} else {
		easprintf(&p, "[%s] %s (%s, %s)", dict_gets(u, "date"),
		    dict_gets(u, "id"), dict_gets(u, "class"),
		    dict_gets(u, "priority"));
	}

	free(t1);
	free(t2);

	printf("%s\n", p);

	free(p);
}

boolean_t
update_applies(prop_dictionary_t u, prop_dictionary_t b, prop_dictionary_t a)
{
	prop_dictionary_t cb;

	if ((cb = branch_monitored(dict_gets(b, "branch"))) != NULL) {
		if (strcasecmp(dict_gets(a, "arch"), "all") == 0)
			return (TRUE);

		if (prop_dictionary_count(cb) == 0)
			return (TRUE);

		if (arch_monitored(dict_gets(b, "branch"),
		    dict_gets(a, "arch")))
			return (TRUE);
	}

	return (FALSE);
}

/*
 * Process an update 'u' for branch 'b', arch 'a'.
 */
void
update_process(prop_dictionary_t u, prop_dictionary_t b, prop_dictionary_t a)
{
	prop_array_t files;
	char *status;
	size_t len, i;

	if (!update_applies(u, b, a))
		return;

	if (dict_getb(cfg, "ignore"))
		status = "ignored";
	else
		status = "installed";

	printf("%s branch %s (%s)\n", dict_getb(cfg, "ignore") ? "Ignoring" :
	    "Updating", dict_gets(b, "branch"), dict_gets(a, "arch"));

	if (!dict_getb(cfg, "ignore")) {
		files = prop_dictionary_get(a, "files");
		len = prop_array_count(files);
		for (i = 0; i < len; i++) {
			prop_dictionary_t file;

			file = prop_array_get(files, i);

			printf("\tUpdating %s to revision %s\n",
			    dict_gets(file, "file"),
			    dict_gets(file, "revision"));
		}
		status = "installed";
	}

	if (!dict_getb(cfg, "dummy-mode")) {
		localdb_add(__UNCONST(dict_gets(u, "id")), status);
	}
}

/*
 * Handle an update.
 */
void
update_handle(prop_dictionary_t u)
{
	prop_array_t a;
	/* prop_dictionary_iterator it; */
	size_t len, i;
	size_t m, j;

	a = prop_dictionary_get(u, "affected");
	len = prop_array_count(a);

	if (len == 0)
		return;

	for (i = 0; i < len; i++) {
		prop_dictionary_t br;
		prop_object_t o;

		br = prop_array_get(a, i);

		if ((o = prop_dictionary_get(br, "archs")) != NULL) {
			m = prop_array_count(o);
			for (j = 0; j < m; j++) {
				update_process(u, br, prop_array_get(o, j));
			}
		}
	}
}

/*
 * Return a full URL to be used for retrieving the update id.
 */
char *
update_mkurl(char *id)
{
	char *loc;

	easprintf(&loc, "%s/%d/%s.plist", dict_gets(cfg, "remote-dir"),
	    id_getyear(id), id);

	return (loc);
}
