/* $NetBSD$ */

/*-
 * Copyright (c) 2006 Elad Efrat <elad@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Elad Efrat.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/queue.h>
#include <prop/proplib.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <util.h>
#include <err.h>

#include "haze.h"

extern int errno;

/*
 * Generate a command string that can be used to fetch a file to a given
 * output file.
 * [MALLOC]
 */
char *
remote_mkcmd(char *outfile, char *path)
{
	char *cmd, *proto = "";
	char res[MAXPATHLEN], *t;

	t = path;

	/* Try to detect local files. */
	if (*path == '/' || strstr(path, "://") == NULL) {
		/* Make absolute path for ftp(1). */
		if (realpath(path, res) == NULL)
			err(1, "Can't resolve path in %s", __func__);
		t = res;

		proto = "file://";
	}

	easprintf(&cmd, "/usr/bin/ftp -V -o %s %s%s", outfile,
	    proto, t);

	return (cmd);
}

/*
 * Fetch the given file to a random local file and return its name.
 * [MALLOC]
 */ 
char *
remote_fetchrnd(char *path)
{
	char *ret, *cmd;
	int r;

	ret = haze_tmpfile(dict_gets(cfg, "work-dir"));
	cmd = remote_mkcmd(ret, path);

	r = haze_exec(cmd);
	if (r) {
		unlink(ret);
		errno = r;
		err(1, "Can't execute in %s", __func__);
	}

	free(cmd);

	return (ret);
}

/*
 * Lookup updates associated with the given year.
 * [REWORK]
 */
void
remote_addyear(int year)
{
	FILE *fp;
	char *path, *cmd, *file;
	char buf[UPDATE_LEN + 1] = "";
	int r;

	easprintf(&path, "%s/%d/%s", dict_gets(cfg, "remote-dir"), year,
	    dict_gets(cfg, "index-name"));

	file = haze_tmpfile(dict_gets(cfg, "work-dir"));
	cmd = remote_mkcmd(file, path);

	r = haze_exec(cmd);
	if (r) {
		unlink(file);
		errno = r;
		err(1, "Can't execute in %s", __func__);
	}

	fp = efopen(file, "r");
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		struct update_batch *b;

		b = ecalloc(1, sizeof(*b));

		estrlcpy(b->id, buf, sizeof(b->id));
		b->id[UPDATE_LEN - 1] = '\0';

		TAILQ_INSERT_TAIL(&batch, b, entries);
	}
	fclose(fp);

	free(path);
	free(cmd);
	unlink(file);
	free(file);
}

/*
 * Generate a list of updates to inspect.
 */
void
remote_buildlist(void)
{
	char *file;
	int from, to;

	TAILQ_INIT(&batch);
	localdb_range(&from, &to);

	file = haze_tmpfile(dict_gets(cfg, "work-dir"));

	for (; from <= to; from++) {
		remote_addyear(from);
	}
}


