/* $NetBSD$ */

/*-
 * Copyright (c) 2006 Elad Efrat <elad@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Elad Efrat.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <prop/proplib.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <err.h>
#include <time.h>

#include "haze.h"

prop_dictionary_t localdb;

/*
 * Print installed (and possibly explicitly ignored) updates.
 * [SKELETON]
 */
void
localdb_report(void)
{
	prop_array_t a;
	size_t len, i;

	a = prop_dictionary_get(localdb, "updates");
	len = prop_array_count(a);
	if (len == 0)
		return;

	printf("Installed%supdates:\n", dict_getb(cfg, "ignore") ?
	    " and ignored " : " ");
	for (i = 0; i < len; i++) {
		prop_dictionary_t tmp;
		boolean_t is_ignored;

		tmp = prop_array_get(a, i);

		if (strcmp(dict_gets(tmp, "status"), "ignored") == 0)
			is_ignored = TRUE;
		else
			is_ignored = FALSE;

		if (is_ignored && !dict_getb(cfg, "ignore"))
			continue;

		printf("  [%s] %s%s", dict_gets(tmp, "timestamp"),
		    dict_gets(tmp, "id"), is_ignored ? " (ignored)\n" : "\n");
	}
}

prop_dictionary_t
localdb_search(char *id)
{
	prop_array_t a;
	size_t len, i;

	a = prop_dictionary_get(localdb, "updates");
	len = prop_array_count(a);
	if (len == 0)
		return (NULL);

	for (i = 0; i < len; i++) {
		prop_dictionary_t entry;

		entry = prop_array_get(a, i);
		if (strcasecmp(dict_gets(entry, "id"), id) == 0)
			return (entry);
	}

	return (NULL);
}

/*
 * Check if we need to skip an update id or not. If we skip it because it
 * was explicitly ignored it, flag it.
 */
boolean_t
localdb_skip(char *id, boolean_t *is_ignored)
{
	prop_dictionary_t entry;

	entry = localdb_search(id);
	if (entry == NULL)
		return (FALSE);

	if (strcasecmp(dict_gets(entry, "status"), "ignored") == 0)
		*is_ignored = TRUE;
	else
		*is_ignored = FALSE;

	if (*is_ignored && dict_getb(cfg, "ignore"))
		return (FALSE);

	return (TRUE);
}

/*
 * Get year range for updates we need to inspect.
 */
void
localdb_range(int *from, int *to)
{
	char *p, *q;
	time_t t;
	struct tm *tm;

	t = time(NULL);
	if ((tm = gmtime(&t)) == NULL)
		err(1, "Can't get current time");

	p = prop_string_cstring(prop_dictionary_get(localdb, "created"));
	q = strchr(p, '-');
	*q = '\0';

	*q = '-';

	*from = atoi(p);
	*to = tm->tm_year + 1900;

	free(p);
}

/*
 * Update the local database with information about the update 'id' with
 * 'status'. Use today's timestamp.
 */
void
localdb_add(char *id, char *status)
{
	prop_dictionary_t entry;
	char *ts;

	/* Do we already have an entry for id? if so, update it. */
	entry = localdb_search(id);
	if (entry == NULL) {
		entry = prop_dictionary_create();
		if (entry == NULL)
			err(1, "No memory in %s", __func__);

		prop_array_add(prop_dictionary_get(localdb, "updates"), entry);
	}

	dict_setsc(entry, "id", id);
	dict_sets(entry, "status", status);
	ts = haze_timestamp();
	dict_sets(entry, "timestamp", ts);
	free(ts);
}

