/* $NetBSD$ */

/*-
 * Copyright (c) 2006 Elad Efrat <elad@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Elad Efrat.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/queue.h>
#include <prop/proplib.h>

#define	DEFAULT_CFG_FILE	"/etc/haze/haze.conf"
#define	DEFAULT_LOCAL_DB	"/etc/haze/local.db"
#define	DEFAULT_REMOTE_DIR	"http://www.NetBSD.org/Updates/"
#define	DEFAULT_INDEX_NAME	"INDEX"
#define	DEFAULT_WORK_DIR	"/tmp"

/*
 * If changing these, make sure id_getyear() still works correctly.
 */
#define	UPDATE_PATTERN	"NetBSD-UPYYYY-NNNN"
#define	UPDATE_LEN	sizeof(UPDATE_PATTERN)
#define	UPDATE_REGEX	"NetBSD-UP[0-9]{4}-[0-9]{4}"

#define	dict_gets(d, k)	\
	prop_string_cstring_nocopy(prop_dictionary_get(d, k))
#define	dict_getsc(d, k) \
	prop_string_cstring(prop_dictionary_get(d, k))
#define	dict_getb(d, k) \
	prop_bool_true(prop_dictionary_get(d, k))
#define	dict_getn(d, k) \
	prop_number_integer_value(prop_dictionary_get(d, k))

#define	dict_sets(d, k, v) \
	prop_dictionary_set(d, k, prop_string_create_cstring_nocopy(v))
#define	dict_setsc(d, k, v) \
	prop_dictionary_set(d, k, prop_string_create_cstring(v))
#define	dict_setb(d, k, v) \
	prop_dictionary_set(d, k, prop_bool_create(v))
#define	dict_seta(d, k) \
	prop_dictionary_set(d, k, prop_array_create())
#define	dict_setn(d, k, v) \
	prop_dictionary_set(d, k, prop_number_create_integer(v))

#define	branch_monitored(b)	\
	prop_dictionary_get(prop_dictionary_get(cfg, "monitor"), \
	b)
#define arch_monitored(b, a)	dict_getb(prop_dictionary_get(\
	prop_dictionary_get(cfg, "monitor"), b), a)

struct update_batch {
	char id[UPDATE_LEN + 1];
	boolean_t ignored;
	TAILQ_ENTRY(update_batch) entries;
};
TAILQ_HEAD(, update_batch) batch;

extern prop_dictionary_t cfg, localdb;

/*
 * Misc. routines.
 */
char *compact_string(char *);
boolean_t haze_regex(char *, char *);
int id_getyear(char *);
char *haze_tmpfile(const char *);
int haze_exec(char *);
char *haze_timestamp(void);
void version(void);
void usage(void);

/*
 * Plist management routines.
 */
prop_dictionary_t plist_load(char *, const char *, boolean_t);
int plist_save(prop_dictionary_t, char *, const char *, boolean_t);

/*
 * Update management routines.
 */
void update_report(prop_dictionary_t);
void update_process(prop_dictionary_t, prop_dictionary_t, prop_dictionary_t);
void update_handle(prop_dictionary_t);
char *update_mkurl(char *);
boolean_t update_applies(prop_dictionary_t, prop_dictionary_t,
    prop_dictionary_t);

/*
 * Local database management routines.
 */
void localdb_report(void);
prop_dictionary_t localdb_search(char *);
boolean_t localdb_skip(char *, boolean_t *);
void localdb_range(int *, int *);
void localdb_add(char *, char *status);

/*
 * Remote database management routines.
 */
char *remote_mkcmd(char *, char *);
char *remote_fetchrnd(char *);
void remote_addyear(int);
void remote_buildlist(void);

/*
 * Batch management routines.
 */
void batch_report(void);
char *batch_getupdate(char *);
void batch_purge(void);
void batch_process(void);

