/* $NetBSD$ */

/*-
 * Copyright (c) 2006 Elad Efrat <elad@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Elad Efrat.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/wait.h>
#include <prop/proplib.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <regex.h>
#include <unistd.h>
#include <err.h>
#include <ctype.h>
#include <util.h>
#include <errno.h>
#include <paths.h>

#include "haze.h"

prop_dictionary_t cfg;

extern int errno;

/*
 * Generic routine to load plist file. Use failmsg for error indication as
 * needed.
 */
prop_dictionary_t
plist_load(char *file, const char *failmsg, boolean_t canfail)
{
	prop_dictionary_t dict;

	dict = prop_dictionary_internalize_from_file(file);
	if ((dict == NULL) && !canfail)
		err(1, "Can't read %s from `%s'", failmsg, file);

	return (dict);
}

int
plist_save(prop_dictionary_t dict, char *file, const char *failmsg,
    boolean_t canfail)
{
	if (prop_dictionary_externalize_to_file(dict, file))
		return (0);
	if (canfail)
		err(1, "Can't save %s to `%s'", failmsg, file);

	return (-1);
}

/*
 * Remove leading and trailing space, and squeeze spaces in the middle of
 * the string to a single space.
 */
char *
compact_string(char *s)
{
	char *q, *n, *m;
	size_t len;

	len = strlen(s) + 1;
	n = ecalloc(len, 1);
	m = n;
	q = s;
	while (isspace((unsigned char)*q))
		q++;
	while (*q != '\0') {
		if (!isspace((unsigned char)*q)) {
			*m++ = *q++;
			continue;
		}
		*m++ = ' ';
		while (isspace((unsigned char)*q))
			q++;
	}
	if (*(m-1) == ' ')
		*(m-1) = '\0';
	estrlcpy(s, n, len);
	free(n);

	return (s);	
}

/*
 * Checks if the string conforms to the regex pattern..
 */
boolean_t
haze_regex(char *s, char *pat)
{
	regex_t re;
	boolean_t r = FALSE;

	if (regcomp(&re, pat, REG_EXTENDED) < 0)
		goto out;

	if (regexec(&re, s, 0, NULL, 0) != 0)
		goto out;

	r = TRUE;

 out:
	regfree(&re);

	return (r);
}


/*
 * Extract year from update id.
 */
int
id_getyear(char *id)
{
	char *p, *q;
	int year;

	p = estrdup(id);
	q = p;
	p += 9;			/* "NetBSD-UP" */
	*(p + 5) = '\0';	/* "-NNNN" */

	year = atoi(p);		/* YYYY */

	free(q);

	return (year);
}

/*
 * Create a temporary file with the given prefix.
 */
char *
haze_tmpfile(const char *prefix)
{
	char *p;

	easprintf(&p, "%s/haze.XXXXXX", prefix);
	if (mktemp(p) == NULL)
		err(1, "Can't create temporary file in %s", __func__);

	return (p);
}

/*
 * Execute a command and return its exit status.
 * Ignore stdin, stdout, and stderr.
 */
int
haze_exec(char *cmd)
{
	char *argp[] = { _PATH_BSHELL, "-c", NULL, NULL };
	pid_t pid;
	int ret;

	argp[2] = cmd;

	pid = fork();
	switch (pid) {
	case 0:
		/* We don't want any output. */
		close(0);
		close(1);
		close(2);

		ret = execve(argp[0], argp, NULL);
		_exit(1);

	case -1:
		err(1, "Can't fork in %s", __func__);
	}

	while (waitpid(pid, &ret, 0) == -1) {
		if (errno != EINTR) {
			ret = -1;
			break;
		}
	}

	return (ret);
}

char *
haze_timestamp(void)
{
	struct tm *tm;
	time_t t;
	char *ts;

	ts = ecalloc(sizeof("YYYY-MM-DD"), 1);
	t = time(NULL);
	if ((tm = gmtime(&t)) == NULL)
		err(1, "Can't get time in %s", __func__);
	if (strftime(ts, sizeof("YYYY-MM-DD"), "%F", tm) == 0)
		err(1, "Error getting time in %s", __func__);

	return (ts);
}

/*
 * Print version.
 */
void
version(void)
{
	fprintf(stderr, "Haze 0.1 -- NetBSD updating utility\n");
}

/*
 * Print usage information.
 */
void
usage(void)
{
	version();
	fprintf(stderr, "Usage:\n"
	    "-c <config_file> # local config file\n"
	    "-l <local_db> # local db file\n"
	    "-u <update_url> # read single update info from file/url\n"
	    "-L # show locally known updates\n"
	    "-B # show to-be processed batch\n"
	    "-i # process ignored entries\n"
	    "-h # this help screen\n"
	);
}

int
main(int argc, char **argv)
{
	char *update_url;
	int r;
	boolean_t list_updates, show_batch;

	list_updates = FALSE;
	update_url = NULL;
	show_batch = FALSE;

	/* Try to load default configuration. */
	cfg = plist_load(DEFAULT_CFG_FILE, "configuration", TRUE);
	if (cfg == NULL) {
		/* No luck? load an empty one. */
		cfg = prop_dictionary_create();
		if (cfg == NULL)
			err(1, "No memory in %s", __func__);

		/* Populate with defaults. */
		dict_sets(cfg, "local-db", DEFAULT_LOCAL_DB);
		dict_sets(cfg, "remote-dir", DEFAULT_REMOTE_DIR);
		dict_sets(cfg, "index-name", DEFAULT_INDEX_NAME);
		dict_sets(cfg, "work-dir", DEFAULT_WORK_DIR);
		dict_sets(cfg, "detail-level", "low");
		dict_setb(cfg, "dummy-mode", FALSE);
	}

	while ((r = getopt(argc, argv, "Bc:d:hiu:l:Lvx")) != -1) {
		switch (r) {
		case 'B':
			show_batch = TRUE;
			break;

		case 'c':
			/* Alternate configuration file. */
			if (cfg != NULL)
				prop_object_release(cfg);
			cfg = plist_load(optarg, "alternate configuration", FALSE);
			break;

		case 'd':
			/* Detail level. */
			dict_sets(cfg, "detail-level", optarg);
			break;

		case 'h':
			/* Show help. */
			usage();
			return (0);

		case 'u':
			update_url = optarg;
			break;

		case 'l':
			dict_sets(cfg, "local-db", optarg);
			break;

		case 'i':
			/* Ignored mode. */
			dict_setb(cfg, "ignore", TRUE);
			break;


		case 'L':
			/* List "known" updates. */
			list_updates = TRUE;
			break;

		case 'v':
			/* Show version. */
			version();
			return (0);

		case 'x':
			/* Dummy mode, just print stuff. */
			dict_setb(cfg, "dummy-mode", TRUE);
			break;
		}
	}

	argc -= optind;
	argv += optind;

	/* Load local database of "known" updates. */
	localdb = plist_load(__UNCONST(dict_gets(cfg, "local-db")),
	    "local database", TRUE);
	if (localdb == NULL) {
		char *ts;

		/* No luck? create and initialize new one. */
		localdb = prop_dictionary_create();

		ts = haze_timestamp();
		dict_setsc(localdb, "created", ts);
		free(ts);

		dict_seta(localdb, "updates");
	}

	/* List locally known updates? */
	if (list_updates) {
		localdb_report();
		return (0);
	}

	if (update_url != NULL) {
		prop_dictionary_t u;
		char *upf;
		boolean_t needs_cleanup = FALSE;

		/* Do we have to figure out the path ourselves? */
		if (haze_regex(update_url, UPDATE_REGEX)) {
			update_url = update_mkurl(update_url);
			needs_cleanup = TRUE;
		}

		/* Fetch update to a temporary file and load it. */
		upf = remote_fetchrnd(update_url);
		u = plist_load(upf, "update", FALSE);

		/* Report the update we're about to process. */
		update_report(u);

		update_handle(u);

		prop_object_release(u);

		if (needs_cleanup) {
			unlink(upf);
			free(update_url);
		}

		free(upf);

		goto out;
	}

	/* Build list of updates to check. */
	remote_buildlist();
	batch_purge();
	if (show_batch)
		batch_report();
	else if (!dict_getb(cfg, "dummy-mode")) {
		batch_process();
	}

 out:
	if (!dict_getb(cfg, "dummy-mode")) {
		plist_save(localdb, __UNCONST(dict_gets(cfg, "local-db")), "local db", FALSE);
	}

	return (0);
}

