/* $NetBSD$ */

/*-
 * Copyright (c) 2006 Elad Efrat <elad@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Elad Efrat.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/queue.h>
#include <prop/proplib.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <err.h>

#include "haze.h"

extern int errno;

/*
 * Report batched updates.
 * [SKELETON]
 */
void
batch_report(void)
{
	struct update_batch *b;

	printf("The following updates will be candidates for processing:\n");
	TAILQ_FOREACH(b, &batch, entries) {
		printf("\t%s%s", b->id, b->ignored ?
		    " (previously ignored)\n" : "\n");
	}
}

/*
 * Retrieve the update for the given id and return a local file from which
 * the update can be loaded.
 * [MALLOC]
 */
char *
batch_getupdate(char *id)
{
	char *loc, *cmd, *f;
	int r;

	loc = update_mkurl(id);

	f = haze_tmpfile(dict_gets(cfg, "work-dir"));
	cmd = remote_mkcmd(f, loc);

	r = haze_exec(cmd);
	if (r) {
		unlink(f);
		errno = r;
		err(1, "Can't execute in %s", __func__);
	}

	free(loc);
	free(cmd);

	return (f);
}

/*
 * For each update in the batch, make sure it's not already installed.
 * Unless explicitly requested, skip "ignored" updates.
 */
void
batch_purge(void)
{
	struct update_batch *b;

	TAILQ_FOREACH(b, &batch, entries) {
		if (localdb_skip(b->id, &b->ignored)) {
			/* Remove entry from batch. */
			TAILQ_REMOVE(&batch, b, entries);
			free(b);
			continue;
		}
	}
}

/*
 * Process batched updates.
 */
void
batch_process(void)
{
	struct update_batch *b;

	printf("Processing updates...\n");
	TAILQ_FOREACH(b, &batch, entries) {
		prop_dictionary_t u;
		char *upf;

		upf = batch_getupdate(b->id);
		u = plist_load(upf, "update file", FALSE);

		update_report(u);
		update_handle(u);

		prop_object_release(u);
		unlink(upf);
		free(upf);
	}
}

